/*
 * $Id: site.sql 101389 2011-11-25 01:51:25Z rlowery $
 *
 * Description:
 *
 * SQL Script to add ManageSoft AD population procedures.
 *
 * The steps performed by this script are:
 * 1. Add AddComputer procedure
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 *
 * Copyright 2011 Flexera Software LLC
 */
 
-- Use ManageSoft database
USE ManageSoft
GO

-- Create 'AddComputer' stored procedure
PRINT 'Creating procedure to add a computer'
IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
	AND name = 'AddComputer'
)
	DROP PROCEDURE AddComputer
GO
CREATE PROCEDURE AddComputer
	@ComputerID INTEGER OUT,
	@ComputerCN NVARCHAR(64),
	@ComputerOU NVARCHAR(800),
	@OperatingSystem NVARCHAR(128) = NULL,
	@ADComputerGUID BINARY(16) = NULL,
	@ADComputerOUGUID BINARY(16) = NULL
AS
BEGIN
	
	-- Calculate DC of OU
	DECLARE @OUDC NVARCHAR(100)
	SET @OUDC = SUBSTRING(
		@ComputerOU,
		CHARINDEX('DC=', @ComputerOU, 0),
		LEN(@ComputerOU) - CHARINDEX('DC=', @ComputerOU, 0) + 1
	)
	
	-- Calculate RDN of OU
	DECLARE @OURDN NVARCHAR(400)
	SET @OURDN = SUBSTRING(
		@ComputerOU,
		1,
		LEN(@ComputerOU) - (LEN(@OUDC) + 1)
	)
	
	-- Create OU in ManageSoft SDOU table if necessary
	DECLARE @OUID INTEGER
	SELECT @OUID = SD.OUID
	FROM (
		SELECT o.OrganizationID as OUID,
			CASE
			    WHEN o.RDN IS NULL THEN	-- Unknown OU in this domain
				'<NULL>,'+d.DN		-- or the universal domain
			    WHEN o.RDN = '' THEN
				d.DN			-- DomainDN for the domain
			    ELSE
				o.RDN+','+d.DN		-- Ordinary OU or container
			END AS OU
		FROM Organization AS o, Domain AS d
		WHERE o.DomainID = d.OrganizationID
	) as SD
	WHERE SD.OU = @ComputerOU
	
	IF (@OUID IS NULL)
	BEGIN
		EXEC OrganizationDNCreate @OURDN, @OUDC, @ADComputerOUGUID
		SELECT @OUID = SD.OUID
		FROM (
			SELECT o.OrganizationID as OUID,
				CASE
				    WHEN o.RDN IS NULL THEN	-- Unknown OU in this domain
					'<NULL>,'+d.DN		-- or the universal domain
				    WHEN o.RDN = '' THEN
					d.DN			-- DomainDN for the domain
				    ELSE
					o.RDN+','+d.DN		-- Ordinary OU or container
				END AS OU
			FROM Organization AS o, Domain AS d
			WHERE o.DomainID = d.OrganizationID
		) as SD
		WHERE SD.OU = @ComputerOU
	END
	
	-- Get domain ID of OU
	DECLARE @DNID INTEGER
	SELECT @DNID = DomainID FROM Organization WHERE OrganizationID = @OUID

	-- Find existing computer in destination OU
	IF @ComputerID IS NULL
	BEGIN
		SELECT TOP 1 @ComputerID = ComputerID
		FROM Computer,
		(
			SELECT	o.OrganizationID as OUID,
				CASE
				    WHEN o.RDN IS NULL THEN	-- Unknown OU in this domain
					'<NULL>,'+d.DN		-- or the universal domain
				    WHEN o.RDN = '' THEN
					d.DN			-- DomainDN for the domain
				    ELSE
					o.RDN+','+d.DN		-- Ordinary OU or container
				END AS OU
			FROM Organization AS o, Domain AS d
			WHERE o.DomainID = d.OrganizationID
		) as SD
		WHERE ComputerCN = @ComputerCN
		AND ComputerOUID = SD.OUID
		AND SD.OU = @ComputerOU
	END
	
	-- Not found, so try computer with same DC
	IF @ComputerID IS NULL
	BEGIN
		-- Look for computer with same DC
		SELECT TOP 1 @ComputerID = ComputerID
		FROM	Computer,
			(
				SELECT	o.OrganizationID as OUID,
					CASE
					    WHEN o.RDN IS NULL THEN	-- Unknown OU in this domain
						'<NULL>,'+d.DN		-- or the universal domain
					    WHEN o.RDN = '' THEN
						d.DN			-- DomainDN for the domain
					    ELSE
						o.RDN+','+d.DN		-- Ordinary OU or container
					END AS OU
				FROM Organization AS o, Domain AS d
				WHERE o.DomainID = d.OrganizationID
			) as SD
		WHERE ComputerCN = @ComputerCN
		AND ComputerOUID = SD.OUID
		AND (@OUDC = SUBSTRING(
			SD.OU,
			CHARINDEX('DC=', SD.OU, 0),
			LEN(SD.OU) - CHARINDEX('DC=', SD.OU, 0) + 1
		))
	END

	-- Determine operating system ID
	DECLARE @OperatingSystemID int
	IF @OperatingSystem IS NOT NULL
	BEGIN
		IF NOT EXISTS (
			SELECT [@OperatingSystemID] = OperatingSystemID
			FROM OperatingSystem
			WHERE OperatingSystemName = @OperatingSystem
		)
		BEGIN
			INSERT INTO OperatingSystem
			(OperatingSystemName) VALUES (@OperatingSystem)

			SELECT @OperatingSystemID = OperatingSystemID
			FROM OperatingSystem
			WHERE OperatingSystemName = @OperatingSystem
		END
	END ELSE SET @OperatingSystemID = NULL

	-- Check to see if computer already exists
	IF @ComputerID IS NOT NULL
	BEGIN
		-- Update existing computer in the ManageSoft computer table
		UPDATE Computer
		SET	ComputerCN = @ComputerCN,
			ComputerOUID = @OUID,
			OperatingSystemID = ISNULL(@OperatingSystemID, OperatingSystemID),
			GUID = ISNULL(@ADComputerGUID, GUID)
		WHERE ComputerID = @ComputerID
		
		-- Update the OU in ManageSoft tables that have links to the computuer
		UPDATE Installation
			SET OrganizationID = @OUID
			WHERE ComputerID = @ComputerID
			AND UserID = 1	-- SYSTEM user
		DELETE FROM PackageApplies
			WHERE TargetTypeID = 1
			AND TargetID = @ComputerID
		
		-- Update name map table
		UPDATE NameMap
		SET NewDomain = @DNID
		WHERE CurrentID = @ComputerID
	END
	ELSE
	BEGIN
		-- Add computer to ManageSoft computer table
		INSERT INTO Computer
		(ComputerCN, ComputerOUID, OperatingSystemID, GUID)
		VALUES (@ComputerCN, @OUID, @OperatingSystemID, @ADComputerGUID)
		SET @ComputerID = @@IDENTITY
	END
END
GO

-- Stored Procedures to Select/Update/Insert/Delete site table
PRINT 'Creating procedures to Select, Update, Insert, Delete site table'
IF EXISTS (
	SELECT * FROM sysobjects 
	WHERE [name] = 'WiMiSiteSelect' AND xtype = 'P'
)
	DROP PROCEDURE WiMiSiteSelect
GO

CREATE PROCEDURE [WiMiSiteSelect]
AS
	SET NOCOUNT ON
	SELECT SiteID, SiteName, OU, IPSubnet, Description FROM Site
GO



IF EXISTS (
	SELECT * FROM sysobjects 
	WHERE [name] = 'WiMiSiteInsert' AND xtype = 'P'
)
	DROP PROCEDURE WiMiSiteInsert
GO

CREATE PROCEDURE [WiMiSiteInsert]
(
	@SiteName nvarchar(64),
	@OU nvarchar(800),
	@IPSubnet varchar(64),
	@Description nvarchar(256)
)
AS
	SET NOCOUNT OFF
	INSERT INTO Site(SiteName, OU, IPSubnet, Description) VALUES (@SiteName, @OU, @IPSubnet, @Description)
	SELECT SiteID, SiteName, OU, IPSubnet, Description FROM Site WHERE (SiteID = @@IDENTITY)
GO



IF EXISTS (
	SELECT * FROM sysobjects 
	WHERE [name] = 'WiMiSiteUpdate' AND xtype = 'P'
)
	DROP PROCEDURE WiMiSiteUpdate
GO

CREATE PROCEDURE [WiMiSiteUpdate]
(
	@SiteName nvarchar(64),
	@OU nvarchar(800),
	@IPSubnet varchar(64),
	@Description nvarchar(256),
	@Original_SiteID int,
	@Original_Description nvarchar(256),
	@Original_IPSubnet varchar(64),
	@Original_OU nvarchar(800),
	@Original_SiteName nvarchar(64),
	@SiteID int
)
AS
	SET NOCOUNT OFF
	UPDATE Site SET SiteName = @SiteName, OU = @OU, IPSubnet = @IPSubnet, Description = @Description WHERE (SiteID = @Original_SiteID) AND (Description = @Original_Description OR @Original_Description IS NULL AND Description IS NULL) AND (IPSubnet = @Original_IPSubnet OR @Original_IPSubnet IS NULL AND IPSubnet IS NULL) AND (OU = @Original_OU OR @Original_OU IS NULL AND OU IS NULL) AND (SiteName = @Original_SiteName OR @Original_SiteName IS NULL AND SiteName IS NULL)
	SELECT SiteID, SiteName, OU, IPSubnet, Description FROM Site WHERE (SiteID = @SiteID)
GO



IF EXISTS (
	SELECT * FROM sysobjects 
	WHERE [name] = 'WiMiSiteDelete' AND xtype = 'P'
)
	DROP PROCEDURE WiMiSiteDelete
GO

CREATE PROCEDURE [WiMiSiteDelete]
(
	@Original_SiteID int,
	@Original_Description nvarchar(256),
	@Original_IPSubnet varchar(64),
	@Original_OU nvarchar(800),
	@Original_SiteName nvarchar(64)
)
AS
	SET NOCOUNT OFF
	DELETE FROM Site WHERE (SiteID = @Original_SiteID) AND (Description = @Original_Description OR @Original_Description IS NULL AND Description IS NULL) AND (IPSubnet = @Original_IPSubnet OR @Original_IPSubnet IS NULL AND IPSubnet IS NULL) AND (OU = @Original_OU OR @Original_OU IS NULL AND OU IS NULL) AND (SiteName = @Original_SiteName OR @Original_SiteName IS NULL AND SiteName IS NULL)
GO
