/*
 * $Id: rolloutconfig.sql 101389 2011-11-25 01:51:25Z rlowery $
 *
 * Description:
 *
 * SQL Script to build the ManageSoft rollout configuration tables on Microsoft SQL
 * Server.
 *
 * The steps performed by this script are:
 * 1. Create new database
 * 2. Create database table and initialiase data (skip if the table exists)
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 *
 * Copyright 2011 Flexera Software LLC
 */
USE ManageSoft
GO

-- Create RolloutConfigFolder table
IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutConfigFolder'
)
BEGIN
	-- Create table
	PRINT 'Creating RolloutConfigFolder table'
	CREATE TABLE RolloutConfigFolder
	(
		-- Auto-generated identity #
		RolloutConfigFolderID INTEGER NOT NULL IDENTITY
			CONSTRAINT PK_RolloutConfigFolder PRIMARY KEY NONCLUSTERED,

		-- GUID of rollout config
		RolloutConfigFolderGUID BINARY(16) NULL DEFAULT(NULL),
			
		-- Parent rollout config folder
		RolloutConfigFolderPath INTEGER REFERENCES RolloutConfigFolder(RolloutConfigFolderID),
			
		-- Name of rollout config
		RolloutConfigFolderName NVARCHAR(256) NULL DEFAULT(NULL)
	);
	
	-- Create indices
	CREATE UNIQUE INDEX RolloutConfigFolderByGUID
	ON RolloutConfigFolder (RolloutConfigFolderGUID)
	CREATE UNIQUE INDEX RolloutConfigFolderByPathName
	ON RolloutConfigFolder (RolloutConfigFolderPath, RolloutConfigFolderName)
	CREATE INDEX RolloutConfigFolderByPath
	ON RolloutConfigFolder (RolloutConfigFolderPath)
END
ELSE
	PRINT 'Using existing RolloutConfigFolder table'
GO

-- Create RolloutConfig table
IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutConfig'
)
BEGIN
	-- Create table
	PRINT 'Creating RolloutConfig table'
	CREATE TABLE RolloutConfig
	(
		-- Auto-generated identity #
		RolloutConfigID INTEGER NOT NULL IDENTITY
			CONSTRAINT PK_RolloutConfig PRIMARY KEY NONCLUSTERED,

		-- GUID of rollout config
		RolloutConfigGUID BINARY(16) NULL DEFAULT(NULL),
			
		-- Parent rollout configuration folder
		RolloutConfigPath INTEGER NULL REFERENCES RolloutConfigFolder(RolloutConfigFolderID),
			
		-- Name of rollout config
		RolloutConfigName NVARCHAR(256) NOT NULL,
		
		-- Version of image
		RolloutConfigVersion NVARCHAR(64) NOT NULL,
		
		-- Modify date and time of rollout config
		ModifyDateTime DATETIME NOT NULL,
			
		-- Description of rollout config
		Description NVARCHAR(256) NULL DEFAULT(NULL),
		
		-- Path of rollout config in repository
		RepositoryPath NVARCHAR(256) NULL DEFAULT(NULL)
	);
	
	-- Create indices
	CREATE UNIQUE INDEX RolloutConfigByGUID
	ON RolloutConfig (RolloutConfigGUID)
	CREATE UNIQUE INDEX RolloutConfigByPathName
	ON RolloutConfig (RolloutConfigPath, RolloutConfigName)
	CREATE INDEX RolloutConfigByPath
	ON RolloutConfig (RolloutConfigPath)
END
ELSE
	PRINT 'Using existing RolloutConfig table'
GO

-- Create RolloutConfigSettings table
IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutConfigSetting'
)
BEGIN
	-- Create table
	PRINT 'Creating RolloutConfigSetting table'
	CREATE TABLE RolloutConfigSetting
	(
		-- Auto-generated identity #
		RolloutConfigSettingID INTEGER NOT NULL IDENTITY
			CONSTRAINT PK_RolloutConfigSetting PRIMARY KEY NONCLUSTERED,

		-- Parent rollout config
		RolloutConfig INTEGER REFERENCES RolloutConfig(RolloutConfigID),
		
		-- Type of rollout config setting
		SettingType NVARCHAR(64) NULL DEFAULT(NULL),
			
		-- Name of rollout config setting
		SettingName NVARCHAR(64) NOT NULL,
		
		-- Value of rollout config setting
		SettingValue NVARCHAR(256) NULL DEFAULT(NULL)
	);
	
	-- Create indices
	CREATE INDEX RolloutConfigSettingByType
	ON RolloutConfigSetting (SettingType)
	CREATE UNIQUE INDEX RolloutConfigSettingByName
	ON RolloutConfigSetting (RolloutConfig, SettingType, SettingName)
END
ELSE
	PRINT 'Using existing RolloutConfigSetting table'
GO

-- The RolloutStatusType table stores the types of operations we can perform on a
-- computer.
IF NOT EXISTS (
	SELECT * FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutStatusType'
)
BEGIN
	CREATE TABLE RolloutStatusType (
		RolloutStatusTypeID	int IDENTITY NOT NULL,
		[Name]			nvarchar(64) NOT NULL,
		CONSTRAINT UQ_RolloutStatusType
			UNIQUE NONCLUSTERED(RolloutStatusTypeID),
		CONSTRAINT PK_RolloutStatusType
			PRIMARY KEY NONCLUSTERED([Name])
	);
END
ELSE BEGIN
	PRINT 'Using existing RolloutStatusType table'
	IF NOT EXISTS (
		SELECT * FROM sysindexes
		WHERE name = 'UQ_RolloutStatusType'
	)
		ALTER TABLE RolloutStatusType
		ADD CONSTRAINT UQ_RolloutStatusType
			UNIQUE NONCLUSTERED(RolloutStatusTypeID)

	IF NOT EXISTS (
		SELECT * FROM sysindexes
		WHERE name = 'PK_RolloutStatusType'
	)
		ALTER TABLE RolloutStatusType
		ADD CONSTRAINT PK_RolloutStatusType
			PRIMARY KEY NONCLUSTERED([Name])
END
GO

-- The RolloutStatus table stores the status for each step in a rollout for a computer.
IF NOT EXISTS (
	SELECT * FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutStatus'
)
BEGIN
	CREATE TABLE RolloutStatus (
		-- The session reporting the status.
		SessionName		nvarchar(64) NOT NULL,
		-- Type of rollout status, such as data capture or computer build.
		RolloutStatusTypeID	int NOT NULL,
		-- The computer reporting the rollout status if known.
		ComputerID		int NULL,
		-- The step number reporting the rollout status.
		StepNumber		int NOT NULL,
		-- The description of the rollout step.
		Description		nvarchar(256) NOT NULL,
		-- The status of this rollout step.
		Status			nvarchar(64) NOT NULL,
		-- The status message of this rollout step.
		StatusMessage		ntext NULL,
		-- The process phase that this particular step took place on.
		Phase			nvarchar(256) NULL,
		CONSTRAINT FK_RolloutStatus_RolloutStatusType
			FOREIGN KEY(RolloutStatusTypeID)
			REFERENCES RolloutStatusType(RolloutStatusTypeID) ON DELETE NO ACTION,
		CONSTRAINT FK_RolloutStatus_Computer
			FOREIGN KEY(ComputerID)
			REFERENCES Computer(ComputerID) ON DELETE CASCADE,
		CONSTRAINT PK_RolloutStatus
			PRIMARY KEY NONCLUSTERED(SessionName, StepNumber, RolloutStatusTypeID)
	);
END
ELSE BEGIN
	PRINT 'Using existing RolloutStatus table'
	IF NOT EXISTS (
		SELECT * FROM sysindexes
		WHERE name = 'PK_RolloutStatus'
	)
		ALTER TABLE RolloutStatus
		ADD CONSTRAINT PK_RolloutStatus
			PRIMARY KEY NONCLUSTERED(SessionName, StepNumber, RolloutStatusTypeID)
END
GO

-- The RolloutStatusSummary table stores the summarized status of the current step in a rollout for a computer.
IF NOT EXISTS (
	SELECT * FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutStatusSummary'
)
BEGIN
	CREATE TABLE RolloutStatusSummary (
		-- The session reporting the status.
		SessionName		nvarchar(64) NOT NULL,
		-- Type of rollout status, such as data capture or computer build.
		RolloutStatusTypeID	int NOT NULL DEFAULT 1,
		-- The computer reporting the rollout status summary if known.
		ComputerID		int NULL,
		-- The current status of the rollout.
		LastStatus		nvarchar(64) NULL,
		-- The current rollout step of the computer.
		LastStep		int NULL,
		-- The current rollout step description.
		LastStepDescription	nvarchar(256) NULL,
		-- The number of rollout steps in the rollout.
		StepCount		int NULL,
		-- The date and time that the rollout was initiated.
		StartDateTime		datetime NULL,
		-- The date and time of the last rollout status.
		LastDateTime		datetime NULL,
		-- The date and time that the rollout was completed.
		CompleteDateTime	datetime NULL,
		CONSTRAINT FK_RolloutStatusStatus_RolloutStatusType
			FOREIGN KEY(RolloutStatusTypeID)
			REFERENCES RolloutStatusType(RolloutStatusTypeID) ON DELETE NO ACTION,
		CONSTRAINT FK_RolloutStatusSummary_Computer
			FOREIGN KEY(ComputerID)
			REFERENCES Computer(ComputerID) ON DELETE CASCADE,
		CONSTRAINT PK_RolloutStatusSummary
			PRIMARY KEY NONCLUSTERED(SessionName, RolloutStatusTypeID)
	);

	CREATE NONCLUSTERED INDEX RolloutStatusSummary_LastStatus
		ON RolloutStatusSummary(LastStatus)
END
ELSE BEGIN
	PRINT 'Using existing RolloutStatusSummary table'
	IF NOT EXISTS (
		SELECT * FROM sysindexes
		WHERE name = 'PK_RolloutStatusSummary'
	)
		ALTER TABLE RolloutStatusSummary
		ADD CONSTRAINT PK_RolloutStatusSummary
			PRIMARY KEY NONCLUSTERED(SessionName, RolloutStatusTypeID)

	IF NOT EXISTS (
		SELECT * FROM sysindexes
		WHERE name = 'RolloutStatusSummary_LastStatus'
	)
		CREATE NONCLUSTERED INDEX RolloutStatusSummary_LastStatus
			ON RolloutStatusSummary(LastStatus)
END
GO

IF NOT EXISTS (
	SELECT * FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'DataCapture'
)
BEGIN
	CREATE TABLE DataCapture (
		SessionName		nvarchar(64) NOT NULL,
		ComputerID		int NULL,
		DataStoreLocation	nvarchar(1024) NULL,
		DataImageCaptureLocation nvarchar(1024) NULL,
		LogFileLocation		nvarchar(1024) NULL,
		CaptureDateTime		datetime NOT NULL DEFAULT getdate(),
		ImageCaptureDateTime	datetime NOT NULL DEFAULT getdate(),
		CONSTRAINT FK_DataCapture_ComputerID
			FOREIGN KEY(ComputerID)
			REFERENCES Computer(ComputerID) ON DELETE CASCADE,
		CONSTRAINT PK_DataCapture
			PRIMARY KEY NONCLUSTERED(SessionName)
	);
END
ELSE BEGIN
	PRINT 'Using existing DataCapture table'
	IF NOT EXISTS (
		SELECT * FROM sysindexes
		WHERE name = 'PK_DataCapture'
	)
		ALTER TABLE DataCapture
		ADD CONSTRAINT PK_DataCapture
			PRIMARY KEY NONCLUSTERED(SessionName)
END
GO

