/*
 * $Id: rolloutconfig.sql 101389 2011-11-25 01:51:25Z rlowery $
 *
 * Description:
 *
 * SQL Script to build the ManageSoft rollout config library stored procedures
 * on Microsoft SQL Server.
 *
 * The steps performed by this script are:
 * 1. Create new database
 * 2. Create database table and initialiase data (skip if the table exists)
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 *
 * Copyright 2011 Flexera Software LLC
 */
USE ManageSoft
GO

-- Procedure for selecting a rollout config folder
PRINT 'Creating procedure to select a rollout config folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigFolderSelect' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigFolderSelect];
GO
CREATE PROCEDURE [RolloutConfigFolderSelect]
(
	@RolloutConfigFolderGUID binary(16)
)
AS
	SET NOCOUNT ON;
	SELECT A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName,
		COUNT(B.RolloutConfigFolderPath) AS ChildRolloutConfigFolderCount, COUNT(C.RolloutConfigPath) AS ChildRolloutConfigCount
		FROM RolloutConfigFolder AS A
		LEFT OUTER JOIN RolloutConfigFolder AS B
		ON A.RolloutConfigFolderID = B.RolloutConfigFolderPath
		LEFT OUTER JOIN RolloutConfig AS C
		ON A.RolloutConfigFolderID = C.RolloutConfigPath
		WHERE (A.RolloutConfigFolderGUID = @RolloutConfigFolderGUID)
		GROUP BY A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName;
GO

-- Procedure for selecting a rollout config folder
PRINT 'Creating procedure to select a rollout config folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigFolderSelectID' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigFolderSelectID];
GO
CREATE PROCEDURE [RolloutConfigFolderSelectID]
(
	@RolloutConfigFolderID int
)
AS
	SET NOCOUNT ON;
	SELECT A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName,
		COUNT(B.RolloutConfigFolderPath) AS ChildRolloutConfigFolderCount, COUNT(C.RolloutConfigPath) AS ChildRolloutConfigCount
		FROM RolloutConfigFolder AS A
		LEFT OUTER JOIN RolloutConfigFolder AS B
		ON A.RolloutConfigFolderID = B.RolloutConfigFolderPath
		LEFT OUTER JOIN RolloutConfig AS C
		ON A.RolloutConfigFolderID = C.RolloutConfigPath
		WHERE (A.RolloutConfigFolderID = @RolloutConfigFolderID)
		GROUP BY A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName;
GO

-- Procedure for selecting child rollout config folders
PRINT 'Creating procedure to select child rollout config folders'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigFolderChildSelect' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigFolderChildSelect];
GO
CREATE PROCEDURE [RolloutConfigFolderChildSelect]
(
	@RolloutConfigFolderPath int = NULL
)
AS
	SET NOCOUNT ON;
	IF (@RolloutConfigFolderPath IS NULL)
		SELECT A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName,
			COUNT(B.RolloutConfigFolderPath) AS ChildRolloutConfigFolderCount, COUNT(C.RolloutConfigPath) AS ChildRolloutConfigCount
			FROM RolloutConfigFolder AS A
			LEFT OUTER JOIN RolloutConfigFolder AS B
			ON A.RolloutConfigFolderID = B.RolloutConfigFolderPath
			LEFT OUTER JOIN RolloutConfig AS C
			ON A.RolloutConfigFolderID = C.RolloutConfigPath
			WHERE (A.RolloutConfigFolderPath IS NULL)
			GROUP BY A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName
			ORDER BY A.RolloutConfigFolderName;
	ELSE
		SELECT A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName,
			COUNT(B.RolloutConfigFolderPath) AS ChildRolloutConfigFolderCount, COUNT(C.RolloutConfigPath) AS ChildRolloutConfigCount
			FROM RolloutConfigFolder AS A
			LEFT OUTER JOIN RolloutConfigFolder AS B
			ON A.RolloutConfigFolderID = B.RolloutConfigFolderPath
			LEFT OUTER JOIN RolloutConfig AS C
			ON A.RolloutConfigFolderID = C.RolloutConfigPath
			WHERE (A.RolloutConfigFolderPath = @RolloutConfigFolderPath)
			GROUP BY A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName
			ORDER BY A.RolloutConfigFolderName;
GO

-- Procedure for inserting a rollout config folder
PRINT 'Creating procedure to insert a rollout config folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigFolderInsert' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigFolderInsert];
GO
CREATE PROCEDURE [RolloutConfigFolderInsert]
(
	@RolloutConfigFolderGUID binary(16),
	@RolloutConfigFolderPath int,
	@RolloutConfigFolderName nvarchar(256)
)
AS
	SET NOCOUNT OFF;
	INSERT INTO RolloutConfigFolder (RolloutConfigFolderGUID, RolloutConfigFolderPath, RolloutConfigFolderName)
		VALUES (@RolloutConfigFolderGUID, @RolloutConfigFolderPath, @RolloutConfigFolderName);
	SELECT A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName,
		COUNT(B.RolloutConfigFolderPath) AS ChildRolloutConfigFolderCount, COUNT(C.RolloutConfigPath) AS ChildRolloutConfigCount
		FROM RolloutConfigFolder AS A
		LEFT OUTER JOIN RolloutConfigFolder AS B
		ON A.RolloutConfigFolderID = B.RolloutConfigFolderPath
		LEFT OUTER JOIN RolloutConfig AS C
		ON A.RolloutConfigFolderID = C.RolloutConfigPath
		WHERE (A.RolloutConfigFolderID = @@IDENTITY)
		GROUP BY A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName;
GO

-- Procedure for updating a rollout config folder
PRINT 'Creating procedure to update a rollout config folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigFolderUpdate' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigFolderUpdate];
GO
CREATE PROCEDURE [RolloutConfigFolderUpdate]
(
	@RolloutConfigFolderGUID binary(16),
	@RolloutConfigFolderPath int,
	@RolloutConfigFolderName nvarchar(256),
	@Original_RolloutConfigFolderID int,
	@Original_RolloutConfigFolderGUID binary(16),
	@Original_RolloutConfigFolderName nvarchar(256),
	@Original_RolloutConfigFolderPath int,
	@RolloutConfigFolderID int
)
AS
	SET NOCOUNT OFF;
	UPDATE RolloutConfigFolder
		SET RolloutConfigFolderGUID = @RolloutConfigFolderGUID, RolloutConfigFolderPath = @RolloutConfigFolderPath, RolloutConfigFolderName = @RolloutConfigFolderName
		WHERE (RolloutConfigFolderID = @Original_RolloutConfigFolderID)
		AND (RolloutConfigFolderGUID = @Original_RolloutConfigFolderGUID OR @Original_RolloutConfigFolderGUID IS NULL AND RolloutConfigFolderGUID IS NULL)
		AND (RolloutConfigFolderName = @Original_RolloutConfigFolderName OR @Original_RolloutConfigFolderName IS NULL AND RolloutConfigFolderName IS NULL)
		AND (RolloutConfigFolderPath = @Original_RolloutConfigFolderPath OR @Original_RolloutConfigFolderPath IS NULL AND RolloutConfigFolderPath IS NULL);
	SELECT A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName,
		COUNT(B.RolloutConfigFolderPath) AS ChildRolloutConfigFolderCount, COUNT(C.RolloutConfigPath) AS ChildRolloutConfigCount
		FROM RolloutConfigFolder AS A
		LEFT OUTER JOIN RolloutConfigFolder AS B
		ON A.RolloutConfigFolderID = B.RolloutConfigFolderPath
		LEFT OUTER JOIN RolloutConfig AS C
		ON A.RolloutConfigFolderID = C.RolloutConfigPath
		WHERE (A.RolloutConfigFolderID = @RolloutConfigFolderID)
		GROUP BY A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName;
GO

-- Procedure for deleting a rollout config folder
PRINT 'Creating procedure to delete a rollout config folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigFolderDelete' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigFolderDelete];
GO

CREATE PROCEDURE [RolloutConfigFolderDelete]
(
	@Original_RolloutConfigFolderID int,
	@Original_RolloutConfigFolderGUID binary(16),
	@Original_RolloutConfigFolderName nvarchar(256),
	@Original_RolloutConfigFolderPath int
)
AS
	SET NOCOUNT OFF;
	DELETE FROM RolloutConfigFolder
		WHERE (RolloutConfigFolderID = @Original_RolloutConfigFolderID)
		AND (RolloutConfigFolderGUID = @Original_RolloutConfigFolderGUID OR @Original_RolloutConfigFolderGUID IS NULL AND RolloutConfigFolderGUID IS NULL)
		AND (RolloutConfigFolderName = @Original_RolloutConfigFolderName OR @Original_RolloutConfigFolderName IS NULL AND RolloutConfigFolderName IS NULL)
		AND (RolloutConfigFolderPath = @Original_RolloutConfigFolderPath OR @Original_RolloutConfigFolderPath IS NULL AND RolloutConfigFolderPath IS NULL);
GO

-- Procedure for selecting a rollout config
PRINT 'Creating procedure to select a rollout config'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSelect' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSelect];
GO

CREATE PROCEDURE [RolloutConfigSelect]
(
	@RolloutConfigGUID binary(16)
)
AS
	SET NOCOUNT ON;
	SELECT RolloutConfigID, RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath
		FROM RolloutConfig WHERE (RolloutConfigGUID = @RolloutConfigGUID);
GO

-- Procedure for selecting a rollout config
PRINT 'Creating procedure to select a rollout config'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSelectID' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSelectID];
GO

CREATE PROCEDURE [RolloutConfigSelectID]
(
	@RolloutConfigID int
)
AS
	SET NOCOUNT ON;
	SELECT RolloutConfigID, RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath
		FROM RolloutConfig WHERE (RolloutConfigID = @RolloutConfigID);
GO

-- Procedure for selecting child rollout configs
PRINT 'Creating procedure to select child rollout configs'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigChildSelect' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigChildSelect];
GO
CREATE PROCEDURE [RolloutConfigChildSelect]
(
	@RolloutConfigPath int = NULL
)
AS
	SET NOCOUNT ON;
	IF (@RolloutConfigPath IS NULL)
		SELECT RolloutConfigID, RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath
			FROM RolloutConfig
			WHERE (RolloutConfigPath IS NULL)
			ORDER BY RolloutConfigName;
	ELSE
		SELECT RolloutConfigID, RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath
			FROM RolloutConfig
			WHERE (RolloutConfigPath = @RolloutConfigPath)
			ORDER BY RolloutConfigName;
GO

-- Procedure for inserting a rollout config
PRINT 'Creating procedure to insert a rollout config'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigInsert' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigInsert];
GO
CREATE PROCEDURE [RolloutConfigInsert]
(
	@RolloutConfigGUID binary(16),
	@RolloutConfigPath int,
	@RolloutConfigName nvarchar(256),
	@RolloutConfigVersion nvarchar(64),
	@ModifyDateTime datetime,
	@Description nvarchar(256),
	@RepositoryPath nvarchar(256)
)
AS
	SET NOCOUNT OFF;
	INSERT INTO RolloutConfig (RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath)
		VALUES (@RolloutConfigGUID, @RolloutConfigPath, @RolloutConfigName, @RolloutConfigVersion, @ModifyDateTime, @Description, @RepositoryPath);
	SELECT RolloutConfigID, RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath
		FROM RolloutConfig
		WHERE (RolloutConfigID = @@IDENTITY);
GO

-- Procedure for updating a rollout config
PRINT 'Creating procedure to update a rollout config'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigUpdate' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigUpdate];
GO
CREATE PROCEDURE [RolloutConfigUpdate]
(
	@RolloutConfigGUID binary(16),
	@RolloutConfigPath int,
	@RolloutConfigName nvarchar(256),
	@RolloutConfigVersion nvarchar(64),
	@ModifyDateTime datetime,
	@Description nvarchar(256),
	@RepositoryPath nvarchar(256),
	@Original_RolloutConfigID int,
	@Original_Description nvarchar(256),
	@Original_RepositoryPath nvarchar(256),
	@Original_RolloutConfigGUID binary(16),
	@Original_RolloutConfigName nvarchar(256),
	@Original_RolloutConfigPath int,
	@Original_RolloutConfigVersion nvarchar(64),
	@Original_ModifyDateTime datetime,
	@RolloutConfigID int
)
AS
	SET NOCOUNT OFF;
	UPDATE RolloutConfig
		SET RolloutConfigGUID = @RolloutConfigGUID, RolloutConfigPath = @RolloutConfigPath, RolloutConfigName = @RolloutConfigName,
		RolloutConfigVersion = @RolloutConfigVersion, ModifyDateTime = @ModifyDateTime, Description = @Description, RepositoryPath = @RepositoryPath
		WHERE (RolloutConfigID = @Original_RolloutConfigID)
		AND (Description = @Original_Description OR @Original_Description IS NULL AND Description IS NULL)
		AND (RepositoryPath = @Original_RepositoryPath OR @Original_RepositoryPath IS NULL AND RepositoryPath IS NULL)
		AND (RolloutConfigGUID = @Original_RolloutConfigGUID OR @Original_RolloutConfigGUID IS NULL AND RolloutConfigGUID IS NULL)
		AND (RolloutConfigName = @Original_RolloutConfigName)
		AND (RolloutConfigPath = @Original_RolloutConfigPath OR @Original_RolloutConfigPath IS NULL AND RolloutConfigPath IS NULL)
		AND (RolloutConfigVersion = @Original_RolloutConfigVersion)
		AND (ModifyDateTime = @Original_ModifyDateTime);
	SELECT RolloutConfigID, RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath
		FROM RolloutConfig
		WHERE (RolloutConfigID = @RolloutConfigID);
GO

-- Procedure for deleting a rollout config
PRINT 'Creating procedure to delete a rollout config'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigDelete' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigDelete];
GO
CREATE PROCEDURE [RolloutConfigDelete]
(
	@Original_RolloutConfigID int,
	@Original_Description nvarchar(256),
	@Original_RepositoryPath nvarchar(256),
	@Original_RolloutConfigGUID binary(16),
	@Original_RolloutConfigName nvarchar(256),
	@Original_RolloutConfigPath int,
	@Original_RolloutConfigVersion nvarchar(64),
	@Original_ModifyDateTime datetime
)
AS
	SET NOCOUNT OFF;
	BEGIN TRANSACTION;
	DELETE FROM RolloutConfigSetting
		WHERE (RolloutConfig = @Original_RolloutConfigID)
	DELETE FROM RolloutConfig
		WHERE (RolloutConfigID = @Original_RolloutConfigID)
		AND (Description = @Original_Description OR @Original_Description IS NULL AND Description IS NULL)
		AND (RepositoryPath = @Original_RepositoryPath OR @Original_RepositoryPath IS NULL AND RepositoryPath IS NULL)
		AND (RolloutConfigGUID = @Original_RolloutConfigGUID OR @Original_RolloutConfigGUID IS NULL AND RolloutConfigGUID IS NULL)
		AND (RolloutConfigName = @Original_RolloutConfigName)
		AND (RolloutConfigPath = @Original_RolloutConfigPath OR @Original_RolloutConfigPath IS NULL AND RolloutConfigPath IS NULL)
		AND (RolloutConfigVersion = @Original_RolloutConfigVersion)
		AND (ModifyDateTime = @Original_ModifyDateTime);
	COMMIT TRANSACTION;
GO

-- Procedure for selecting rollout config settings
PRINT 'Creating procedure to select rollout config settings'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSettingSelect' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSettingSelect];
GO
CREATE PROCEDURE [RolloutConfigSettingSelect]
(
	@RolloutConfig int
)
AS
	SET NOCOUNT ON;
	SELECT RolloutConfigSettingID, RolloutConfig, SettingType, SettingName, SettingValue
		FROM RolloutConfigSetting
		WHERE (RolloutConfig = @RolloutConfig);
GO

-- Procedure for inserting rollout config settings
PRINT 'Creating procedure to insert rollout config settings'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSettingInsert' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSettingInsert];
GO
CREATE PROCEDURE [RolloutConfigSettingInsert]
(
	@RolloutConfig int,
	@SettingType nvarchar(64),
	@SettingName nvarchar(64),
	@SettingValue nvarchar(256)
)
AS
	SET NOCOUNT OFF;
	INSERT INTO RolloutConfigSetting(RolloutConfig, SettingType, SettingName, SettingValue)
		VALUES (@RolloutConfig, @SettingType, @SettingName, @SettingValue);
	SELECT RolloutConfigSettingID, RolloutConfig, SettingType, SettingName, SettingValue
		FROM RolloutConfigSetting
		WHERE (RolloutConfigSettingID = @@IDENTITY);
GO

-- Procedure for updating rollout config settings
PRINT 'Creating procedure to update rollout config settings'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSettingUpdate' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSettingUpdate];
GO
CREATE PROCEDURE [RolloutConfigSettingUpdate]
(
	@RolloutConfig int,
	@SettingType nvarchar(64),
	@SettingName nvarchar(64),
	@SettingValue nvarchar(256),
	@Original_RolloutConfigSettingID int,
	@Original_RolloutConfig int,
	@Original_SettingName nvarchar(64),
	@Original_SettingType nvarchar(64),
	@Original_SettingValue nvarchar(256),
	@RolloutConfigSettingID int
)
AS
	SET NOCOUNT OFF;
	UPDATE RolloutConfigSetting
		SET RolloutConfig = @RolloutConfig, SettingType = @SettingType, SettingName = @SettingName, SettingValue = @SettingValue
		WHERE (RolloutConfigSettingID = @Original_RolloutConfigSettingID)
		AND (RolloutConfig = @Original_RolloutConfig OR @Original_RolloutConfig IS NULL AND RolloutConfig IS NULL)
		AND (SettingName = @Original_SettingName)
		AND (SettingType = @Original_SettingType OR @Original_SettingType IS NULL AND SettingType IS NULL)
		AND (SettingValue = @Original_SettingValue OR @Original_SettingValue IS NULL AND SettingValue IS NULL);
	SELECT RolloutConfigSettingID, RolloutConfig, SettingType, SettingName, SettingValue
		FROM RolloutConfigSetting
		WHERE (RolloutConfigSettingID = @RolloutConfigSettingID);
GO

-- Procedure for delete rollout config settings
PRINT 'Creating procedure to delete rollout config settings'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSettingDelete' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSettingDelete];
GO
CREATE PROCEDURE [RolloutConfigSettingDelete]
(
	@Original_RolloutConfigSettingID int,
	@Original_RolloutConfig int,
	@Original_SettingName nvarchar(64),
	@Original_SettingType nvarchar(64),
	@Original_SettingValue nvarchar(256)
)
AS
	SET NOCOUNT OFF;
	DELETE FROM RolloutConfigSetting
		WHERE (RolloutConfigSettingID = @Original_RolloutConfigSettingID)
		AND (RolloutConfig = @Original_RolloutConfig OR @Original_RolloutConfig IS NULL AND RolloutConfig IS NULL)
		AND (SettingName = @Original_SettingName)
		AND (SettingType = @Original_SettingType OR @Original_SettingType IS NULL AND SettingType IS NULL)
		AND (SettingValue = @Original_SettingValue OR @Original_SettingValue IS NULL AND SettingValue IS NULL);
GO

-- Procedure for selecting bootable wim reference in rollout config settings
PRINT 'Creating procedure to select bootable wim reference in rollout config settings'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSettingSelectBySettingTypeSettingNameSettingValue' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSettingSelectBySettingTypeSettingNameSettingValue]
GO
CREATE PROCEDURE [RolloutConfigSettingSelectBySettingTypeSettingNameSettingValue]
(
	@SettingType nvarchar(64),
	@SettingName nvarchar(64),
	@SettingValue nvarchar(256)
)
AS
	SET NOCOUNT OFF;
	SELECT RolloutConfigSettingID, RolloutConfig, SettingType, SettingName, SettingValue
		FROM RolloutConfigSetting
		WHERE SettingType = @SettingType
			AND	SettingName = @SettingName
			AND SettingValue LIKE '%\{' + @SettingValue + '}\%.wim'

	return @@ROWCOUNT

GO

-- Rollout status section
IF NOT EXISTS(SELECT * FROM RolloutStatusType)
BEGIN
	SET IDENTITY_INSERT RolloutStatusType ON
	INSERT INTO RolloutStatusType (Name, RolloutStatusTypeID)
		VALUES ('ComputerBuild', 1)
	INSERT INTO RolloutStatusType (Name, RolloutStatusTypeID)
		VALUES ('ComputerDataCapture', 2)
	INSERT INTO RolloutStatusType (Name, RolloutStatusTypeID)
		VALUES ('ComputerImageCapture', 3)
	SET IDENTITY_INSERT RolloutStatusType OFF
END
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusTypeGetByID')
	DROP PROCEDURE RolloutStatusTypeGetByID
GO
CREATE PROCEDURE RolloutStatusTypeGetByID
	@RolloutStatusTypeID	int
AS
	SELECT	RolloutStatusTypeID,
		[Name]
	FROM	RolloutStatusType
	WHERE	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusTypeGetByName')
	DROP PROCEDURE RolloutStatusTypeGetByName
GO
CREATE PROCEDURE RolloutStatusTypeGetByName
	@Name			nvarchar(64)
AS
	SELECT	RolloutStatusTypeID,
		[Name]
	FROM	RolloutStatusType
	WHERE	[Name] = @Name
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusTypeGetAll')
	DROP PROCEDURE RolloutStatusTypeGetAll
GO
CREATE PROCEDURE RolloutStatusTypeGetAll
AS
	SELECT	RolloutStatusTypeID,
		[Name]
	FROM	RolloutStatusType
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusTypeBind')
	DROP PROCEDURE RolloutStatusTypeBind
GO
CREATE PROCEDURE RolloutStatusTypeBind
	@Name			nvarchar(64)
AS
	DECLARE	@ID	int

	SELECT	@ID = RolloutStatusTypeID
	FROM	RolloutStatusType
	WHERE	[Name] = @Name

	IF @@ROWCOUNT = 0
	BEGIN
		SET @ID = 0
	END

	RETURN @ID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusGetBySessionNameStepNumberTypeID')
	DROP PROCEDURE RolloutStatusGetBySessionNameStepNumberTypeID
GO
CREATE PROCEDURE RolloutStatusGetBySessionNameStepNumberTypeID
	@SessionName		nvarchar(64),
	@StepNumber		int,
	@RolloutStatusTypeID	int
AS
	SELECT	SessionName,
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	FROM	RolloutStatus
	WHERE	SessionName = @SessionName
	  AND	StepNumber = @StepNumber
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusGetAllBySessionNameStepNumber')
	DROP PROCEDURE RolloutStatusGetAllBySessionNameStepNumber
GO
CREATE PROCEDURE RolloutStatusGetAllBySessionNameStepNumber
	@SessionName		nvarchar(64),
	@StepNumber		int
AS
	SELECT	SessionName,
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	FROM	RolloutStatus
	WHERE	SessionName = @SessionName
	  AND	StepNumber = @StepNumber
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusGetAllBySessionName')
	DROP PROCEDURE RolloutStatusGetAllBySessionName
GO
CREATE PROCEDURE RolloutStatusGetAllBySessionName
	@SessionName		nvarchar(64)
AS
	SELECT	SessionName,
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	FROM	RolloutStatus
	WHERE	SessionName = @SessionName
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusGetAll')
	DROP PROCEDURE RolloutStatusGetAll
GO
CREATE PROCEDURE RolloutStatusGetAll
AS
	SELECT	SessionName,
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	FROM	RolloutStatus
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusAdd')
	DROP PROCEDURE RolloutStatusAdd
GO
CREATE PROCEDURE RolloutStatusAdd
	@SessionName		nvarchar(64),
	@RolloutStatusTypeID	int,
	@ComputerID		int,
	@StepNumber		int,
	@Description		nvarchar(256),
	@Status			nvarchar(64),
	@StatusMessage		ntext = NULL,
	@Phase			nvarchar(256) = NULL
AS
	INSERT INTO RolloutStatus (
		SessionName,
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
	VALUES (
		@SessionName,
		@RolloutStatusTypeID,
		@ComputerID,
		@StepNumber,
		@Description,
		@Status,
		@StatusMessage,
		@Phase
	)
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSchema')
	DROP PROCEDURE RolloutStatusSchema
GO
CREATE PROCEDURE RolloutStatusSchema
AS
	SELECT
	'IF OBJECT_ID(''tempdb..#RolloutStatus'') IS NULL' +
	'	CREATE TABLE #RolloutStatus (' +
	'		SessionName		nvarchar(64) '+dbo.ColumnCollate('RolloutStatus', 'SessionName')+' NOT NULL,' +
	'		RolloutStatusTypeID	int NOT NULL,' +
	'		ComputerID		int NULL,' +
	'		StepNumber		int NOT NULL,' +
	'		Description		nvarchar(256) '+dbo.ColumnCollate('RolloutStatus', 'Description')+' NOT NULL,' +
	'		Status			nvarchar(64) '+dbo.ColumnCollate('RolloutStatus', 'Status')+' NOT NULL,' +
	'		StatusMessage		ntext '+dbo.ColumnCollate('RolloutStatus', 'StatusMessage')+' NULL,' +
	'		Phase			nvarchar(256) '+dbo.ColumnCollate('RolloutStatus', 'Phase')+' NULL' +
	'	)' +
	'ELSE' +
	'	DELETE FROM #RolloutStatus' +
	'' AS SQL
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusTempAdd')
	DROP PROCEDURE RolloutStatusTempAdd
GO
CREATE PROCEDURE RolloutStatusTempAdd
	@SessionName		nvarchar(64),
	@RolloutStatusTypeID	int,
	@ComputerID		int,
	@StepNumber		int,
	@Description		nvarchar(256),
	@Status			nvarchar(64),
	@StatusMessage		ntext = NULL,
	@Phase			nvarchar(256) = NULL
AS
	INSERT INTO #RolloutStatus (
		SessionName,
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
	VALUES (
		@SessionName,
		@RolloutStatusTypeID,
		@ComputerID,
		@StepNumber,
		@Description,
		@Status,
		@StatusMessage,
		@Phase
	)
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusAddBatch')
	DROP PROCEDURE RolloutStatusAddBatch
GO
CREATE PROCEDURE RolloutStatusAddBatch
AS
	DECLARE @count int
	INSERT RolloutStatus (
		SessionName,
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
		SELECT	SessionName,
			RolloutStatusTypeID,
			ComputerID,
			StepNumber,
			Description,
			Status,
			StatusMessage,
			Phase
		FROM #RolloutStatus
	SET @count = @@ROWCOUNT
	DROP TABLE #RolloutStatus	-- Drop temporary table
	RETURN @count
GO

/*
Add new RolloutStatus bound to
	RolloutStatusType using SessionName
	with StepNumber, Description, Status, StatusMessage, Phase
*/

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusUpdateBySessionNameStepNumberTypeID')
	DROP PROCEDURE RolloutStatusUpdateBySessionNameStepNumberTypeID
GO
CREATE PROCEDURE RolloutStatusUpdateBySessionNameStepNumberTypeID
	@SessionName		nvarchar(64),
	@StepNumber		int,
	@RolloutStatusTypeID	int,
	@ComputerID		int,
	@Description		nvarchar(256),
	@Status			nvarchar(64),
	@StatusMessage		ntext = NULL,
	@Phase			nvarchar(256) = NULL
AS
	UPDATE	RolloutStatus
	SET	ComputerID = @ComputerID,
		Description = @Description,
		Status = @Status,
		StatusMessage = @StatusMessage,
		Phase = @Phase
	WHERE	SessionName = @SessionName
	  AND	StepNumber = @StepNumber
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusUpdateBatchBySessionNameStepNumberTypeID')
	DROP PROCEDURE RolloutStatusUpdateBatchBySessionNameStepNumberTypeID
GO
CREATE PROCEDURE RolloutStatusUpdateBatchBySessionNameStepNumberTypeID
AS
	DECLARE @count int

	CREATE INDEX [TempRolloutStatusSessionNameStepNumberTypeID]
		ON #RolloutStatus (SessionName, StepNumber, RolloutStatusTypeID)

	UPDATE	RolloutStatus
	SET	ComputerID = t.ComputerID,
		Description = t.Description,
		Status = t.Status,
		StatusMessage = t.StatusMessage,
		Phase = t.Phase
	FROM	#RolloutStatus AS t
	WHERE	RolloutStatus.SessionName = t.SessionName
	  AND	RolloutStatus.StepNumber = t.StepNumber
	  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT
	DROP TABLE #RolloutStatus	-- Drop the temporary table
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusPutBySessionNameStepNumberTypeID')
	DROP PROCEDURE RolloutStatusPutBySessionNameStepNumberTypeID
GO
CREATE PROCEDURE RolloutStatusPutBySessionNameStepNumberTypeID
	@SessionName		nvarchar(64),
	@StepNumber		int,
	@RolloutStatusTypeID	int,
	@ComputerID		int,
	@Description		nvarchar(256),
	@Status			nvarchar(64),
	@StatusMessage		ntext = NULL,
	@Phase			nvarchar(256) = NULL
AS
	IF NOT EXISTS(
		SELECT	1
		FROM	RolloutStatus
		WHERE	SessionName = @SessionName
		  AND	StepNumber = @StepNumber
		  AND	RolloutStatusTypeID = @RolloutStatusTypeID
	)
		EXEC RolloutStatusAdd @SessionName,
			@RolloutStatusTypeID,
			@ComputerID,
			@StepNumber,
			@Description,
			@Status,
			@StatusMessage,
			@Phase
	ELSE
		EXEC RolloutStatusUpdateBySessionNameStepNumberTypeID @SessionName,
			@StepNumber,
			@RolloutStatusTypeID,
			@ComputerID,
			@Description,
			@Status,
			@StatusMessage,
			@Phase
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusPutBatchBySessionNameStepNumberTypeID')
	DROP PROCEDURE RolloutStatusPutBatchBySessionNameStepNumberTypeID
GO
CREATE PROCEDURE RolloutStatusPutBatchBySessionNameStepNumberTypeID
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusSessionNameStepNumberTypeID]
		ON #RolloutStatus (SessionName, StepNumber, RolloutStatusTypeID)

	BEGIN TRANSACTION
	UPDATE RolloutStatus
		SET	ComputerID = t.ComputerID,
			Description = t.Description,
			Status = t.Status,
			StatusMessage = t.StatusMessage,
			Phase = t.Phase
		FROM	#RolloutStatus AS t
		WHERE	RolloutStatus.SessionName = t.SessionName
		  AND	RolloutStatus.StepNumber = t.StepNumber
		  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatus (
		SessionName,
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
		SELECT	SessionName,
			RolloutStatusTypeID,
			ComputerID,
			StepNumber,
			Description,
			Status,
			StatusMessage,
			Phase
		FROM	#RolloutStatus AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatus
			WHERE	RolloutStatus.SessionName = t.SessionName
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatus	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusRemoveBySessionNameStepNumberTypeID')
	DROP PROCEDURE RolloutStatusRemoveBySessionNameStepNumberTypeID
GO
CREATE PROCEDURE RolloutStatusRemoveBySessionNameStepNumberTypeID
	@SessionName		nvarchar(64),
	@StepNumber		int,
	@RolloutStatusTypeID	int
AS
	DELETE	RolloutStatus
	WHERE	SessionName = @SessionName
	  AND	StepNumber = @StepNumber
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusRemoveAllBySessionNameStepNumber')
	DROP PROCEDURE RolloutStatusRemoveAllBySessionNameStepNumber
GO
CREATE PROCEDURE RolloutStatusRemoveAllBySessionNameStepNumber
	@SessionName		nvarchar(64),
	@StepNumber		int
AS
	DELETE	RolloutStatus
	WHERE	SessionName = @SessionName
	  AND	StepNumber = @StepNumber
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusRemoveAllBySessionName')
	DROP PROCEDURE RolloutStatusRemoveAllBySessionName
GO
CREATE PROCEDURE RolloutStatusRemoveAllBySessionName
	@SessionName		nvarchar(64)
AS
	DELETE	RolloutStatus
	WHERE	SessionName = @SessionName
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusRemoveAll')
	DROP PROCEDURE RolloutStatusRemoveAll
GO
CREATE PROCEDURE RolloutStatusRemoveAll
	
AS
	DELETE	RolloutStatus
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusReplaceSubsetBySessionNameStepNumber')
	DROP PROCEDURE RolloutStatusReplaceSubsetBySessionNameStepNumber
GO
-- Replace the record subset selected by (@SessionName, @StepNumber)
-- using unique identifier (SessionName, StepNumber, RolloutStatusTypeID)
-- updating fields (ComputerID, Description, Status, StatusMessage, Phase)
CREATE PROCEDURE RolloutStatusReplaceSubsetBySessionNameStepNumber
	@SessionName		int,
	@StepNumber		int
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusBySessionNameStepNumber]
		ON #RolloutStatus (SessionName, StepNumber, RolloutStatusTypeID)

	BEGIN TRANSACTION
	DELETE RolloutStatus
		WHERE	SessionName = @SessionName
		  AND	StepNumber = @StepNumber
		AND NOT EXISTS(		-- Replacement record doesn't exist
			SELECT * FROM #RolloutStatus AS t
			WHERE	RolloutStatus.SessionName = t.SessionName
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)

	UPDATE RolloutStatus
		SET	ComputerID = t.ComputerID,
			Description = t.Description,
			Status = t.Status,
			StatusMessage = t.StatusMessage,
			Phase = t.Phase
		FROM	#RolloutStatus AS t
		WHERE	RolloutStatus.SessionName = t.SessionName
		  AND	RolloutStatus.StepNumber = t.StepNumber
		  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatus (
		SessionName,
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
		SELECT	SessionName,
			RolloutStatusTypeID,
			ComputerID,
			StepNumber,
			Description,
			Status,
			StatusMessage,
			Phase
		FROM	#RolloutStatus AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatus
			WHERE	RolloutStatus.SessionName = t.SessionName
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatus	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusRemoveBatch')
	DROP PROCEDURE RolloutStatusRemoveBatch
GO

CREATE PROCEDURE RolloutStatusRemoveBatch
AS

	CREATE INDEX [TempRolloutStatusBySessionNameStepNumber]
		ON #RolloutStatus (SessionName, StepNumber, RolloutStatusTypeID)	

	-- Delete records from RolloutStatus identified by fields of [TempRolloutStatusBySessionNameStepNumber]
	DELETE RolloutStatus
	WHERE EXISTS(
			SELECT * FROM #RolloutStatus AS t
			WHERE	RolloutStatus.SessionName = t.SessionName
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	DROP TABLE #RolloutStatus      -- Drop the temporary table
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusReplaceSubsetBySessionName')
	DROP PROCEDURE RolloutStatusReplaceSubsetBySessionName
GO
-- Replace the record subset selected by (@SessionName)
-- using unique identifier (SessionName, StepNumber, RolloutStatusTypeID)
-- updating fields (ComputerID, Description, Status, StatusMessage, Phase)
CREATE PROCEDURE RolloutStatusReplaceSubsetBySessionName
	@SessionName		nvarchar(64)
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusBySessionName]
		ON #RolloutStatus (SessionName, StepNumber, RolloutStatusTypeID)

	BEGIN TRANSACTION
	DELETE RolloutStatus
		WHERE	SessionName = @SessionName
		AND NOT EXISTS(		-- Replacement record doesn't exist
			SELECT * FROM #RolloutStatus AS t
			WHERE	RolloutStatus.SessionName = t.SessionName
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)

	UPDATE RolloutStatus
		SET	ComputerID = t.ComputerID,
			Description = t.Description,
			Status = t.Status,
			StatusMessage = t.StatusMessage,
			Phase = t.Phase
		FROM	#RolloutStatus AS t
		WHERE	RolloutStatus.SessionName = t.SessionName
		  AND	RolloutStatus.StepNumber = t.StepNumber
		  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatus (
		SessionName,
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
		SELECT	SessionName,
			RolloutStatusTypeID,
			ComputerID,
			StepNumber,
			Description,
			Status,
			StatusMessage,
			Phase
		FROM	#RolloutStatus AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatus
			WHERE	RolloutStatus.SessionName = t.SessionName
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatus	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryGetBySessionNameRolloutStatusTypeID')
	DROP PROCEDURE RolloutStatusSummaryGetBySessionNameRolloutStatusTypeID
GO
CREATE PROCEDURE RolloutStatusSummaryGetBySessionNameRolloutStatusTypeID
	@SessionName		nvarchar(64),
	@RolloutStatusTypeID	int = 1
AS
	SELECT	SessionName,
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	FROM	RolloutStatusSummary
	WHERE	SessionName = @SessionName
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryGetAllBySessionName')
	DROP PROCEDURE RolloutStatusSummaryGetAllBySessionName
GO
CREATE PROCEDURE RolloutStatusSummaryGetAllBySessionName
	@SessionName		nvarchar(64)
AS
	SELECT	SessionName,
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	FROM	RolloutStatusSummary
	WHERE	SessionName = @SessionName
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryGetAllByLastStatus')
	DROP PROCEDURE RolloutStatusSummaryGetAllByLastStatus
GO
CREATE PROCEDURE RolloutStatusSummaryGetAllByLastStatus
	@LastStatus		nvarchar(64) = NULL
AS
	SELECT	SessionName,
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	FROM	RolloutStatusSummary
	WHERE	(LastStatus = @LastStatus OR (LastStatus IS NULL AND @LastStatus IS NULL))
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryGetAll')
	DROP PROCEDURE RolloutStatusSummaryGetAll
GO
CREATE PROCEDURE RolloutStatusSummaryGetAll
AS
	SELECT	SessionName,
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	FROM	RolloutStatusSummary
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryAdd')
	DROP PROCEDURE RolloutStatusSummaryAdd
GO
CREATE PROCEDURE RolloutStatusSummaryAdd
	@SessionName		nvarchar(64),
	@RolloutStatusTypeID	int = 1,
	@ComputerID		int,
	@LastStatus		nvarchar(64) = NULL,
	@LastStep		int = NULL,
	@LastStepDescription	nvarchar(256) = NULL,
	@StepCount		int = NULL,
	@StartDateTime		datetime = NULL,
	@LastDateTime		datetime = NULL,
	@CompleteDateTime	datetime = NULL
AS
	INSERT INTO RolloutStatusSummary (
		SessionName,
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	)
	VALUES (
		@SessionName,
		@RolloutStatusTypeID,
		@ComputerID,
		@LastStatus,
		@LastStep,
		@LastStepDescription,
		@StepCount,
		@StartDateTime,
		@LastDateTime,
		@CompleteDateTime
	)
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummarySchema')
	DROP PROCEDURE RolloutStatusSummarySchema
GO
CREATE PROCEDURE RolloutStatusSummarySchema
AS
	SELECT
	'IF OBJECT_ID(''tempdb..#RolloutStatusSummary'') IS NULL' +
	'	CREATE TABLE #RolloutStatusSummary (' +
	'		SessionName		nvarchar(64) '+dbo.ColumnCollate('RolloutStatusSummary', 'SessionName')+' NULL,' +
	'		RolloutStatusTypeID	int NOT NULL DEFAULT 1,' +
	'		ComputerID		int NULL,' +
	'		LastStatus		nvarchar(64) '+dbo.ColumnCollate('RolloutStatusSummary', 'LastStatus')+' NULL,' +
	'		LastStep		int NULL,' +
	'		LastStepDescription	nvarchar(256) '+dbo.ColumnCollate('RolloutStatusSummary', 'LastStepDescription')+' NULL,' +
	'		StepCount		int NULL,' +
	'		StartDateTime		datetime NULL,' +
	'		LastDateTime		datetime NULL,' +
	'		CompleteDateTime	datetime NULL' +
	'	)' +
	'ELSE' +
	'	DELETE FROM #RolloutStatusSummary' +
	'' AS SQL
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryTempAdd')
	DROP PROCEDURE RolloutStatusSummaryTempAdd
GO
CREATE PROCEDURE RolloutStatusSummaryTempAdd
	@SessionName		nvarchar(64),
	@RolloutStatusTypeID	int = 1,
	@ComputerID		int,
	@LastStatus		nvarchar(64) = NULL,
	@LastStep		int = NULL,
	@LastStepDescription	nvarchar(256) = NULL,
	@StepCount		int = NULL,
	@StartDateTime		datetime = NULL,
	@LastDateTime		datetime = NULL,
	@CompleteDateTime	datetime = NULL
AS
	INSERT INTO #RolloutStatusSummary (
		SessionName,
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	)
	VALUES (
		@SessionName,
		@RolloutStatusTypeID,
		@ComputerID,
		@LastStatus,
		@LastStep,
		@LastStepDescription,
		@StepCount,
		@StartDateTime,
		@LastDateTime,
		@CompleteDateTime
	)
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryAddBatch')
	DROP PROCEDURE RolloutStatusSummaryAddBatch
GO
CREATE PROCEDURE RolloutStatusSummaryAddBatch
AS
	DECLARE @count int
	INSERT RolloutStatusSummary (
		SessionName,
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	)
		SELECT	SessionName,
			RolloutStatusTypeID,
			ComputerID,
			LastStatus,
			LastStep,
			LastStepDescription,
			StepCount,
			StartDateTime,
			LastDateTime,
			CompleteDateTime
		FROM #RolloutStatusSummary
	SET @count = @@ROWCOUNT
	DROP TABLE #RolloutStatusSummary	-- Drop temporary table
	RETURN @count
GO

/*
Add new RolloutStatusSummary bound to
	RolloutStatusType using SessionName
	with LastStatus, LastStep, LastStepDescription, StepCount, StartDateTime, LastDateTime, CompleteDateTime
*/

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryUpdateBySessionNameRolloutStatusTypeID')
	DROP PROCEDURE RolloutStatusSummaryUpdateBySessionNameRolloutStatusTypeID
GO
CREATE PROCEDURE RolloutStatusSummaryUpdateBySessionNameRolloutStatusTypeID
	@SessionName		nvarchar(64),
	@RolloutStatusTypeID	int = 1,
	@ComputerID		int,
	@LastStatus		nvarchar(64) = NULL,
	@LastStep		int = NULL,
	@LastStepDescription	nvarchar(256) = NULL,
	@StepCount		int = NULL,
	@StartDateTime		datetime = NULL,
	@LastDateTime		datetime = NULL,
	@CompleteDateTime	datetime = NULL
AS
	UPDATE	RolloutStatusSummary
	SET	ComputerID = @ComputerID,
		LastStatus = @LastStatus,
		LastStep = @LastStep,
		LastStepDescription = @LastStepDescription,
		StepCount = @StepCount,
		StartDateTime = @StartDateTime,
		LastDateTime = @LastDateTime,
		CompleteDateTime = @CompleteDateTime
	WHERE	SessionName = @SessionName
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryUpdateBatchBySessionNameRolloutStatusTypeID')
	DROP PROCEDURE RolloutStatusSummaryUpdateBatchBySessionNameRolloutStatusTypeID
GO
CREATE PROCEDURE RolloutStatusSummaryUpdateBatchBySessionNameRolloutStatusTypeID
AS
	DECLARE @count int

	CREATE INDEX [TempRolloutStatusSummarySessionNameRolloutStatusTypeID]
		ON #RolloutStatusSummary (SessionName, RolloutStatusTypeID)

	UPDATE	RolloutStatusSummary
	SET	ComputerID = t.ComputerID,
		LastStatus = t.LastStatus,
		LastStep = t.LastStep,
		LastStepDescription = t.LastStepDescription,
		StepCount = t.StepCount,
		StartDateTime = t.StartDateTime,
		LastDateTime = t.LastDateTime,
		CompleteDateTime = t.CompleteDateTime
	FROM	#RolloutStatusSummary AS t
	WHERE	RolloutStatusSummary.SessionName = t.SessionName
	  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT
	DROP TABLE #RolloutStatusSummary	-- Drop the temporary table
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryPutBySessionNameRolloutStatusTypeID')
	DROP PROCEDURE RolloutStatusSummaryPutBySessionNameRolloutStatusTypeID
GO
CREATE PROCEDURE RolloutStatusSummaryPutBySessionNameRolloutStatusTypeID
	@SessionName		nvarchar(64),
	@RolloutStatusTypeID	int = 1,
	@ComputerID		int,
	@LastStatus		nvarchar(64) = NULL,
	@LastStep		int = NULL,
	@LastStepDescription	nvarchar(256) = NULL,
	@StepCount		int = NULL,
	@StartDateTime		datetime = NULL,
	@LastDateTime		datetime = NULL,
	@CompleteDateTime	datetime = NULL
AS
	IF NOT EXISTS(
		SELECT	1
		FROM	RolloutStatusSummary
		WHERE	SessionName = @SessionName
		  AND	RolloutStatusTypeID = @RolloutStatusTypeID
	)
		EXEC RolloutStatusSummaryAdd @SessionName,
			@RolloutStatusTypeID,
			@ComputerID,
			@LastStatus,
			@LastStep,
			@LastStepDescription,
			@StepCount,
			@StartDateTime,
			@LastDateTime,
			@CompleteDateTime
	ELSE
		EXEC RolloutStatusSummaryUpdateBySessionNameRolloutStatusTypeID @SessionName,
			@RolloutStatusTypeID,
			@ComputerID,
			@LastStatus,
			@LastStep,
			@LastStepDescription,
			@StepCount,
			@StartDateTime,
			@LastDateTime,
			@CompleteDateTime
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryPutBatchBySessionNameRolloutStatusTypeID')
	DROP PROCEDURE RolloutStatusSummaryPutBatchBySessionNameRolloutStatusTypeID
GO
CREATE PROCEDURE RolloutStatusSummaryPutBatchBySessionNameRolloutStatusTypeID
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusSummarySessionNameRolloutStatusTypeID]
		ON #RolloutStatusSummary (SessionName, RolloutStatusTypeID)

	BEGIN TRANSACTION
	UPDATE RolloutStatusSummary
		SET	ComputerID = t.ComputerID,
			LastStatus = t.LastStatus,
			LastStep = t.LastStep,
			LastStepDescription = t.LastStepDescription,
			StepCount = t.StepCount,
			StartDateTime = t.StartDateTime,
			LastDateTime = t.LastDateTime,
			CompleteDateTime = t.CompleteDateTime
		FROM	#RolloutStatusSummary AS t
		WHERE	RolloutStatusSummary.SessionName = t.SessionName
		  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatusSummary (
		SessionName,
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	)
		SELECT	SessionName,
			RolloutStatusTypeID,
			ComputerID,
			LastStatus,
			LastStep,
			LastStepDescription,
			StepCount,
			StartDateTime,
			LastDateTime,
			CompleteDateTime
		FROM	#RolloutStatusSummary AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatusSummary
			WHERE	RolloutStatusSummary.SessionName = t.SessionName
			  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatusSummary	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryRemoveBySessionNameRolloutStatusTypeID')
	DROP PROCEDURE RolloutStatusSummaryRemoveBySessionNameRolloutStatusTypeID
GO
CREATE PROCEDURE RolloutStatusSummaryRemoveBySessionNameRolloutStatusTypeID
	@SessionName		nvarchar(64),
	@RolloutStatusTypeID	int = 1
AS
	DELETE	RolloutStatusSummary
	WHERE	SessionName = @SessionName
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryRemoveAllBySessionName')
	DROP PROCEDURE RolloutStatusSummaryRemoveAllBySessionName
GO
CREATE PROCEDURE RolloutStatusSummaryRemoveAllBySessionName
	@SessionName		nvarchar(64)
AS
	DELETE	RolloutStatusSummary
	WHERE	SessionName = @SessionName
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryRemoveAllByLastStatus')
	DROP PROCEDURE RolloutStatusSummaryRemoveAllByLastStatus
GO
CREATE PROCEDURE RolloutStatusSummaryRemoveAllByLastStatus
	@LastStatus		nvarchar(64) = NULL
AS
	DELETE	RolloutStatusSummary
	WHERE	(LastStatus = @LastStatus OR (LastStatus IS NULL AND @LastStatus IS NULL))
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryRemoveAll')
	DROP PROCEDURE RolloutStatusSummaryRemoveAll
GO
CREATE PROCEDURE RolloutStatusSummaryRemoveAll
	
AS
	DELETE	RolloutStatusSummary
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryReplaceSubsetBySessionName')
	DROP PROCEDURE RolloutStatusSummaryReplaceSubsetBySessionName
GO
-- Replace the record subset selected by (@SessionName)
-- using unique identifier (SessionName, RolloutStatusTypeID)
-- updating fields (ComputerID, LastStatus, LastStep, LastStepDescription, StepCount, StartDateTime, LastDateTime, CompleteDateTime)
CREATE PROCEDURE RolloutStatusSummaryReplaceSubsetBySessionName
	@SessionName		nvarchar(64)
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusSummaryBySessionName]
		ON #RolloutStatusSummary (SessionName, RolloutStatusTypeID)

	BEGIN TRANSACTION
	DELETE RolloutStatusSummary
		WHERE	SessionName = @SessionName
		AND NOT EXISTS(		-- Replacement record doesn't exist
			SELECT * FROM #RolloutStatusSummary AS t
			WHERE	RolloutStatusSummary.SessionName = t.SessionName
			  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
		)

	UPDATE RolloutStatusSummary
		SET	ComputerID = t.ComputerID,
			LastStatus = t.LastStatus,
			LastStep = t.LastStep,
			LastStepDescription = t.LastStepDescription,
			StepCount = t.StepCount,
			StartDateTime = t.StartDateTime,
			LastDateTime = t.LastDateTime,
			CompleteDateTime = t.CompleteDateTime
		FROM	#RolloutStatusSummary AS t
		WHERE	RolloutStatusSummary.SessionName = t.SessionName
		  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatusSummary (
		SessionName,
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	)
		SELECT	SessionName,
			RolloutStatusTypeID,
			ComputerID,
			LastStatus,
			LastStep,
			LastStepDescription,
			StepCount,
			StartDateTime,
			LastDateTime,
			CompleteDateTime
		FROM	#RolloutStatusSummary AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatusSummary
			WHERE	RolloutStatusSummary.SessionName = t.SessionName
			  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatusSummary	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryRemoveBatch')
	DROP PROCEDURE RolloutStatusSummaryRemoveBatch
GO

CREATE PROCEDURE RolloutStatusSummaryRemoveBatch
AS

	CREATE INDEX [TempRolloutStatusSummaryBySessionName]
		ON #RolloutStatusSummary (SessionName, RolloutStatusTypeID)	

	-- Delete records from RolloutStatusSummary identified by fields of [TempRolloutStatusSummaryBySessionName]
	DELETE RolloutStatusSummary
	WHERE EXISTS(
			SELECT * FROM #RolloutStatusSummary AS t
			WHERE	RolloutStatusSummary.SessionName = t.SessionName
			  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	DROP TABLE #RolloutStatusSummary      -- Drop the temporary table
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryReplaceSubsetByLastStatus')
	DROP PROCEDURE RolloutStatusSummaryReplaceSubsetByLastStatus
GO
-- Replace the record subset selected by (@LastStatus)
-- using unique identifier (ComputerID, RolloutStatusTypeID)
-- updating fields (LastStep, LastStepDescription, StepCount, StartDateTime, LastDateTime, CompleteDateTime)
CREATE PROCEDURE RolloutStatusSummaryReplaceSubsetByLastStatus
	@LastStatus		nvarchar(64) = NULL
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusSummaryByLastStatus]
		ON #RolloutStatusSummary (ComputerID, RolloutStatusTypeID)

	BEGIN TRANSACTION
	DELETE RolloutStatusSummary
		WHERE	(LastStatus = @LastStatus OR (LastStatus IS NULL AND @LastStatus IS NULL))
		AND NOT EXISTS(		-- Replacement record doesn't exist
			SELECT * FROM #RolloutStatusSummary AS t
			WHERE	RolloutStatusSummary.SessionName = t.SessionName
			  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
		)

	UPDATE RolloutStatusSummary
		SET	ComputerID = t.ComputerID,
			LastStep = t.LastStep,
			LastStepDescription = t.LastStepDescription,
			StepCount = t.StepCount,
			StartDateTime = t.StartDateTime,
			LastDateTime = t.LastDateTime,
			CompleteDateTime = t.CompleteDateTime
		FROM	#RolloutStatusSummary AS t
		WHERE	RolloutStatusSummary.SessionName = t.SessionName
		  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatusSummary (
		SessionName,
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	)
		SELECT	SessionName,
			RolloutStatusTypeID,
			ComputerID,
			LastStatus,
			LastStep,
			LastStepDescription,
			StepCount,
			StartDateTime,
			LastDateTime,
			CompleteDateTime
		FROM	#RolloutStatusSummary AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatusSummary
			WHERE	RolloutStatusSummary.SessionName = t.SessionName
			  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatusSummary	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusReplaceSubsetBySessionNameWithDeleteOffset')
	DROP PROCEDURE RolloutStatusReplaceSubsetBySessionNameWithDeleteOffset
GO
-- Replace the record subset selected by (@SessionName), (@DeleteOffset)
-- using unique identifier (SessionName, StepNumber)
-- updating fields (ComputerID, RolloutStatusTypeID, Description, Status, StatusMessage)
-- whilst deleting entries with an StepNumber greater than the specified @DeleteOffset
CREATE PROCEDURE RolloutStatusReplaceSubsetBySessionNameWithDeleteOffset
	@SessionName		nvarchar(64),
	@RolloutStatusTypeID	int,
	@DeleteOffset		int
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusBySessionName]
		ON #RolloutStatus (SessionName, StepNumber, RolloutStatusTypeID)

	BEGIN TRANSACTION
	DELETE RolloutStatus
		WHERE	SessionName = @SessionName
		AND NOT EXISTS(		-- Replacement record doesn't exist
			SELECT * FROM #RolloutStatus AS t
			WHERE	RolloutStatus.SessionName = t.SessionName
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
		AND RolloutStatusTypeID = @RolloutStatusTypeID
		AND StepNumber > @DeleteOffset

	UPDATE RolloutStatus
		SET	ComputerID = t.ComputerID,
			RolloutStatusTypeID = t.RolloutStatusTypeID,
			Description = t.Description,
			Status = t.Status,
			StatusMessage = t.StatusMessage,
			Phase = t.Phase
		FROM	#RolloutStatus AS t
		WHERE	RolloutStatus.SessionName = t.SessionName
		  AND	RolloutStatus.StepNumber = t.StepNumber
		  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatus (
		SessionName,
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
		SELECT	SessionName,
			RolloutStatusTypeID,
			ComputerID,
			StepNumber,
			Description,
			Status,
			StatusMessage,
			Phase
		FROM	#RolloutStatus AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatus
			WHERE	RolloutStatus.SessionName = t.SessionName
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatus	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusGetAllBySessionNameTypeID')
	DROP PROCEDURE RolloutStatusGetAllBySessionNameTypeID
GO
CREATE PROCEDURE RolloutStatusGetAllBySessionNameTypeID
	@SessionName		nvarchar(64),
	@RolloutStatusTypeID	int
AS
	SELECT	SessionName,
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	FROM	RolloutStatus
	WHERE	SessionName = @SessionName
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
	  			ORDER BY StepNumber
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureGetBySessionName')
	DROP PROCEDURE DataCaptureGetBySessionName
GO
CREATE PROCEDURE DataCaptureGetBySessionName
	@SessionName		nvarchar(64)
AS
	SELECT	SessionName,
		ComputerID,
		DataStoreLocation,
		DataImageCaptureLocation,
		LogFileLocation,
		CaptureDateTime,
		ImageCaptureDateTime
	FROM	DataCapture
	WHERE	SessionName = @SessionName
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureGetAll')
	DROP PROCEDURE DataCaptureGetAll
GO
CREATE PROCEDURE DataCaptureGetAll
AS
	SELECT	SessionName,
		ComputerID,
		DataStoreLocation,
		DataImageCaptureLocation,
		LogFileLocation,
		CaptureDateTime,
		ImageCaptureDateTime
	FROM	DataCapture
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureAdd')
	DROP PROCEDURE DataCaptureAdd
GO
CREATE PROCEDURE DataCaptureAdd
	@SessionName		nvarchar(64),
	@ComputerID		int,
	@DataStoreLocation	nvarchar(1024) = NULL,
	@DataImageCaptureLocation nvarchar(1024) = NULL,
	@LogFileLocation	nvarchar(1024) = NULL,
	@CaptureDateTime	datetime,
	@ImageCaptureDateTime	datetime
AS
	INSERT INTO DataCapture (
		SessionName,
		ComputerID,
		DataStoreLocation,
		DataImageCaptureLocation,
		LogFileLocation,
		CaptureDateTime,
		ImageCaptureDateTime
	)
	VALUES (
		@SessionName,
		@ComputerID,
		@DataStoreLocation,
		@DataImageCaptureLocation,
		@LogFileLocation,
		@CaptureDateTime,
		@ImageCaptureDateTime
	)
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureSchema')
	DROP PROCEDURE DataCaptureSchema
GO
CREATE PROCEDURE DataCaptureSchema
AS
	SELECT
	'IF OBJECT_ID(''tempdb..#DataCapture'') IS NULL' +
	'	CREATE TABLE #DataCapture (' +
	'		SessionName		nvarchar(64) '+dbo.ColumnCollate('DataCapture', 'SessionName')+' NULL,' +
	'		ComputerID		int NOT NULL,' +
	'		DataStoreLocation	nvarchar(1024) '+dbo.ColumnCollate('DataCapture', 'DataStoreLocation')+' NULL,' +
	'		DataImageCaptureLocation nvarchar(1024) '+dbo.ColumnCollate('DataCapture', 'DataImageCaptureLocation')+' NULL,' +
	'		LogFileLocation		nvarchar(1024) '+dbo.ColumnCollate('DataCapture', 'LogFileLocation')+' NULL,' +
	'		CaptureDateTime		datetime NOT NULL DEFAULT getdate(),' +
	'		ImageCaptureDateTime	datetime NOT NULL DEFAULT getdate()' +
	'	)' +
	'ELSE' +
	'	DELETE FROM #DataCapture' +
	'' AS SQL
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureTempAdd')
	DROP PROCEDURE DataCaptureTempAdd
GO
CREATE PROCEDURE DataCaptureTempAdd
	@SessionName		nvarchar(64),
	@ComputerID		int,
	@DataStoreLocation	nvarchar(1024) = NULL,
	@DataImageCaptureLocation nvarchar(1024) = NULL,
	@LogFileLocation	nvarchar(1024) = NULL,
	@CaptureDateTime	datetime,
	@ImageCaptureDateTime	datetime
AS
	INSERT INTO #DataCapture (
		SessionName,
		ComputerID,
		DataStoreLocation,
		DataImageCaptureLocation,
		LogFileLocation,
		CaptureDateTime,
		ImageCaptureDateTime
	)
	VALUES (
		@SessionName,
		@ComputerID,
		@DataStoreLocation,
		@DataImageCaptureLocation,
		@LogFileLocation,
		@CaptureDateTime,
		@ImageCaptureDateTime
	)
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureAddBatch')
	DROP PROCEDURE DataCaptureAddBatch
GO
CREATE PROCEDURE DataCaptureAddBatch
AS
	DECLARE @count int
	INSERT DataCapture (
		SessionName,
		ComputerID,
		DataStoreLocation,
		DataImageCaptureLocation,
		LogFileLocation,
		CaptureDateTime,
		ImageCaptureDateTime
	)
		SELECT	SessionName,
			ComputerID,
			DataStoreLocation,
			DataImageCaptureLocation,
			LogFileLocation,
			CaptureDateTime,
			ImageCaptureDateTime
		FROM #DataCapture
	SET @count = @@ROWCOUNT
	DROP TABLE #DataCapture	-- Drop temporary table
	RETURN @count
GO

/*
Add new DataCapture bound to
	SessionName
	with ComputerID, DataStoreLocation, DataImageCaptureLocation, LogFileLocation, CaptureDateTime, ImageCaptureDateTime
*/

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureUpdateBySessionName')
	DROP PROCEDURE DataCaptureUpdateBySessionName
GO
CREATE PROCEDURE DataCaptureUpdateBySessionName
	@SessionName		nvarchar(64),
	@ComputerID		int,
	@DataStoreLocation	nvarchar(1024) = NULL,
	@DataImageCaptureLocation nvarchar(1024) = NULL,
	@LogFileLocation	nvarchar(1024) = NULL,
	@CaptureDateTime	datetime,
	@ImageCaptureDateTime	datetime
AS
	UPDATE	DataCapture
	SET	ComputerID = @ComputerID,
		DataStoreLocation = @DataStoreLocation,
		DataImageCaptureLocation = @DataImageCaptureLocation,
		LogFileLocation = @LogFileLocation,
		CaptureDateTime = @CaptureDateTime,
		ImageCaptureDateTime = @ImageCaptureDateTime
	WHERE	SessionName = @SessionName
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureUpdateBatchBySessionName')
	DROP PROCEDURE DataCaptureUpdateBatchBySessionName
GO
CREATE PROCEDURE DataCaptureUpdateBatchBySessionName
AS
	DECLARE @count int

	CREATE INDEX [TempDataCaptureSessionName]
		ON #DataCapture (SessionName)

	UPDATE	DataCapture
	SET	ComputerID = t.ComputerID,
		DataStoreLocation = t.DataStoreLocation,
		DataImageCaptureLocation = t.DataImageCaptureLocation,
		LogFileLocation = t.LogFileLocation,
		CaptureDateTime = t.CaptureDateTime,
		ImageCaptureDateTime = t.ImageCaptureDateTime
	FROM	#DataCapture AS t
	WHERE	DataCapture.SessionName = t.SessionName
	SET @count = @@ROWCOUNT
	DROP TABLE #DataCapture	-- Drop the temporary table
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCapturePutBySessionName')
	DROP PROCEDURE DataCapturePutBySessionName
GO
CREATE PROCEDURE DataCapturePutBySessionName
	@SessionName		nvarchar(64),
	@ComputerID		int,
	@DataStoreLocation	nvarchar(1024) = NULL,
	@DataImageCaptureLocation nvarchar(1024) = NULL,
	@LogFileLocation	nvarchar(1024) = NULL,
	@CaptureDateTime	datetime,
	@ImageCaptureDateTime	datetime
AS
	IF NOT EXISTS(
		SELECT	1
		FROM	DataCapture
		WHERE	SessionName = @SessionName
	)
		EXEC DataCaptureAdd @SessionName,
			@ComputerID,
			@DataStoreLocation,
			@DataImageCaptureLocation,
			@LogFileLocation,
			@CaptureDateTime,
			@ImageCaptureDateTime
	ELSE
		EXEC DataCaptureUpdateBySessionName @SessionName,
			@ComputerID,
			@DataStoreLocation,
			@DataImageCaptureLocation,
			@LogFileLocation,
			@CaptureDateTime,
			@ImageCaptureDateTime
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCapturePutBatchBySessionName')
	DROP PROCEDURE DataCapturePutBatchBySessionName
GO
CREATE PROCEDURE DataCapturePutBatchBySessionName
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempDataCaptureSessionName]
		ON #DataCapture (SessionName)

	BEGIN TRANSACTION
	UPDATE DataCapture
		SET	ComputerID = t.ComputerID,
			DataStoreLocation = t.DataStoreLocation,
			DataImageCaptureLocation = t.DataImageCaptureLocation,
			LogFileLocation = t.LogFileLocation,
			CaptureDateTime = t.CaptureDateTime,
			ImageCaptureDateTime = t.ImageCaptureDateTime
		FROM	#DataCapture AS t
		WHERE	DataCapture.SessionName = t.SessionName
	SET @count = @@ROWCOUNT

	INSERT DataCapture (
		SessionName,
		ComputerID,
		DataStoreLocation,
		DataImageCaptureLocation,
		LogFileLocation,
		CaptureDateTime,
		ImageCaptureDateTime
	)
		SELECT	SessionName,
			ComputerID,
			DataStoreLocation,
			DataImageCaptureLocation,
			LogFileLocation,
			CaptureDateTime,
			ImageCaptureDateTime
		FROM	#DataCapture AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM DataCapture
			WHERE	DataCapture.SessionName = t.SessionName
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #DataCapture	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureRemoveAll')
	DROP PROCEDURE DataCaptureRemoveAll
GO
CREATE PROCEDURE DataCaptureRemoveAll
	
AS
	DELETE	DataCapture
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'ComputerGetByOUIDCN')
	DROP PROCEDURE ComputerGetByOUIDCN
GO
CREATE PROCEDURE ComputerGetByOUIDCN
	@ComputerOUID		int,
	@ComputerCN		nvarchar(64)
AS
	SELECT	ComputerID,
		ComputerOUID,
		ComputerCN,
		OperatingSystemID,
		GUID
	FROM	Computer
	WHERE	ComputerOUID = @ComputerOUID
	  AND	ComputerCN = @ComputerCN
GO
