/*
 * $Id: rolloutconfig.sql,v 1.5 2007/07/24 06:31:02 pca Exp $
 *
 * Description:
 *
 * SQL Script to build the ManageSoft rollout configuration tables on Microsoft SQL
 * Server.
 *
 * The steps performed by this script are:
 * 1. Create new database
 * 2. Create database table and initialiase data (skip if the table exists)
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 *
 * Copyright 2011 Flexera Software LLC
 */
USE ManageSoft
GO

-- Create RolloutConfigFolder table
IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutConfigFolder'
)
BEGIN
	-- Create table
	PRINT 'Creating RolloutConfigFolder table'
	CREATE TABLE RolloutConfigFolder
	(
		-- Auto-generated identity #
		RolloutConfigFolderID INTEGER NOT NULL IDENTITY
			CONSTRAINT PK_RolloutConfigFolder PRIMARY KEY NONCLUSTERED,

		-- GUID of rollout config
		RolloutConfigFolderGUID BINARY(16) NULL DEFAULT(NULL),
			
		-- Parent rollout config folder
		RolloutConfigFolderPath INTEGER REFERENCES RolloutConfigFolder(RolloutConfigFolderID),
			
		-- Name of rollout config
		RolloutConfigFolderName NVARCHAR(256) NULL DEFAULT(NULL)
	);
	
	-- Create indices
	CREATE UNIQUE INDEX RolloutConfigFolderByGUID
	ON RolloutConfigFolder (RolloutConfigFolderGUID)
	CREATE UNIQUE INDEX RolloutConfigFolderByPathName
	ON RolloutConfigFolder (RolloutConfigFolderPath, RolloutConfigFolderName)
	CREATE INDEX RolloutConfigFolderByPath
	ON RolloutConfigFolder (RolloutConfigFolderPath)
END
ELSE
	PRINT 'Using existing RolloutConfigFolder table'
GO

-- Create RolloutConfig table
IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutConfig'
)
BEGIN
	-- Create table
	PRINT 'Creating RolloutConfig table'
	CREATE TABLE RolloutConfig
	(
		-- Auto-generated identity #
		RolloutConfigID INTEGER NOT NULL IDENTITY
			CONSTRAINT PK_RolloutConfig PRIMARY KEY NONCLUSTERED,

		-- GUID of rollout config
		RolloutConfigGUID BINARY(16) NULL DEFAULT(NULL),
			
		-- Parent rollout configuration folder
		RolloutConfigPath INTEGER NULL REFERENCES RolloutConfigFolder(RolloutConfigFolderID),
			
		-- Name of rollout config
		RolloutConfigName NVARCHAR(256) NOT NULL,
		
		-- Version of image
		RolloutConfigVersion NVARCHAR(64) NOT NULL,
		
		-- Modify date and time of rollout config
		ModifyDateTime DATETIME NOT NULL,
			
		-- Description of rollout config
		Description NVARCHAR(256) NULL DEFAULT(NULL),
		
		-- Path of rollout config in repository
		RepositoryPath NVARCHAR(256) NULL DEFAULT(NULL)
	);
	
	-- Create indices
	CREATE UNIQUE INDEX RolloutConfigByGUID
	ON RolloutConfig (RolloutConfigGUID)
	CREATE UNIQUE INDEX RolloutConfigByPathName
	ON RolloutConfig (RolloutConfigPath, RolloutConfigName)
	CREATE INDEX RolloutConfigByPath
	ON RolloutConfig (RolloutConfigPath)
END
ELSE
	PRINT 'Using existing RolloutConfig table'
GO

-- Create RolloutConfigSettings table
IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutConfigSetting'
)
BEGIN
	-- Create table
	PRINT 'Creating RolloutConfigSetting table'
	CREATE TABLE RolloutConfigSetting
	(
		-- Auto-generated identity #
		RolloutConfigSettingID INTEGER NOT NULL IDENTITY
			CONSTRAINT PK_RolloutConfigSetting PRIMARY KEY NONCLUSTERED,

		-- Parent rollout config
		RolloutConfig INTEGER REFERENCES RolloutConfig(RolloutConfigID),
		
		-- Type of rollout config setting
		SettingType NVARCHAR(64) NULL DEFAULT(NULL),
			
		-- Name of rollout config setting
		SettingName NVARCHAR(64) NOT NULL,
		
		-- Value of rollout config setting
		SettingValue NVARCHAR(256) NULL DEFAULT(NULL)
	);
	
	-- Create indices
	CREATE INDEX RolloutConfigSettingByType
	ON RolloutConfigSetting (SettingType)
	CREATE UNIQUE INDEX RolloutConfigSettingByName
	ON RolloutConfigSetting (RolloutConfig, SettingType, SettingName)
END
ELSE
	PRINT 'Using existing RolloutConfigSetting table'
GO

-- The RolloutStatus table stores the status for each step in a rollout for a computer.
IF NOT EXISTS (
	SELECT * FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutStatusType'
)
BEGIN
	CREATE TABLE RolloutStatusType (
		RolloutStatusTypeID	int IDENTITY NOT NULL,
		[Name]			nvarchar(64) NOT NULL,
		CONSTRAINT UQ_RolloutStatusType
			UNIQUE NONCLUSTERED(RolloutStatusTypeID),
		CONSTRAINT PK_RolloutStatusType
			PRIMARY KEY NONCLUSTERED([Name])
	);
END
ELSE BEGIN
	PRINT 'Using existing RolloutStatusType table'
	IF NOT EXISTS (
		SELECT * FROM sysindexes
		WHERE name = 'UQ_RolloutStatusType'
	)
		ALTER TABLE RolloutStatusType
		ADD CONSTRAINT UQ_RolloutStatusType
			UNIQUE NONCLUSTERED(RolloutStatusTypeID)

	IF NOT EXISTS (
		SELECT * FROM sysindexes
		WHERE name = 'PK_RolloutStatusType'
	)
		ALTER TABLE RolloutStatusType
		ADD CONSTRAINT PK_RolloutStatusType
			PRIMARY KEY NONCLUSTERED([Name])
END
GO

-- The RolloutStatus table stores the status for each step in a rollout for a computer.
IF NOT EXISTS (
	SELECT * FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutStatus'
)
BEGIN
	CREATE TABLE RolloutStatus (
		-- Type of rollout status, such as data capture or computer build.
		RolloutStatusTypeID	int NOT NULL,
		-- The computer reporting the rollout status.
		ComputerID		int NOT NULL,
		-- The step number reporting the rollout status.
		StepNumber		int NOT NULL,
		-- The description of the rollout step.
		Description		nvarchar(256) NOT NULL,
		-- The status of this rollout step.
		Status			nvarchar(64) NOT NULL,
		-- The status message of this rollout step.
		StatusMessage		ntext NULL,
		-- The process phase that this particular step took place on.
		Phase			nvarchar(256) NULL,
		CONSTRAINT FK_RolloutStatus_RolloutStatusType
			FOREIGN KEY(RolloutStatusTypeID)
			REFERENCES RolloutStatusType(RolloutStatusTypeID) ON DELETE NO ACTION,
		CONSTRAINT FK_RolloutStatus_Computer
			FOREIGN KEY(ComputerID)
			REFERENCES Computer(ComputerID) ON DELETE CASCADE,
		CONSTRAINT PK_RolloutStatus
			PRIMARY KEY NONCLUSTERED(ComputerID, StepNumber, RolloutStatusTypeID)
	);
END
ELSE BEGIN
	PRINT 'Using existing RolloutStatus table'
	IF NOT EXISTS (
		SELECT * FROM sysindexes
		WHERE name = 'PK_RolloutStatus'
	)
		ALTER TABLE RolloutStatus
		ADD CONSTRAINT PK_RolloutStatus
			PRIMARY KEY NONCLUSTERED(ComputerID, StepNumber, RolloutStatusTypeID)
END
GO

-- The RolloutStatusSummary table stores the summarized status of the current step in a rollout for a computer.
IF NOT EXISTS (
	SELECT * FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutStatusSummary'
)
BEGIN
	CREATE TABLE RolloutStatusSummary (
		-- Type of rollout status, such as data capture or computer build.
		RolloutStatusTypeID	int NOT NULL DEFAULT 1,
		-- The computer reporting the rollout status summary.
		ComputerID		int NOT NULL,
		-- The current status of the rollout.
		LastStatus		nvarchar(64) NULL,
		-- The current rollout step of the computer.
		LastStep		int NULL,
		-- The current rollout step description.
		LastStepDescription	nvarchar(256) NULL,
		-- The number of rollout steps in the rollout.
		StepCount		int NULL,
		-- The date and time that the rollout was initiated.
		StartDateTime		datetime NULL,
		-- The date and time of the last rollout status.
		LastDateTime		datetime NULL,
		-- The date and time that the rollout was completed.
		CompleteDateTime	datetime NULL,
		CONSTRAINT FK_RolloutStatusStatus_RolloutStatusType
			FOREIGN KEY(RolloutStatusTypeID)
			REFERENCES RolloutStatusType(RolloutStatusTypeID) ON DELETE NO ACTION,
		CONSTRAINT FK_RolloutStatusSummary_Computer
			FOREIGN KEY(ComputerID)
			REFERENCES Computer(ComputerID) ON DELETE CASCADE,
		CONSTRAINT PK_RolloutStatusSummary
			PRIMARY KEY NONCLUSTERED(ComputerID, RolloutStatusTypeID)
	);

	CREATE NONCLUSTERED INDEX RolloutStatusSummary_LastStatus
		ON RolloutStatusSummary(LastStatus)
END
ELSE BEGIN
	PRINT 'Using existing RolloutStatusSummary table'
	IF NOT EXISTS (
		SELECT * FROM sysindexes
		WHERE name = 'PK_RolloutStatusSummary'
	)
		ALTER TABLE RolloutStatusSummary
		ADD CONSTRAINT PK_RolloutStatusSummary
			PRIMARY KEY NONCLUSTERED(ComputerID, RolloutStatusTypeID)

	IF NOT EXISTS (
		SELECT * FROM sysindexes
		WHERE name = 'RolloutStatusSummary_LastStatus'
	)
		CREATE NONCLUSTERED INDEX RolloutStatusSummary_LastStatus
			ON RolloutStatusSummary(LastStatus)
END
GO

IF NOT EXISTS (
	SELECT * FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'DataCapture'
)
BEGIN
	CREATE TABLE DataCapture (
		ComputerID		int NOT NULL,
		DataStoreLocation	nvarchar(1024) NULL,
		DataImageCaptureLocation nvarchar(1024) NULL,
		LogFileLocation		nvarchar(1024) NULL,
		CaptureDateTime		datetime NOT NULL DEFAULT getdate(),
		ImageCaptureDateTime	datetime NOT NULL DEFAULT getdate(),
		CONSTRAINT FK_DataCapture_ComputerID
			FOREIGN KEY(ComputerID)
			REFERENCES Computer(ComputerID) ON DELETE CASCADE,
		CONSTRAINT PK_DataCapture
			PRIMARY KEY NONCLUSTERED(ComputerID)
	);
END
ELSE BEGIN
	PRINT 'Using existing DataCapture table'
	IF NOT EXISTS (
		SELECT * FROM sysindexes
		WHERE name = 'PK_DataCapture'
	)
		ALTER TABLE DataCapture
		ADD CONSTRAINT PK_DataCapture
			PRIMARY KEY NONCLUSTERED(ComputerID)
END
GO

/*
 * $Id: rolloutconfig.sql,v 1.6 2007/07/24 06:30:56 pca Exp $
 *
 * Description:
 *
 * SQL Script to build the ManageSoft rollout config library stored procedures
 * on Microsoft SQL Server.
 *
 * The steps performed by this script are:
 * 1. Create new database
 * 2. Create database table and initialiase data (skip if the table exists)
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 *
 * Copyright 2011 Flexera Software LLC
 */
USE ManageSoft
GO

-- Procedure for selecting a rollout config folder
PRINT 'Creating procedure to select a rollout config folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigFolderSelect' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigFolderSelect];
GO
CREATE PROCEDURE [RolloutConfigFolderSelect]
(
	@RolloutConfigFolderGUID binary(16)
)
AS
	SET NOCOUNT ON;
	SELECT A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName,
		COUNT(B.RolloutConfigFolderPath) AS ChildRolloutConfigFolderCount, COUNT(C.RolloutConfigPath) AS ChildRolloutConfigCount
		FROM RolloutConfigFolder AS A
		LEFT OUTER JOIN RolloutConfigFolder AS B
		ON A.RolloutConfigFolderID = B.RolloutConfigFolderPath
		LEFT OUTER JOIN RolloutConfig AS C
		ON A.RolloutConfigFolderID = C.RolloutConfigPath
		WHERE (A.RolloutConfigFolderGUID = @RolloutConfigFolderGUID)
		GROUP BY A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName;
GO

-- Procedure for selecting a rollout config folder
PRINT 'Creating procedure to select a rollout config folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigFolderSelectID' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigFolderSelectID];
GO
CREATE PROCEDURE [RolloutConfigFolderSelectID]
(
	@RolloutConfigFolderID int
)
AS
	SET NOCOUNT ON;
	SELECT A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName,
		COUNT(B.RolloutConfigFolderPath) AS ChildRolloutConfigFolderCount, COUNT(C.RolloutConfigPath) AS ChildRolloutConfigCount
		FROM RolloutConfigFolder AS A
		LEFT OUTER JOIN RolloutConfigFolder AS B
		ON A.RolloutConfigFolderID = B.RolloutConfigFolderPath
		LEFT OUTER JOIN RolloutConfig AS C
		ON A.RolloutConfigFolderID = C.RolloutConfigPath
		WHERE (A.RolloutConfigFolderID = @RolloutConfigFolderID)
		GROUP BY A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName;
GO

-- Procedure for selecting child rollout config folders
PRINT 'Creating procedure to select child rollout config folders'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigFolderChildSelect' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigFolderChildSelect];
GO
CREATE PROCEDURE [RolloutConfigFolderChildSelect]
(
	@RolloutConfigFolderPath int = NULL
)
AS
	SET NOCOUNT ON;
	IF (@RolloutConfigFolderPath IS NULL)
		SELECT A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName,
			COUNT(B.RolloutConfigFolderPath) AS ChildRolloutConfigFolderCount, COUNT(C.RolloutConfigPath) AS ChildRolloutConfigCount
			FROM RolloutConfigFolder AS A
			LEFT OUTER JOIN RolloutConfigFolder AS B
			ON A.RolloutConfigFolderID = B.RolloutConfigFolderPath
			LEFT OUTER JOIN RolloutConfig AS C
			ON A.RolloutConfigFolderID = C.RolloutConfigPath
			WHERE (A.RolloutConfigFolderPath IS NULL)
			GROUP BY A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName
			ORDER BY A.RolloutConfigFolderName;
	ELSE
		SELECT A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName,
			COUNT(B.RolloutConfigFolderPath) AS ChildRolloutConfigFolderCount, COUNT(C.RolloutConfigPath) AS ChildRolloutConfigCount
			FROM RolloutConfigFolder AS A
			LEFT OUTER JOIN RolloutConfigFolder AS B
			ON A.RolloutConfigFolderID = B.RolloutConfigFolderPath
			LEFT OUTER JOIN RolloutConfig AS C
			ON A.RolloutConfigFolderID = C.RolloutConfigPath
			WHERE (A.RolloutConfigFolderPath = @RolloutConfigFolderPath)
			GROUP BY A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName
			ORDER BY A.RolloutConfigFolderName;
GO

-- Procedure for inserting a rollout config folder
PRINT 'Creating procedure to insert a rollout config folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigFolderInsert' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigFolderInsert];
GO
CREATE PROCEDURE [RolloutConfigFolderInsert]
(
	@RolloutConfigFolderGUID binary(16),
	@RolloutConfigFolderPath int,
	@RolloutConfigFolderName nvarchar(256)
)
AS
	SET NOCOUNT OFF;
	INSERT INTO RolloutConfigFolder (RolloutConfigFolderGUID, RolloutConfigFolderPath, RolloutConfigFolderName)
		VALUES (@RolloutConfigFolderGUID, @RolloutConfigFolderPath, @RolloutConfigFolderName);
	SELECT A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName,
		COUNT(B.RolloutConfigFolderPath) AS ChildRolloutConfigFolderCount, COUNT(C.RolloutConfigPath) AS ChildRolloutConfigCount
		FROM RolloutConfigFolder AS A
		LEFT OUTER JOIN RolloutConfigFolder AS B
		ON A.RolloutConfigFolderID = B.RolloutConfigFolderPath
		LEFT OUTER JOIN RolloutConfig AS C
		ON A.RolloutConfigFolderID = C.RolloutConfigPath
		WHERE (A.RolloutConfigFolderID = @@IDENTITY)
		GROUP BY A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName;
GO

-- Procedure for updating a rollout config folder
PRINT 'Creating procedure to update a rollout config folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigFolderUpdate' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigFolderUpdate];
GO
CREATE PROCEDURE [RolloutConfigFolderUpdate]
(
	@RolloutConfigFolderGUID binary(16),
	@RolloutConfigFolderPath int,
	@RolloutConfigFolderName nvarchar(256),
	@Original_RolloutConfigFolderID int,
	@Original_RolloutConfigFolderGUID binary(16),
	@Original_RolloutConfigFolderName nvarchar(256),
	@Original_RolloutConfigFolderPath int,
	@RolloutConfigFolderID int
)
AS
	SET NOCOUNT OFF;
	UPDATE RolloutConfigFolder
		SET RolloutConfigFolderGUID = @RolloutConfigFolderGUID, RolloutConfigFolderPath = @RolloutConfigFolderPath, RolloutConfigFolderName = @RolloutConfigFolderName
		WHERE (RolloutConfigFolderID = @Original_RolloutConfigFolderID)
		AND (RolloutConfigFolderGUID = @Original_RolloutConfigFolderGUID OR @Original_RolloutConfigFolderGUID IS NULL AND RolloutConfigFolderGUID IS NULL)
		AND (RolloutConfigFolderName = @Original_RolloutConfigFolderName OR @Original_RolloutConfigFolderName IS NULL AND RolloutConfigFolderName IS NULL)
		AND (RolloutConfigFolderPath = @Original_RolloutConfigFolderPath OR @Original_RolloutConfigFolderPath IS NULL AND RolloutConfigFolderPath IS NULL);
	SELECT A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName,
		COUNT(B.RolloutConfigFolderPath) AS ChildRolloutConfigFolderCount, COUNT(C.RolloutConfigPath) AS ChildRolloutConfigCount
		FROM RolloutConfigFolder AS A
		LEFT OUTER JOIN RolloutConfigFolder AS B
		ON A.RolloutConfigFolderID = B.RolloutConfigFolderPath
		LEFT OUTER JOIN RolloutConfig AS C
		ON A.RolloutConfigFolderID = C.RolloutConfigPath
		WHERE (A.RolloutConfigFolderID = @RolloutConfigFolderID)
		GROUP BY A.RolloutConfigFolderID, A.RolloutConfigFolderGUID, A.RolloutConfigFolderPath, A.RolloutConfigFolderName;
GO

-- Procedure for deleting a rollout config folder
PRINT 'Creating procedure to delete a rollout config folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigFolderDelete' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigFolderDelete];
GO

CREATE PROCEDURE [RolloutConfigFolderDelete]
(
	@Original_RolloutConfigFolderID int,
	@Original_RolloutConfigFolderGUID binary(16),
	@Original_RolloutConfigFolderName nvarchar(256),
	@Original_RolloutConfigFolderPath int
)
AS
	SET NOCOUNT OFF;
	DELETE FROM RolloutConfigFolder
		WHERE (RolloutConfigFolderID = @Original_RolloutConfigFolderID)
		AND (RolloutConfigFolderGUID = @Original_RolloutConfigFolderGUID OR @Original_RolloutConfigFolderGUID IS NULL AND RolloutConfigFolderGUID IS NULL)
		AND (RolloutConfigFolderName = @Original_RolloutConfigFolderName OR @Original_RolloutConfigFolderName IS NULL AND RolloutConfigFolderName IS NULL)
		AND (RolloutConfigFolderPath = @Original_RolloutConfigFolderPath OR @Original_RolloutConfigFolderPath IS NULL AND RolloutConfigFolderPath IS NULL);
GO

-- Procedure for selecting a rollout config
PRINT 'Creating procedure to select a rollout config'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSelect' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSelect];
GO

CREATE PROCEDURE [RolloutConfigSelect]
(
	@RolloutConfigGUID binary(16)
)
AS
	SET NOCOUNT ON;
	SELECT RolloutConfigID, RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath
		FROM RolloutConfig WHERE (RolloutConfigGUID = @RolloutConfigGUID);
GO

-- Procedure for selecting a rollout config
PRINT 'Creating procedure to select a rollout config'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSelectID' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSelectID];
GO

CREATE PROCEDURE [RolloutConfigSelectID]
(
	@RolloutConfigID int
)
AS
	SET NOCOUNT ON;
	SELECT RolloutConfigID, RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath
		FROM RolloutConfig WHERE (RolloutConfigID = @RolloutConfigID);
GO

-- Procedure for selecting child rollout configs
PRINT 'Creating procedure to select child rollout configs'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigChildSelect' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigChildSelect];
GO
CREATE PROCEDURE [RolloutConfigChildSelect]
(
	@RolloutConfigPath int = NULL
)
AS
	SET NOCOUNT ON;
	IF (@RolloutConfigPath IS NULL)
		SELECT RolloutConfigID, RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath
			FROM RolloutConfig
			WHERE (RolloutConfigPath IS NULL)
			ORDER BY RolloutConfigName;
	ELSE
		SELECT RolloutConfigID, RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath
			FROM RolloutConfig
			WHERE (RolloutConfigPath = @RolloutConfigPath)
			ORDER BY RolloutConfigName;
GO

-- Procedure for inserting a rollout config
PRINT 'Creating procedure to insert a rollout config'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigInsert' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigInsert];
GO
CREATE PROCEDURE [RolloutConfigInsert]
(
	@RolloutConfigGUID binary(16),
	@RolloutConfigPath int,
	@RolloutConfigName nvarchar(256),
	@RolloutConfigVersion nvarchar(64),
	@ModifyDateTime datetime,
	@Description nvarchar(256),
	@RepositoryPath nvarchar(256)
)
AS
	SET NOCOUNT OFF;
	INSERT INTO RolloutConfig (RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath)
		VALUES (@RolloutConfigGUID, @RolloutConfigPath, @RolloutConfigName, @RolloutConfigVersion, @ModifyDateTime, @Description, @RepositoryPath);
	SELECT RolloutConfigID, RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath
		FROM RolloutConfig
		WHERE (RolloutConfigID = @@IDENTITY);
GO

-- Procedure for updating a rollout config
PRINT 'Creating procedure to update a rollout config'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigUpdate' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigUpdate];
GO
CREATE PROCEDURE [RolloutConfigUpdate]
(
	@RolloutConfigGUID binary(16),
	@RolloutConfigPath int,
	@RolloutConfigName nvarchar(256),
	@RolloutConfigVersion nvarchar(64),
	@ModifyDateTime datetime,
	@Description nvarchar(256),
	@RepositoryPath nvarchar(256),
	@Original_RolloutConfigID int,
	@Original_Description nvarchar(256),
	@Original_RepositoryPath nvarchar(256),
	@Original_RolloutConfigGUID binary(16),
	@Original_RolloutConfigName nvarchar(256),
	@Original_RolloutConfigPath int,
	@Original_RolloutConfigVersion nvarchar(64),
	@Original_ModifyDateTime datetime,
	@RolloutConfigID int
)
AS
	SET NOCOUNT OFF;
	UPDATE RolloutConfig
		SET RolloutConfigGUID = @RolloutConfigGUID, RolloutConfigPath = @RolloutConfigPath, RolloutConfigName = @RolloutConfigName,
		RolloutConfigVersion = @RolloutConfigVersion, ModifyDateTime = @ModifyDateTime, Description = @Description, RepositoryPath = @RepositoryPath
		WHERE (RolloutConfigID = @Original_RolloutConfigID)
		AND (Description = @Original_Description OR @Original_Description IS NULL AND Description IS NULL)
		AND (RepositoryPath = @Original_RepositoryPath OR @Original_RepositoryPath IS NULL AND RepositoryPath IS NULL)
		AND (RolloutConfigGUID = @Original_RolloutConfigGUID OR @Original_RolloutConfigGUID IS NULL AND RolloutConfigGUID IS NULL)
		AND (RolloutConfigName = @Original_RolloutConfigName)
		AND (RolloutConfigPath = @Original_RolloutConfigPath OR @Original_RolloutConfigPath IS NULL AND RolloutConfigPath IS NULL)
		AND (RolloutConfigVersion = @Original_RolloutConfigVersion)
		AND (ModifyDateTime = @Original_ModifyDateTime);
	SELECT RolloutConfigID, RolloutConfigGUID, RolloutConfigPath, RolloutConfigName, RolloutConfigVersion, ModifyDateTime, Description, RepositoryPath
		FROM RolloutConfig
		WHERE (RolloutConfigID = @RolloutConfigID);
GO

-- Procedure for deleting a rollout config
PRINT 'Creating procedure to delete a rollout config'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigDelete' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigDelete];
GO
CREATE PROCEDURE [RolloutConfigDelete]
(
	@Original_RolloutConfigID int,
	@Original_Description nvarchar(256),
	@Original_RepositoryPath nvarchar(256),
	@Original_RolloutConfigGUID binary(16),
	@Original_RolloutConfigName nvarchar(256),
	@Original_RolloutConfigPath int,
	@Original_RolloutConfigVersion nvarchar(64),
	@Original_ModifyDateTime datetime
)
AS
	SET NOCOUNT OFF;
	BEGIN TRANSACTION;
	DELETE FROM RolloutConfigSetting
		WHERE (RolloutConfig = @Original_RolloutConfigID)
	DELETE FROM RolloutConfig
		WHERE (RolloutConfigID = @Original_RolloutConfigID)
		AND (Description = @Original_Description OR @Original_Description IS NULL AND Description IS NULL)
		AND (RepositoryPath = @Original_RepositoryPath OR @Original_RepositoryPath IS NULL AND RepositoryPath IS NULL)
		AND (RolloutConfigGUID = @Original_RolloutConfigGUID OR @Original_RolloutConfigGUID IS NULL AND RolloutConfigGUID IS NULL)
		AND (RolloutConfigName = @Original_RolloutConfigName)
		AND (RolloutConfigPath = @Original_RolloutConfigPath OR @Original_RolloutConfigPath IS NULL AND RolloutConfigPath IS NULL)
		AND (RolloutConfigVersion = @Original_RolloutConfigVersion)
		AND (ModifyDateTime = @Original_ModifyDateTime);
	COMMIT TRANSACTION;
GO

-- Procedure for selecting rollout config settings
PRINT 'Creating procedure to select rollout config settings'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSettingSelect' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSettingSelect];
GO
CREATE PROCEDURE [RolloutConfigSettingSelect]
(
	@RolloutConfig int
)
AS
	SET NOCOUNT ON;
	SELECT RolloutConfigSettingID, RolloutConfig, SettingType, SettingName, SettingValue
		FROM RolloutConfigSetting
		WHERE (RolloutConfig = @RolloutConfig);
GO

-- Procedure for inserting rollout config settings
PRINT 'Creating procedure to insert rollout config settings'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSettingInsert' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSettingInsert];
GO
CREATE PROCEDURE [RolloutConfigSettingInsert]
(
	@RolloutConfig int,
	@SettingType nvarchar(64),
	@SettingName nvarchar(64),
	@SettingValue nvarchar(256)
)
AS
	SET NOCOUNT OFF;
	INSERT INTO RolloutConfigSetting(RolloutConfig, SettingType, SettingName, SettingValue)
		VALUES (@RolloutConfig, @SettingType, @SettingName, @SettingValue);
	SELECT RolloutConfigSettingID, RolloutConfig, SettingType, SettingName, SettingValue
		FROM RolloutConfigSetting
		WHERE (RolloutConfigSettingID = @@IDENTITY);
GO

-- Procedure for updating rollout config settings
PRINT 'Creating procedure to update rollout config settings'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSettingUpdate' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSettingUpdate];
GO
CREATE PROCEDURE [RolloutConfigSettingUpdate]
(
	@RolloutConfig int,
	@SettingType nvarchar(64),
	@SettingName nvarchar(64),
	@SettingValue nvarchar(256),
	@Original_RolloutConfigSettingID int,
	@Original_RolloutConfig int,
	@Original_SettingName nvarchar(64),
	@Original_SettingType nvarchar(64),
	@Original_SettingValue nvarchar(256),
	@RolloutConfigSettingID int
)
AS
	SET NOCOUNT OFF;
	UPDATE RolloutConfigSetting
		SET RolloutConfig = @RolloutConfig, SettingType = @SettingType, SettingName = @SettingName, SettingValue = @SettingValue
		WHERE (RolloutConfigSettingID = @Original_RolloutConfigSettingID)
		AND (RolloutConfig = @Original_RolloutConfig OR @Original_RolloutConfig IS NULL AND RolloutConfig IS NULL)
		AND (SettingName = @Original_SettingName)
		AND (SettingType = @Original_SettingType OR @Original_SettingType IS NULL AND SettingType IS NULL)
		AND (SettingValue = @Original_SettingValue OR @Original_SettingValue IS NULL AND SettingValue IS NULL);
	SELECT RolloutConfigSettingID, RolloutConfig, SettingType, SettingName, SettingValue
		FROM RolloutConfigSetting
		WHERE (RolloutConfigSettingID = @RolloutConfigSettingID);
GO

-- Procedure for delete rollout config settings
PRINT 'Creating procedure to delete rollout config settings'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSettingDelete' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSettingDelete];
GO
CREATE PROCEDURE [RolloutConfigSettingDelete]
(
	@Original_RolloutConfigSettingID int,
	@Original_RolloutConfig int,
	@Original_SettingName nvarchar(64),
	@Original_SettingType nvarchar(64),
	@Original_SettingValue nvarchar(256)
)
AS
	SET NOCOUNT OFF;
	DELETE FROM RolloutConfigSetting
		WHERE (RolloutConfigSettingID = @Original_RolloutConfigSettingID)
		AND (RolloutConfig = @Original_RolloutConfig OR @Original_RolloutConfig IS NULL AND RolloutConfig IS NULL)
		AND (SettingName = @Original_SettingName)
		AND (SettingType = @Original_SettingType OR @Original_SettingType IS NULL AND SettingType IS NULL)
		AND (SettingValue = @Original_SettingValue OR @Original_SettingValue IS NULL AND SettingValue IS NULL);
GO

-- Procedure for selecting bootable wim reference in rollout config settings
PRINT 'Creating procedure to select bootable wim reference in rollout config settings'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'RolloutConfigSettingSelectBySettingTypeSettingNameSettingValue' AND xtype = 'P')
	DROP PROCEDURE [RolloutConfigSettingSelectBySettingTypeSettingNameSettingValue]
GO
CREATE PROCEDURE [RolloutConfigSettingSelectBySettingTypeSettingNameSettingValue]
(
	@SettingType nvarchar(64),
	@SettingName nvarchar(64),
	@SettingValue nvarchar(256)
)
AS
	SET NOCOUNT OFF;
	SELECT RolloutConfigSettingID, RolloutConfig, SettingType, SettingName, SettingValue
		FROM RolloutConfigSetting
		WHERE SettingType = @SettingType
			AND	SettingName = @SettingName
			AND SettingValue LIKE '%\{' + @SettingValue + '}\%.wim'

	return @@ROWCOUNT

GO

-- Rollout status section
IF NOT EXISTS(SELECT * FROM RolloutStatusType)
BEGIN
	SET IDENTITY_INSERT RolloutStatusType ON
	INSERT INTO RolloutStatusType (Name, RolloutStatusTypeID)
		VALUES ('ComputerBuild', 1)
	INSERT INTO RolloutStatusType (Name, RolloutStatusTypeID)
		VALUES ('ComputerDataCapture', 2)
	INSERT INTO RolloutStatusType (Name, RolloutStatusTypeID)
		VALUES ('ComputerImageCapture', 3)
	SET IDENTITY_INSERT RolloutStatusType OFF
END
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusTypeGetByID')
	DROP PROCEDURE RolloutStatusTypeGetByID
GO
CREATE PROCEDURE RolloutStatusTypeGetByID
	@RolloutStatusTypeID	int
AS
	SELECT	RolloutStatusTypeID,
		[Name]
	FROM	RolloutStatusType
	WHERE	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusTypeGetByName')
	DROP PROCEDURE RolloutStatusTypeGetByName
GO
CREATE PROCEDURE RolloutStatusTypeGetByName
	@Name			nvarchar(64)
AS
	SELECT	RolloutStatusTypeID,
		[Name]
	FROM	RolloutStatusType
	WHERE	[Name] = @Name
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusTypeGetAll')
	DROP PROCEDURE RolloutStatusTypeGetAll
GO
CREATE PROCEDURE RolloutStatusTypeGetAll
AS
	SELECT	RolloutStatusTypeID,
		[Name]
	FROM	RolloutStatusType
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusTypeBind')
	DROP PROCEDURE RolloutStatusTypeBind
GO
CREATE PROCEDURE RolloutStatusTypeBind
	@Name			nvarchar(64)
AS
	DECLARE	@ID	int

	SELECT	@ID = RolloutStatusTypeID
	FROM	RolloutStatusType
	WHERE	[Name] = @Name

	IF @@ROWCOUNT = 0
	BEGIN
		SET @ID = 0
	END

	RETURN @ID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusGetByComputerIDStepNumberTypeID')
	DROP PROCEDURE RolloutStatusGetByComputerIDStepNumberTypeID
GO
CREATE PROCEDURE RolloutStatusGetByComputerIDStepNumberTypeID
	@ComputerID		int,
	@StepNumber		int,
	@RolloutStatusTypeID	int
AS
	SELECT	RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	FROM	RolloutStatus
	WHERE	ComputerID = @ComputerID
	  AND	StepNumber = @StepNumber
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusGetAllByComputerIDStepNumber')
	DROP PROCEDURE RolloutStatusGetAllByComputerIDStepNumber
GO
CREATE PROCEDURE RolloutStatusGetAllByComputerIDStepNumber
	@ComputerID		int,
	@StepNumber		int
AS
	SELECT	RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	FROM	RolloutStatus
	WHERE	ComputerID = @ComputerID
	  AND	StepNumber = @StepNumber
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusGetAllByComputerID')
	DROP PROCEDURE RolloutStatusGetAllByComputerID
GO
CREATE PROCEDURE RolloutStatusGetAllByComputerID
	@ComputerID		int
AS
	SELECT	RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	FROM	RolloutStatus
	WHERE	ComputerID = @ComputerID
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusGetAll')
	DROP PROCEDURE RolloutStatusGetAll
GO
CREATE PROCEDURE RolloutStatusGetAll
AS
	SELECT	RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	FROM	RolloutStatus
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusAdd')
	DROP PROCEDURE RolloutStatusAdd
GO
CREATE PROCEDURE RolloutStatusAdd
	@RolloutStatusTypeID	int,
	@ComputerID		int,
	@StepNumber		int,
	@Description		nvarchar(256),
	@Status			nvarchar(64),
	@StatusMessage		ntext = NULL,
	@Phase			nvarchar(256) = NULL
AS
	INSERT INTO RolloutStatus (
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
	VALUES (
		@RolloutStatusTypeID,
		@ComputerID,
		@StepNumber,
		@Description,
		@Status,
		@StatusMessage,
		@Phase
	)
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSchema')
	DROP PROCEDURE RolloutStatusSchema
GO
CREATE PROCEDURE RolloutStatusSchema
AS
	SELECT
	'IF OBJECT_ID(''tempdb..#RolloutStatus'') IS NULL' +
	'	CREATE TABLE #RolloutStatus (' +
	'		RolloutStatusTypeID	int NOT NULL,' +
	'		ComputerID		int NOT NULL,' +
	'		StepNumber		int NOT NULL,' +
	'		Description		nvarchar(256) '+dbo.ColumnCollate('RolloutStatus', 'Description')+' NOT NULL,' +
	'		Status			nvarchar(64) '+dbo.ColumnCollate('RolloutStatus', 'Status')+' NOT NULL,' +
	'		StatusMessage		ntext '+dbo.ColumnCollate('RolloutStatus', 'StatusMessage')+' NULL,' +
	'		Phase			nvarchar(256) '+dbo.ColumnCollate('RolloutStatus', 'Phase')+' NULL' +
	'	)' +
	'ELSE' +
	'	DELETE FROM #RolloutStatus' +
	'' AS SQL
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusTempAdd')
	DROP PROCEDURE RolloutStatusTempAdd
GO
CREATE PROCEDURE RolloutStatusTempAdd
	@RolloutStatusTypeID	int,
	@ComputerID		int,
	@StepNumber		int,
	@Description		nvarchar(256),
	@Status			nvarchar(64),
	@StatusMessage		ntext = NULL,
	@Phase			nvarchar(256) = NULL
AS
	INSERT INTO #RolloutStatus (
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
	VALUES (
		@RolloutStatusTypeID,
		@ComputerID,
		@StepNumber,
		@Description,
		@Status,
		@StatusMessage,
		@Phase
	)
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusAddBatch')
	DROP PROCEDURE RolloutStatusAddBatch
GO
CREATE PROCEDURE RolloutStatusAddBatch
AS
	DECLARE @count int
	INSERT RolloutStatus (
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
		SELECT	RolloutStatusTypeID,
			ComputerID,
			StepNumber,
			Description,
			Status,
			StatusMessage,
			Phase
		FROM #RolloutStatus
	SET @count = @@ROWCOUNT
	DROP TABLE #RolloutStatus	-- Drop temporary table
	RETURN @count
GO

/*
Add new RolloutStatus bound to
	RolloutStatusType using Name
	Computer using ComputerOUID, ComputerCN
	with StepNumber, Description, Status, StatusMessage, Phase
*/

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusUpdateByComputerIDStepNumberTypeID')
	DROP PROCEDURE RolloutStatusUpdateByComputerIDStepNumberTypeID
GO
CREATE PROCEDURE RolloutStatusUpdateByComputerIDStepNumberTypeID
	@ComputerID		int,
	@StepNumber		int,
	@RolloutStatusTypeID	int,
	@Description		nvarchar(256),
	@Status			nvarchar(64),
	@StatusMessage		ntext = NULL,
	@Phase			nvarchar(256) = NULL
AS
	UPDATE	RolloutStatus
	SET	Description = @Description,
		Status = @Status,
		StatusMessage = @StatusMessage,
		Phase = @Phase
	WHERE	ComputerID = @ComputerID
	  AND	StepNumber = @StepNumber
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusUpdateBatchByComputerIDStepNumberTypeID')
	DROP PROCEDURE RolloutStatusUpdateBatchByComputerIDStepNumberTypeID
GO
CREATE PROCEDURE RolloutStatusUpdateBatchByComputerIDStepNumberTypeID
AS
	DECLARE @count int

	CREATE INDEX [TempRolloutStatusComputerIDStepNumberTypeID]
		ON #RolloutStatus (ComputerID, StepNumber, RolloutStatusTypeID)

	UPDATE	RolloutStatus
	SET	Description = t.Description,
		Status = t.Status,
		StatusMessage = t.StatusMessage,
		Phase = t.Phase
	FROM	#RolloutStatus AS t
	WHERE	RolloutStatus.ComputerID = t.ComputerID
	  AND	RolloutStatus.StepNumber = t.StepNumber
	  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT
	DROP TABLE #RolloutStatus	-- Drop the temporary table
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusPutByComputerIDStepNumberTypeID')
	DROP PROCEDURE RolloutStatusPutByComputerIDStepNumberTypeID
GO
CREATE PROCEDURE RolloutStatusPutByComputerIDStepNumberTypeID
	@ComputerID		int,
	@StepNumber		int,
	@RolloutStatusTypeID	int,
	@Description		nvarchar(256),
	@Status			nvarchar(64),
	@StatusMessage		ntext = NULL,
	@Phase			nvarchar(256) = NULL
AS
	IF NOT EXISTS(
		SELECT	1
		FROM	RolloutStatus
		WHERE	ComputerID = @ComputerID
		  AND	StepNumber = @StepNumber
		  AND	RolloutStatusTypeID = @RolloutStatusTypeID
	)
		EXEC RolloutStatusAdd @RolloutStatusTypeID,
			@ComputerID,
			@StepNumber,
			@Description,
			@Status,
			@StatusMessage,
			@Phase
	ELSE
		EXEC RolloutStatusUpdateByComputerIDStepNumberTypeID @ComputerID,
			@StepNumber,
			@RolloutStatusTypeID,
			@Description,
			@Status,
			@StatusMessage,
			@Phase
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusPutBatchByComputerIDStepNumberTypeID')
	DROP PROCEDURE RolloutStatusPutBatchByComputerIDStepNumberTypeID
GO
CREATE PROCEDURE RolloutStatusPutBatchByComputerIDStepNumberTypeID
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusComputerIDStepNumberTypeID]
		ON #RolloutStatus (ComputerID, StepNumber, RolloutStatusTypeID)

	BEGIN TRANSACTION
	UPDATE RolloutStatus
		SET	Description = t.Description,
			Status = t.Status,
			StatusMessage = t.StatusMessage,
			Phase = t.Phase
		FROM	#RolloutStatus AS t
		WHERE	RolloutStatus.ComputerID = t.ComputerID
		  AND	RolloutStatus.StepNumber = t.StepNumber
		  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatus (
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
		SELECT	RolloutStatusTypeID,
			ComputerID,
			StepNumber,
			Description,
			Status,
			StatusMessage,
			Phase
		FROM	#RolloutStatus AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatus
			WHERE	RolloutStatus.ComputerID = t.ComputerID
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatus	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusRemoveByComputerIDStepNumberTypeID')
	DROP PROCEDURE RolloutStatusRemoveByComputerIDStepNumberTypeID
GO
CREATE PROCEDURE RolloutStatusRemoveByComputerIDStepNumberTypeID
	@ComputerID		int,
	@StepNumber		int,
	@RolloutStatusTypeID	int
AS
	DELETE	RolloutStatus
	WHERE	ComputerID = @ComputerID
	  AND	StepNumber = @StepNumber
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusRemoveAllByComputerIDStepNumber')
	DROP PROCEDURE RolloutStatusRemoveAllByComputerIDStepNumber
GO
CREATE PROCEDURE RolloutStatusRemoveAllByComputerIDStepNumber
	@ComputerID		int,
	@StepNumber		int
AS
	DELETE	RolloutStatus
	WHERE	ComputerID = @ComputerID
	  AND	StepNumber = @StepNumber
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusRemoveAllByComputerID')
	DROP PROCEDURE RolloutStatusRemoveAllByComputerID
GO
CREATE PROCEDURE RolloutStatusRemoveAllByComputerID
	@ComputerID		int
AS
	DELETE	RolloutStatus
	WHERE	ComputerID = @ComputerID
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusRemoveAll')
	DROP PROCEDURE RolloutStatusRemoveAll
GO
CREATE PROCEDURE RolloutStatusRemoveAll
	
AS
	DELETE	RolloutStatus
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusReplaceSubsetByComputerIDStepNumber')
	DROP PROCEDURE RolloutStatusReplaceSubsetByComputerIDStepNumber
GO
-- Replace the record subset selected by (@ComputerID, @StepNumber)
-- using unique identifier (ComputerID, StepNumber, RolloutStatusTypeID)
-- updating fields (Description, Status, StatusMessage, Phase)
CREATE PROCEDURE RolloutStatusReplaceSubsetByComputerIDStepNumber
	@ComputerID		int,
	@StepNumber		int
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusByComputerIDStepNumber]
		ON #RolloutStatus (ComputerID, StepNumber, RolloutStatusTypeID)

	BEGIN TRANSACTION
	DELETE RolloutStatus
		WHERE	ComputerID = @ComputerID
		  AND	StepNumber = @StepNumber
		AND NOT EXISTS(		-- Replacement record doesn't exist
			SELECT * FROM #RolloutStatus AS t
			WHERE	RolloutStatus.ComputerID = t.ComputerID
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)

	UPDATE RolloutStatus
		SET	Description = t.Description,
			Status = t.Status,
			StatusMessage = t.StatusMessage,
			Phase = t.Phase
		FROM	#RolloutStatus AS t
		WHERE	RolloutStatus.ComputerID = t.ComputerID
		  AND	RolloutStatus.StepNumber = t.StepNumber
		  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatus (
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
		SELECT	RolloutStatusTypeID,
			ComputerID,
			StepNumber,
			Description,
			Status,
			StatusMessage,
			Phase
		FROM	#RolloutStatus AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatus
			WHERE	RolloutStatus.ComputerID = t.ComputerID
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatus	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusRemoveBatch')
	DROP PROCEDURE RolloutStatusRemoveBatch
GO

CREATE PROCEDURE RolloutStatusRemoveBatch
AS

	CREATE INDEX [TempRolloutStatusByComputerIDStepNumber]
		ON #RolloutStatus (ComputerID, StepNumber, RolloutStatusTypeID)	

	-- Delete records from RolloutStatus identified by fields of [TempRolloutStatusByComputerIDStepNumber]
	DELETE RolloutStatus
	WHERE EXISTS(
			SELECT * FROM #RolloutStatus AS t
			WHERE	RolloutStatus.ComputerID = t.ComputerID
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	DROP TABLE #RolloutStatus      -- Drop the temporary table
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusReplaceSubsetByComputerID')
	DROP PROCEDURE RolloutStatusReplaceSubsetByComputerID
GO
-- Replace the record subset selected by (@ComputerID)
-- using unique identifier (ComputerID, StepNumber, RolloutStatusTypeID)
-- updating fields (Description, Status, StatusMessage, Phase)
CREATE PROCEDURE RolloutStatusReplaceSubsetByComputerID
	@ComputerID		int
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusByComputerID]
		ON #RolloutStatus (ComputerID, StepNumber, RolloutStatusTypeID)

	BEGIN TRANSACTION
	DELETE RolloutStatus
		WHERE	ComputerID = @ComputerID
		AND NOT EXISTS(		-- Replacement record doesn't exist
			SELECT * FROM #RolloutStatus AS t
			WHERE	RolloutStatus.ComputerID = t.ComputerID
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)

	UPDATE RolloutStatus
		SET	Description = t.Description,
			Status = t.Status,
			StatusMessage = t.StatusMessage,
			Phase = t.Phase
		FROM	#RolloutStatus AS t
		WHERE	RolloutStatus.ComputerID = t.ComputerID
		  AND	RolloutStatus.StepNumber = t.StepNumber
		  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatus (
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
		SELECT	RolloutStatusTypeID,
			ComputerID,
			StepNumber,
			Description,
			Status,
			StatusMessage,
			Phase
		FROM	#RolloutStatus AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatus
			WHERE	RolloutStatus.ComputerID = t.ComputerID
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatus	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryGetByComputerIDRolloutStatusTypeID')
	DROP PROCEDURE RolloutStatusSummaryGetByComputerIDRolloutStatusTypeID
GO
CREATE PROCEDURE RolloutStatusSummaryGetByComputerIDRolloutStatusTypeID
	@ComputerID		int,
	@RolloutStatusTypeID	int = 1
AS
	SELECT	RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	FROM	RolloutStatusSummary
	WHERE	ComputerID = @ComputerID
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryGetAllByComputerID')
	DROP PROCEDURE RolloutStatusSummaryGetAllByComputerID
GO
CREATE PROCEDURE RolloutStatusSummaryGetAllByComputerID
	@ComputerID		int
AS
	SELECT	RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	FROM	RolloutStatusSummary
	WHERE	ComputerID = @ComputerID
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryGetAllByLastStatus')
	DROP PROCEDURE RolloutStatusSummaryGetAllByLastStatus
GO
CREATE PROCEDURE RolloutStatusSummaryGetAllByLastStatus
	@LastStatus		nvarchar(64) = NULL
AS
	SELECT	RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	FROM	RolloutStatusSummary
	WHERE	(LastStatus = @LastStatus OR (LastStatus IS NULL AND @LastStatus IS NULL))
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryGetAll')
	DROP PROCEDURE RolloutStatusSummaryGetAll
GO
CREATE PROCEDURE RolloutStatusSummaryGetAll
AS
	SELECT	RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	FROM	RolloutStatusSummary
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryAdd')
	DROP PROCEDURE RolloutStatusSummaryAdd
GO
CREATE PROCEDURE RolloutStatusSummaryAdd
	@RolloutStatusTypeID	int = 1,
	@ComputerID		int,
	@LastStatus		nvarchar(64) = NULL,
	@LastStep		int = NULL,
	@LastStepDescription	nvarchar(256) = NULL,
	@StepCount		int = NULL,
	@StartDateTime		datetime = NULL,
	@LastDateTime		datetime = NULL,
	@CompleteDateTime	datetime = NULL
AS
	INSERT INTO RolloutStatusSummary (
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	)
	VALUES (
		@RolloutStatusTypeID,
		@ComputerID,
		@LastStatus,
		@LastStep,
		@LastStepDescription,
		@StepCount,
		@StartDateTime,
		@LastDateTime,
		@CompleteDateTime
	)
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummarySchema')
	DROP PROCEDURE RolloutStatusSummarySchema
GO
CREATE PROCEDURE RolloutStatusSummarySchema
AS
	SELECT
	'IF OBJECT_ID(''tempdb..#RolloutStatusSummary'') IS NULL' +
	'	CREATE TABLE #RolloutStatusSummary (' +
	'		RolloutStatusTypeID	int NOT NULL DEFAULT 1,' +
	'		ComputerID		int NOT NULL,' +
	'		LastStatus		nvarchar(64) '+dbo.ColumnCollate('RolloutStatusSummary', 'LastStatus')+' NULL,' +
	'		LastStep		int NULL,' +
	'		LastStepDescription	nvarchar(256) '+dbo.ColumnCollate('RolloutStatusSummary', 'LastStepDescription')+' NULL,' +
	'		StepCount		int NULL,' +
	'		StartDateTime		datetime NULL,' +
	'		LastDateTime		datetime NULL,' +
	'		CompleteDateTime	datetime NULL' +
	'	)' +
	'ELSE' +
	'	DELETE FROM #RolloutStatusSummary' +
	'' AS SQL
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryTempAdd')
	DROP PROCEDURE RolloutStatusSummaryTempAdd
GO
CREATE PROCEDURE RolloutStatusSummaryTempAdd
	@RolloutStatusTypeID	int = 1,
	@ComputerID		int,
	@LastStatus		nvarchar(64) = NULL,
	@LastStep		int = NULL,
	@LastStepDescription	nvarchar(256) = NULL,
	@StepCount		int = NULL,
	@StartDateTime		datetime = NULL,
	@LastDateTime		datetime = NULL,
	@CompleteDateTime	datetime = NULL
AS
	INSERT INTO #RolloutStatusSummary (
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	)
	VALUES (
		@RolloutStatusTypeID,
		@ComputerID,
		@LastStatus,
		@LastStep,
		@LastStepDescription,
		@StepCount,
		@StartDateTime,
		@LastDateTime,
		@CompleteDateTime
	)
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryAddBatch')
	DROP PROCEDURE RolloutStatusSummaryAddBatch
GO
CREATE PROCEDURE RolloutStatusSummaryAddBatch
AS
	DECLARE @count int
	INSERT RolloutStatusSummary (
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	)
		SELECT	RolloutStatusTypeID,
			ComputerID,
			LastStatus,
			LastStep,
			LastStepDescription,
			StepCount,
			StartDateTime,
			LastDateTime,
			CompleteDateTime
		FROM #RolloutStatusSummary
	SET @count = @@ROWCOUNT
	DROP TABLE #RolloutStatusSummary	-- Drop temporary table
	RETURN @count
GO

/*
Add new RolloutStatusSummary bound to
	RolloutStatusType using Name
	Computer using ComputerOUID, ComputerCN
	with LastStatus, LastStep, LastStepDescription, StepCount, StartDateTime, LastDateTime, CompleteDateTime
*/

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryUpdateByComputerIDRolloutStatusTypeID')
	DROP PROCEDURE RolloutStatusSummaryUpdateByComputerIDRolloutStatusTypeID
GO
CREATE PROCEDURE RolloutStatusSummaryUpdateByComputerIDRolloutStatusTypeID
	@ComputerID		int,
	@RolloutStatusTypeID	int = 1,
	@LastStatus		nvarchar(64) = NULL,
	@LastStep		int = NULL,
	@LastStepDescription	nvarchar(256) = NULL,
	@StepCount		int = NULL,
	@StartDateTime		datetime = NULL,
	@LastDateTime		datetime = NULL,
	@CompleteDateTime	datetime = NULL
AS
	UPDATE	RolloutStatusSummary
	SET	LastStatus = @LastStatus,
		LastStep = @LastStep,
		LastStepDescription = @LastStepDescription,
		StepCount = @StepCount,
		StartDateTime = @StartDateTime,
		LastDateTime = @LastDateTime,
		CompleteDateTime = @CompleteDateTime
	WHERE	ComputerID = @ComputerID
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryUpdateBatchByComputerIDRolloutStatusTypeID')
	DROP PROCEDURE RolloutStatusSummaryUpdateBatchByComputerIDRolloutStatusTypeID
GO
CREATE PROCEDURE RolloutStatusSummaryUpdateBatchByComputerIDRolloutStatusTypeID
AS
	DECLARE @count int

	CREATE INDEX [TempRolloutStatusSummaryComputerIDRolloutStatusTypeID]
		ON #RolloutStatusSummary (ComputerID, RolloutStatusTypeID)

	UPDATE	RolloutStatusSummary
	SET	LastStatus = t.LastStatus,
		LastStep = t.LastStep,
		LastStepDescription = t.LastStepDescription,
		StepCount = t.StepCount,
		StartDateTime = t.StartDateTime,
		LastDateTime = t.LastDateTime,
		CompleteDateTime = t.CompleteDateTime
	FROM	#RolloutStatusSummary AS t
	WHERE	RolloutStatusSummary.ComputerID = t.ComputerID
	  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT
	DROP TABLE #RolloutStatusSummary	-- Drop the temporary table
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryPutByComputerIDRolloutStatusTypeID')
	DROP PROCEDURE RolloutStatusSummaryPutByComputerIDRolloutStatusTypeID
GO
CREATE PROCEDURE RolloutStatusSummaryPutByComputerIDRolloutStatusTypeID
	@ComputerID		int,
	@RolloutStatusTypeID	int = 1,
	@LastStatus		nvarchar(64) = NULL,
	@LastStep		int = NULL,
	@LastStepDescription	nvarchar(256) = NULL,
	@StepCount		int = NULL,
	@StartDateTime		datetime = NULL,
	@LastDateTime		datetime = NULL,
	@CompleteDateTime	datetime = NULL
AS
	IF NOT EXISTS(
		SELECT	1
		FROM	RolloutStatusSummary
		WHERE	ComputerID = @ComputerID
		  AND	RolloutStatusTypeID = @RolloutStatusTypeID
	)
		EXEC RolloutStatusSummaryAdd @RolloutStatusTypeID,
			@ComputerID,
			@LastStatus,
			@LastStep,
			@LastStepDescription,
			@StepCount,
			@StartDateTime,
			@LastDateTime,
			@CompleteDateTime
	ELSE
		EXEC RolloutStatusSummaryUpdateByComputerIDRolloutStatusTypeID @ComputerID,
			@RolloutStatusTypeID,
			@LastStatus,
			@LastStep,
			@LastStepDescription,
			@StepCount,
			@StartDateTime,
			@LastDateTime,
			@CompleteDateTime
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryPutBatchByComputerIDRolloutStatusTypeID')
	DROP PROCEDURE RolloutStatusSummaryPutBatchByComputerIDRolloutStatusTypeID
GO
CREATE PROCEDURE RolloutStatusSummaryPutBatchByComputerIDRolloutStatusTypeID
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusSummaryComputerIDRolloutStatusTypeID]
		ON #RolloutStatusSummary (ComputerID, RolloutStatusTypeID)

	BEGIN TRANSACTION
	UPDATE RolloutStatusSummary
		SET	LastStatus = t.LastStatus,
			LastStep = t.LastStep,
			LastStepDescription = t.LastStepDescription,
			StepCount = t.StepCount,
			StartDateTime = t.StartDateTime,
			LastDateTime = t.LastDateTime,
			CompleteDateTime = t.CompleteDateTime
		FROM	#RolloutStatusSummary AS t
		WHERE	RolloutStatusSummary.ComputerID = t.ComputerID
		  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatusSummary (
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	)
		SELECT	RolloutStatusTypeID,
			ComputerID,
			LastStatus,
			LastStep,
			LastStepDescription,
			StepCount,
			StartDateTime,
			LastDateTime,
			CompleteDateTime
		FROM	#RolloutStatusSummary AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatusSummary
			WHERE	RolloutStatusSummary.ComputerID = t.ComputerID
			  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatusSummary	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryRemoveByComputerIDRolloutStatusTypeID')
	DROP PROCEDURE RolloutStatusSummaryRemoveByComputerIDRolloutStatusTypeID
GO
CREATE PROCEDURE RolloutStatusSummaryRemoveByComputerIDRolloutStatusTypeID
	@ComputerID		int,
	@RolloutStatusTypeID	int = 1
AS
	DELETE	RolloutStatusSummary
	WHERE	ComputerID = @ComputerID
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryRemoveAllByComputerID')
	DROP PROCEDURE RolloutStatusSummaryRemoveAllByComputerID
GO
CREATE PROCEDURE RolloutStatusSummaryRemoveAllByComputerID
	@ComputerID		int
AS
	DELETE	RolloutStatusSummary
	WHERE	ComputerID = @ComputerID
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryRemoveAllByLastStatus')
	DROP PROCEDURE RolloutStatusSummaryRemoveAllByLastStatus
GO
CREATE PROCEDURE RolloutStatusSummaryRemoveAllByLastStatus
	@LastStatus		nvarchar(64) = NULL
AS
	DELETE	RolloutStatusSummary
	WHERE	(LastStatus = @LastStatus OR (LastStatus IS NULL AND @LastStatus IS NULL))
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryRemoveAll')
	DROP PROCEDURE RolloutStatusSummaryRemoveAll
GO
CREATE PROCEDURE RolloutStatusSummaryRemoveAll
	
AS
	DELETE	RolloutStatusSummary
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryReplaceSubsetByComputerID')
	DROP PROCEDURE RolloutStatusSummaryReplaceSubsetByComputerID
GO
-- Replace the record subset selected by (@ComputerID)
-- using unique identifier (ComputerID, RolloutStatusTypeID)
-- updating fields (LastStatus, LastStep, LastStepDescription, StepCount, StartDateTime, LastDateTime, CompleteDateTime)
CREATE PROCEDURE RolloutStatusSummaryReplaceSubsetByComputerID
	@ComputerID		int
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusSummaryByComputerID]
		ON #RolloutStatusSummary (ComputerID, RolloutStatusTypeID)

	BEGIN TRANSACTION
	DELETE RolloutStatusSummary
		WHERE	ComputerID = @ComputerID
		AND NOT EXISTS(		-- Replacement record doesn't exist
			SELECT * FROM #RolloutStatusSummary AS t
			WHERE	RolloutStatusSummary.ComputerID = t.ComputerID
			  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
		)

	UPDATE RolloutStatusSummary
		SET	LastStatus = t.LastStatus,
			LastStep = t.LastStep,
			LastStepDescription = t.LastStepDescription,
			StepCount = t.StepCount,
			StartDateTime = t.StartDateTime,
			LastDateTime = t.LastDateTime,
			CompleteDateTime = t.CompleteDateTime
		FROM	#RolloutStatusSummary AS t
		WHERE	RolloutStatusSummary.ComputerID = t.ComputerID
		  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatusSummary (
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	)
		SELECT	RolloutStatusTypeID,
			ComputerID,
			LastStatus,
			LastStep,
			LastStepDescription,
			StepCount,
			StartDateTime,
			LastDateTime,
			CompleteDateTime
		FROM	#RolloutStatusSummary AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatusSummary
			WHERE	RolloutStatusSummary.ComputerID = t.ComputerID
			  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatusSummary	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryRemoveBatch')
	DROP PROCEDURE RolloutStatusSummaryRemoveBatch
GO

CREATE PROCEDURE RolloutStatusSummaryRemoveBatch
AS

	CREATE INDEX [TempRolloutStatusSummaryByComputerID]
		ON #RolloutStatusSummary (ComputerID, RolloutStatusTypeID)	

	-- Delete records from RolloutStatusSummary identified by fields of [TempRolloutStatusSummaryByComputerID]
	DELETE RolloutStatusSummary
	WHERE EXISTS(
			SELECT * FROM #RolloutStatusSummary AS t
			WHERE	RolloutStatusSummary.ComputerID = t.ComputerID
			  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	DROP TABLE #RolloutStatusSummary      -- Drop the temporary table
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusSummaryReplaceSubsetByLastStatus')
	DROP PROCEDURE RolloutStatusSummaryReplaceSubsetByLastStatus
GO
-- Replace the record subset selected by (@LastStatus)
-- using unique identifier (ComputerID, RolloutStatusTypeID)
-- updating fields (LastStep, LastStepDescription, StepCount, StartDateTime, LastDateTime, CompleteDateTime)
CREATE PROCEDURE RolloutStatusSummaryReplaceSubsetByLastStatus
	@LastStatus		nvarchar(64) = NULL
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusSummaryByLastStatus]
		ON #RolloutStatusSummary (ComputerID, RolloutStatusTypeID)

	BEGIN TRANSACTION
	DELETE RolloutStatusSummary
		WHERE	(LastStatus = @LastStatus OR (LastStatus IS NULL AND @LastStatus IS NULL))
		AND NOT EXISTS(		-- Replacement record doesn't exist
			SELECT * FROM #RolloutStatusSummary AS t
			WHERE	RolloutStatusSummary.ComputerID = t.ComputerID
			  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
		)

	UPDATE RolloutStatusSummary
		SET	LastStep = t.LastStep,
			LastStepDescription = t.LastStepDescription,
			StepCount = t.StepCount,
			StartDateTime = t.StartDateTime,
			LastDateTime = t.LastDateTime,
			CompleteDateTime = t.CompleteDateTime
		FROM	#RolloutStatusSummary AS t
		WHERE	RolloutStatusSummary.ComputerID = t.ComputerID
		  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatusSummary (
		RolloutStatusTypeID,
		ComputerID,
		LastStatus,
		LastStep,
		LastStepDescription,
		StepCount,
		StartDateTime,
		LastDateTime,
		CompleteDateTime
	)
		SELECT	RolloutStatusTypeID,
			ComputerID,
			LastStatus,
			LastStep,
			LastStepDescription,
			StepCount,
			StartDateTime,
			LastDateTime,
			CompleteDateTime
		FROM	#RolloutStatusSummary AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatusSummary
			WHERE	RolloutStatusSummary.ComputerID = t.ComputerID
			  AND	RolloutStatusSummary.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatusSummary	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusReplaceSubsetByComputerIDWithDeleteOffset')
	DROP PROCEDURE RolloutStatusReplaceSubsetByComputerIDWithDeleteOffset
GO
-- Replace the record subset selected by (@ComputerID), (@DeleteOffset)
-- using unique identifier (ComputerID, StepNumber)
-- updating fields (RolloutStatusTypeID, Description, Status, StatusMessage)
-- whilst deleting entries with an StepNumber greater than the specified @DeleteOffset
CREATE PROCEDURE RolloutStatusReplaceSubsetByComputerIDWithDeleteOffset
	@ComputerID				int,
	@RolloutStatusTypeID	int,
	@DeleteOffset			int
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempRolloutStatusByComputerID]
		ON #RolloutStatus (ComputerID, StepNumber, RolloutStatusTypeID)

	BEGIN TRANSACTION
	DELETE RolloutStatus
		WHERE	ComputerID = @ComputerID
		AND NOT EXISTS(		-- Replacement record doesn't exist
			SELECT * FROM #RolloutStatus AS t
			WHERE	RolloutStatus.ComputerID = t.ComputerID
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
		AND RolloutStatusTypeID = @RolloutStatusTypeID
		AND StepNumber > @DeleteOffset

	UPDATE RolloutStatus
		SET	RolloutStatusTypeID = t.RolloutStatusTypeID,
			Description = t.Description,
			Status = t.Status,
			StatusMessage = t.StatusMessage,
			Phase = t.Phase
		FROM	#RolloutStatus AS t
		WHERE	RolloutStatus.ComputerID = t.ComputerID
		  AND	RolloutStatus.StepNumber = t.StepNumber
		  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
	SET @count = @@ROWCOUNT

	INSERT RolloutStatus (
		RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	)
		SELECT	RolloutStatusTypeID,
			ComputerID,
			StepNumber,
			Description,
			Status,
			StatusMessage,
			Phase
		FROM	#RolloutStatus AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM RolloutStatus
			WHERE	RolloutStatus.ComputerID = t.ComputerID
			  AND	RolloutStatus.StepNumber = t.StepNumber
			  AND	RolloutStatus.RolloutStatusTypeID = t.RolloutStatusTypeID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #RolloutStatus	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'RolloutStatusGetAllByComputerIDTypeID')
	DROP PROCEDURE RolloutStatusGetAllByComputerIDTypeID
GO
CREATE PROCEDURE RolloutStatusGetAllByComputerIDTypeID
	@ComputerID				int,
	@RolloutStatusTypeID	int
AS
	SELECT	RolloutStatusTypeID,
		ComputerID,
		StepNumber,
		Description,
		Status,
		StatusMessage,
		Phase
	FROM	RolloutStatus
	WHERE	ComputerID = @ComputerID
	  AND	RolloutStatusTypeID = @RolloutStatusTypeID
	  			ORDER BY StepNumber
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureGetByComputerID')
	DROP PROCEDURE DataCaptureGetByComputerID
GO
CREATE PROCEDURE DataCaptureGetByComputerID
	@ComputerID		int
AS
	SELECT	ComputerID,
		DataStoreLocation,
		DataImageCaptureLocation,
		LogFileLocation,
		CaptureDateTime,
		ImageCaptureDateTime
	FROM	DataCapture
	WHERE	ComputerID = @ComputerID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureGetAll')
	DROP PROCEDURE DataCaptureGetAll
GO
CREATE PROCEDURE DataCaptureGetAll
AS
	SELECT	ComputerID,
		DataStoreLocation,
		DataImageCaptureLocation,
		LogFileLocation,
		CaptureDateTime,
		ImageCaptureDateTime
	FROM	DataCapture
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureAdd')
	DROP PROCEDURE DataCaptureAdd
GO
CREATE PROCEDURE DataCaptureAdd
	@ComputerID		int,
	@DataStoreLocation	nvarchar(1024) = NULL,
	@DataImageCaptureLocation nvarchar(1024) = NULL,
	@LogFileLocation	nvarchar(1024) = NULL,
	@CaptureDateTime	datetime,
	@ImageCaptureDateTime	datetime
AS
	INSERT INTO DataCapture (
		ComputerID,
		DataStoreLocation,
		DataImageCaptureLocation,
		LogFileLocation,
		CaptureDateTime,
		ImageCaptureDateTime
	)
	VALUES (
		@ComputerID,
		@DataStoreLocation,
		@DataImageCaptureLocation,
		@LogFileLocation,
		@CaptureDateTime,
		@ImageCaptureDateTime
	)
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureSchema')
	DROP PROCEDURE DataCaptureSchema
GO
CREATE PROCEDURE DataCaptureSchema
AS
	SELECT
	'IF OBJECT_ID(''tempdb..#DataCapture'') IS NULL' +
	'	CREATE TABLE #DataCapture (' +
	'		ComputerID		int NOT NULL,' +
	'		DataStoreLocation	nvarchar(1024) '+dbo.ColumnCollate('DataCapture', 'DataStoreLocation')+' NULL,' +
	'		DataImageCaptureLocation nvarchar(1024) '+dbo.ColumnCollate('DataCapture', 'DataImageCaptureLocation')+' NULL,' +
	'		LogFileLocation		nvarchar(1024) '+dbo.ColumnCollate('DataCapture', 'LogFileLocation')+' NULL,' +
	'		CaptureDateTime		datetime NOT NULL DEFAULT getdate(),' +
	'		ImageCaptureDateTime	datetime NOT NULL DEFAULT getdate()' +
	'	)' +
	'ELSE' +
	'	DELETE FROM #DataCapture' +
	'' AS SQL
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureTempAdd')
	DROP PROCEDURE DataCaptureTempAdd
GO
CREATE PROCEDURE DataCaptureTempAdd
	@ComputerID		int,
	@DataStoreLocation	nvarchar(1024) = NULL,
	@DataImageCaptureLocation nvarchar(1024) = NULL,
	@LogFileLocation	nvarchar(1024) = NULL,
	@CaptureDateTime	datetime,
	@ImageCaptureDateTime	datetime
AS
	INSERT INTO #DataCapture (
		ComputerID,
		DataStoreLocation,
		DataImageCaptureLocation,
		LogFileLocation,
		CaptureDateTime,
		ImageCaptureDateTime
	)
	VALUES (
		@ComputerID,
		@DataStoreLocation,
		@DataImageCaptureLocation,
		@LogFileLocation,
		@CaptureDateTime,
		@ImageCaptureDateTime
	)
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureAddBatch')
	DROP PROCEDURE DataCaptureAddBatch
GO
CREATE PROCEDURE DataCaptureAddBatch
AS
	DECLARE @count int
	INSERT DataCapture (
		ComputerID,
		DataStoreLocation,
		DataImageCaptureLocation,
		LogFileLocation,
		CaptureDateTime,
		ImageCaptureDateTime
	)
		SELECT	ComputerID,
			DataStoreLocation,
			DataImageCaptureLocation,
			LogFileLocation,
			CaptureDateTime,
			ImageCaptureDateTime
		FROM #DataCapture
	SET @count = @@ROWCOUNT
	DROP TABLE #DataCapture	-- Drop temporary table
	RETURN @count
GO

/*
Add new DataCapture bound to
	Computer using ComputerOUID, ComputerCN
	with DataStoreLocation, DataImageCaptureLocation, LogFileLocation, CaptureDateTime, ImageCaptureDateTime
*/

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureUpdateByComputerID')
	DROP PROCEDURE DataCaptureUpdateByComputerID
GO
CREATE PROCEDURE DataCaptureUpdateByComputerID
	@ComputerID		int,
	@DataStoreLocation	nvarchar(1024) = NULL,
	@DataImageCaptureLocation nvarchar(1024) = NULL,
	@LogFileLocation	nvarchar(1024) = NULL,
	@CaptureDateTime	datetime,
	@ImageCaptureDateTime	datetime
AS
	UPDATE	DataCapture
	SET	DataStoreLocation = @DataStoreLocation,
		DataImageCaptureLocation = @DataImageCaptureLocation,
		LogFileLocation = @LogFileLocation,
		CaptureDateTime = @CaptureDateTime,
		ImageCaptureDateTime = @ImageCaptureDateTime
	WHERE	ComputerID = @ComputerID
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureUpdateBatchByComputerID')
	DROP PROCEDURE DataCaptureUpdateBatchByComputerID
GO
CREATE PROCEDURE DataCaptureUpdateBatchByComputerID
AS
	DECLARE @count int

	CREATE INDEX [TempDataCaptureComputerID]
		ON #DataCapture (ComputerID)

	UPDATE	DataCapture
	SET	DataStoreLocation = t.DataStoreLocation,
		DataImageCaptureLocation = t.DataImageCaptureLocation,
		LogFileLocation = t.LogFileLocation,
		CaptureDateTime = t.CaptureDateTime,
		ImageCaptureDateTime = t.ImageCaptureDateTime
	FROM	#DataCapture AS t
	WHERE	DataCapture.ComputerID = t.ComputerID
	SET @count = @@ROWCOUNT
	DROP TABLE #DataCapture	-- Drop the temporary table
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCapturePutByComputerID')
	DROP PROCEDURE DataCapturePutByComputerID
GO
CREATE PROCEDURE DataCapturePutByComputerID
	@ComputerID		int,
	@DataStoreLocation	nvarchar(1024) = NULL,
	@DataImageCaptureLocation nvarchar(1024) = NULL,
	@LogFileLocation	nvarchar(1024) = NULL,
	@CaptureDateTime	datetime,
	@ImageCaptureDateTime	datetime
AS
	IF NOT EXISTS(
		SELECT	1
		FROM	DataCapture
		WHERE	ComputerID = @ComputerID
	)
		EXEC DataCaptureAdd @ComputerID,
			@DataStoreLocation,
			@DataImageCaptureLocation,
			@LogFileLocation,
			@CaptureDateTime,
			@ImageCaptureDateTime
	ELSE
		EXEC DataCaptureUpdateByComputerID @ComputerID,
			@DataStoreLocation,
			@DataImageCaptureLocation,
			@LogFileLocation,
			@CaptureDateTime,
			@ImageCaptureDateTime
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCapturePutBatchByComputerID')
	DROP PROCEDURE DataCapturePutBatchByComputerID
GO
CREATE PROCEDURE DataCapturePutBatchByComputerID
AS
	DECLARE @count AS int
	SET @count = 0

	CREATE INDEX [TempDataCaptureComputerID]
		ON #DataCapture (ComputerID)

	BEGIN TRANSACTION
	UPDATE DataCapture
		SET	DataStoreLocation = t.DataStoreLocation,
			DataImageCaptureLocation = t.DataImageCaptureLocation,
			LogFileLocation = t.LogFileLocation,
			CaptureDateTime = t.CaptureDateTime,
			ImageCaptureDateTime = t.ImageCaptureDateTime
		FROM	#DataCapture AS t
		WHERE	DataCapture.ComputerID = t.ComputerID
	SET @count = @@ROWCOUNT

	INSERT DataCapture (
		ComputerID,
		DataStoreLocation,
		DataImageCaptureLocation,
		LogFileLocation,
		CaptureDateTime,
		ImageCaptureDateTime
	)
		SELECT	ComputerID,
			DataStoreLocation,
			DataImageCaptureLocation,
			LogFileLocation,
			CaptureDateTime,
			ImageCaptureDateTime
		FROM	#DataCapture AS t
		WHERE NOT EXISTS(	-- Current record doesn't exist
			SELECT * FROM DataCapture
			WHERE	DataCapture.ComputerID = t.ComputerID
		)
	SET @count = @count + @@ROWCOUNT
	DROP TABLE #DataCapture	-- Drop the temporary table
	COMMIT WORK
	RETURN @count
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'DataCaptureRemoveAll')
	DROP PROCEDURE DataCaptureRemoveAll
GO
CREATE PROCEDURE DataCaptureRemoveAll
	
AS
	DELETE	DataCapture
	RETURN @@ROWCOUNT
GO

IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
		AND name = 'ComputerGetByOUIDCN')
	DROP PROCEDURE ComputerGetByOUIDCN
GO
CREATE PROCEDURE ComputerGetByOUIDCN
	@ComputerOUID		int,
	@ComputerCN		nvarchar(64)
AS
	SELECT	ComputerID,
		ComputerOUID,
		ComputerCN,
		OperatingSystemID,
		GUID
	FROM	Computer
	WHERE	ComputerOUID = @ComputerOUID
	  AND	ComputerCN = @ComputerCN
GO
/* AUTOMATICALLY GENERATED on Sat Jan 17 18:57:05 AUSEDT 2009
 * for ManageSoft release 8.0 (build 8.4033)
 *
 * DO NOT EDIT THIS FILE BY HAND!
 * 
 * Protected by US patents 6,006,034 and 6,360,366.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

USE ManageSoft
GO

PRINT '------------------------'
PRINT 'Creating database tables'
PRINT '------------------------'
GO


PRINT '------------------------'
PRINT 'Creating database views'
PRINT '------------------------'
GO

IF EXISTS (SELECT * FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_NAME = 'WiMiMigrationStateByComputer') BEGIN
	DROP VIEW [WiMiMigrationStateByComputer]
END
GO

PRINT 'Creating WiMiMigrationStateByComputer view'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/schema/src/WiMiMigrationStateByComputer.sql,v 1.12 2007/07/24 06:31:02 pca Exp $
 *
 * Description
 *	Migration state for each computer
 */

-- Create migration status view
IF EXISTS (SELECT TABLE_NAME FROM INFORMATION_SCHEMA.VIEWS
         WHERE TABLE_NAME = 'WiMiMigrationStateByComputer')
   DROP VIEW WiMiMigrationStateByComputer
GO

CREATE VIEW WiMiMigrationStateByComputer
AS
SELECT
CASE
-- Find any installation or policy failures
WHEN r.LastStatus = 'Failure' OR EXISTS (
	SELECT *
	FROM Installation i
	WHERE i.ComputerID = c.ComputerID
	AND i.Result = 'failure'
)
	THEN 'Migration failed'
-- Check operating system
WHEN (r.CompleteDateTime IS NOT NULL)
	THEN 'Migration succeeded'
ELSE	'Not migrated'
END AS State,
c.ComputerID,
c.ComputerCN,
c.ComputerOUID,
o.OperatingSystemName,
r.LastStatus,
r.LastStep,
r.LastStepDescription,
r.StepCount,
r.StartDateTime,
r.LastDateTime,
r.CompleteDateTime
FROM	Computer c
	LEFT OUTER JOIN OperatingSystem o ON c.OperatingSystemID = o.OperatingSystemID
	LEFT OUTER JOIN
	(
		SELECT	rs.*
		FROM	RolloutStatusSummary AS rs
		INNER JOIN	RolloutStatusType AS rt ON rs.RolloutStatusTypeID = rt.RolloutStatusTypeID
			AND rt.[Name] = 'ComputerBuild'
		
	) AS r ON c.ComputerID = r.ComputerID
	LEFT OUTER JOIN RolloutStatusType t ON r.RolloutStatusTypeID = t.RolloutStatusTypeID
GO

GO

IF EXISTS (SELECT * FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_NAME = 'WiMiImageCaptureStateByComputer') BEGIN
	DROP VIEW [WiMiImageCaptureStateByComputer]
END
GO

PRINT 'Creating WiMiImageCaptureStateByComputer view'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/schema/src/WiMiImageCaptureStateByComputer.sql,v 1.2 2007/07/24 06:31:02 pca Exp $
 *
 * Description
 *	Image capture state for each computer
 */

-- Create image capture state status view
IF EXISTS (SELECT TABLE_NAME FROM INFORMATION_SCHEMA.VIEWS
         WHERE TABLE_NAME = 'WiMiImageCaptureStateByComputer')
   DROP VIEW WiMiImageCaptureStateByComputer
GO

CREATE VIEW WiMiImageCaptureStateByComputer
AS
SELECT
CASE
-- Find any installation or policy failures
WHEN r.LastStatus = 'Failure'
	THEN 'ImageCaptureFailed'
WHEN (r.LastStatus = 'Success' AND r.CompleteDateTime IS NOT NULL)
	THEN 'ImageCaptureSucceeded'
ELSE
	'ImageCaptureInProgress'
END AS State,
c.ComputerID,
c.ComputerCN,
c.ComputerOUID,
o.OperatingSystemName,
r.LastStatus,
r.LastStep,
r.LastStepDescription,
r.StepCount,
r.StartDateTime,
r.LastDateTime,
r.CompleteDateTime,
d.DataImageCaptureLocation,
d.ImageCaptureDateTime
FROM	Computer c
	LEFT OUTER JOIN OperatingSystem o ON c.OperatingSystemID = o.OperatingSystemID
	INNER JOIN
	(
		SELECT	rs.*
		FROM	RolloutStatusSummary AS rs
		INNER JOIN	RolloutStatusType AS rt ON rs.RolloutStatusTypeID = rt.RolloutStatusTypeID
			AND rt.[Name] = 'ComputerImageCapture'
		
	) AS r ON c.ComputerID = r.ComputerID
	LEFT OUTER JOIN RolloutStatusType t ON r.RolloutStatusTypeID = t.RolloutStatusTypeID
	LEFT OUTER JOIN DataCapture d ON c.ComputerID = d.ComputerID
GO

GO

IF EXISTS (SELECT * FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_NAME = 'WiMiDataCaptureStateByComputer') BEGIN
	DROP VIEW [WiMiDataCaptureStateByComputer]
END
GO

PRINT 'Creating WiMiDataCaptureStateByComputer view'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/schema/src/WiMiDataCaptureStateByComputer.sql,v 1.2 2007/07/24 06:31:02 pca Exp $
 *
 * Description
 *	Data capture state for each computer
 */

-- Create data capture status view
IF EXISTS (SELECT TABLE_NAME FROM INFORMATION_SCHEMA.VIEWS
         WHERE TABLE_NAME = 'WiMiDataCaptureStateByComputer')
   DROP VIEW WiMiDataCaptureStateByComputer
GO

CREATE VIEW WiMiDataCaptureStateByComputer
AS
SELECT
CASE
-- Find any installation or policy failures
WHEN r.LastStatus = 'Failure'
	THEN 'DataCaptureFailed'
WHEN (r.LastStatus = 'Success' AND r.CompleteDateTime IS NOT NULL)
	THEN 'DataCaptureSucceeded'
ELSE
	'DataCaptureInProgress'
END AS State,
c.ComputerID,
c.ComputerCN,
c.ComputerOUID,
o.OperatingSystemName,
r.LastStatus,
r.LastStep,
r.LastStepDescription,
r.StepCount,
r.StartDateTime,
r.LastDateTime,
r.CompleteDateTime,
d.DataStoreLocation,
d.CaptureDateTime
FROM	Computer c
	LEFT OUTER JOIN OperatingSystem o ON c.OperatingSystemID = o.OperatingSystemID
	INNER JOIN
	(
		SELECT	rs.*
		FROM	RolloutStatusSummary AS rs
		INNER JOIN	RolloutStatusType AS rt ON rs.RolloutStatusTypeID = rt.RolloutStatusTypeID
			AND rt.[Name] = 'ComputerDataCapture'
		
	) AS r ON c.ComputerID = r.ComputerID
	LEFT OUTER JOIN RolloutStatusType t ON r.RolloutStatusTypeID = t.RolloutStatusTypeID
	LEFT OUTER JOIN DataCapture d ON c.ComputerID = d.ComputerID
GO

GO

/*
 * $Header: /cvsroot/nd/branched/src/database/schema/src/HardwareAssessment-schema.sql,v 1.8 2007/07/24 06:31:01 pca Exp $
 *
 * DESCRIPTION:
 *
 *	Schema for Windows Migration - Hardware Assessment.
 *	-Creates required TABLES
 *	
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */


USE ManageSoft
GO

/* ----------------------------------------------------------------- */
PRINT '-----------------------------------------';
PRINT '--- Creating Windows Migration tables ---'
PRINT '-----------------------------------------';
/* ----------------------------------------------------------------- */

IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'HardwareAssessmentSeverity'
)
BEGIN
	PRINT 'HardwareAssessmentSeverity Table:'
	CREATE TABLE HardwareAssessmentSeverity(

		-- Unique Identifier of HardwareAssessmentSeverity
		AssessmentSeverityID int NOT NULL PRIMARY KEY,			

		--Unique number representing the order of the severity for assessment
		--For calculating final assessment, AssessmentSeverity with higher number overrides 
		--Example: if a computer has assessment 'Requires replacement' for one rule and 
		--	assessment 'Requires upgrade' for another. Final assessment for the said
		--	computer will be 'Requires replacement' as it has higher severity.
		AssessmentSeverityNumber int NOT NULL 
			CONSTRAINT UN_AssessmentSeverityNunmber UNIQUE,

		--Text representing the assessment severity
		--Example: 'Requires replacement', 'Requires upgrade'
		AssessmentSeverityMessage varchar(64) NOT NULL	
			CONSTRAINT UN_AssessmentSeverityMessage UNIQUE
	);
	
	-- Populate table with initial/default hardware assessment severity
	INSERT INTO HardwareAssessmentSeverity VALUES (1, 1, 'Meets requirements');
	INSERT INTO HardwareAssessmentSeverity VALUES (2, 2, 'Requires upgrade');
	INSERT INTO HardwareAssessmentSeverity VALUES (3, 3, 'Requires replacement');
	INSERT INTO HardwareAssessmentSeverity VALUES (4, 4, 'Not inventoried');

END /* HardwareAssessmentSeverity */
ELSE
	PRINT 'Using existing HardwareAssessmentSeverity table'

GO

/* ----------------------------------------------------------------- */

IF EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'HardwareAssessmentRule'
)
BEGIN
	PRINT 'Using existing HardwareAssessmentRule table'
	IF NOT EXISTS (
		SELECT *
		FROM INFORMATION_SCHEMA.COLUMNS
		WHERE TABLE_NAME = 'HardwareAssessmentRule' AND COLUMN_NAME = 'FriendlyName'
	)
	BEGIN
		PRINT 'Adding FriendlyName column to HardwareAssessmentRule table'
		ALTER TABLE HardwareAssessmentRule ADD FriendlyName nvarchar(256) NULL
		PRINT 'Adding DefaultRule values to HardwareAssessmentRule entries'
	END
	IF NOT EXISTS (
		SELECT *
		FROM INFORMATION_SCHEMA.COLUMNS
		WHERE TABLE_NAME = 'HardwareAssessmentRule' AND COLUMN_NAME = 'Enabled'
	)
	BEGIN
		PRINT 'Adding Enabled column to HardwareAssessmentRule table'
		ALTER TABLE HardwareAssessmentRule ADD Enabled bit DEFAULT (1) NOT NULL 
	END
	IF NOT EXISTS (
		SELECT *
		FROM INFORMATION_SCHEMA.COLUMNS
		WHERE TABLE_NAME = 'HardwareAssessmentRule' AND COLUMN_NAME = 'DefaultRule'
	)
	BEGIN
		PRINT 'Adding DefaultRule column to HardwareAssessmentRule table'
		ALTER TABLE HardwareAssessmentRule ADD DefaultRule bit DEFAULT (0) NOT NULL
	END
END
GO

IF EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'HardwareAssessmentRule'
)
BEGIN
	-- If we added new columns, set their values for the default rules.
	UPDATE HardwareAssessmentRule
		SET FriendlyName = 'Total Physical Memory', DefaultRule = 1
		WHERE FriendlyName IS NULL
		  AND Class = 'Win32_ComputerSystem'
		  AND Property = 'TotalPhysicalMemory'
	UPDATE HardwareAssessmentRule
		SET FriendlyName = 'Clock speed of the Processor', DefaultRule = 1
		WHERE FriendlyName IS NULL
		  AND Class = 'Win32_Processor'
		  AND Property = 'CurrentClockSpeed'
	UPDATE HardwareAssessmentRule
		SET FriendlyName = 'ACPI support in BIOS', DefaultRule = 1
		WHERE FriendlyName IS NULL
		  AND Class = 'Win32_BIOS'
		  AND Property = 'BiosCharacteristics'
	UPDATE HardwareAssessmentRule
		SET FriendlyName = 'Size of Disk', DefaultRule = 1
		WHERE FriendlyName IS NULL
		  AND Class = 'Win32_DiskDrive'
		  AND Property = 'Size'
	UPDATE HardwareAssessmentRule
		SET FriendlyName = 'Custom Rule', DefaultRule = 0
		WHERE FriendlyName IS NULL
		
	--Update Property for HW Class 'Win32_Processor'
	UPDATE HardwareAssessmentRule
		SET Property = 'MaxClockSpeed', RuleModifiedDate = GetDate()
		WHERE Class = 'Win32_Processor'
			AND Property = 'CurrentClockSpeed'
END
GO

IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'HardwareAssessmentRule'
)
BEGIN
	PRINT 'HardwareAssessmentRule Table:'
	CREATE TABLE HardwareAssessmentRule(
		-- Auto-generated identity
		AssessmentRuleID int NOT NULL IDENTITY
			CONSTRAINT PK_HardwareAssessmentRule PRIMARY KEY NONCLUSTERED,

		--DateTime the rule was either created or modified. 
		RuleModifiedDate datetime 
			DEFAULT GETDATE() NOT NULL,

		--AssessmentRule belongs to which set of rules
		--By default: all rules belong to one default set and value is NULL 
		AssessmentRuleSet varchar(64),  --DEFAULT 'default', 

		--Assessment of computer if this assessment rule FAILS. 
		--Must be one of 'Requires upgrade' or 'Requires replacement'. 
		AssessmentSeverityID int NOT NULL
			CONSTRAINT FK_HardwareAssessmentServerity_SeverityMessage 
				REFERENCES HardwareAssessmentSeverity(AssessmentSeverityID),			

		-- Class of property to check.
		-- Example: 'Win32_OperatingSystem', 'Win32_BIOS'
		Class varchar(256) NOT NULL,

		-- Name of the property to check.
		-- Example: 'TotalPhysicalMemory', 'Version'
		Property varchar(64) NOT NULL,

		-- Comparsion expression to use.
		-- Examples: '< 261664', 'in ('A', 'B', 'C')'
		ComparisonExpr varchar(1024) NOT NULL,

		-- SQL type to cast the value to for comparision.
		-- NULL if no cast is required (i.e. use string comparison).
		-- Example: 'int'
		SQLType varchar(20) NULL
			CHECK (	SQLType IS NULL
				OR SQLType IN ('bigint', 'int', 'smallint', 'tinyint', 'bit', 'float', 'real')
				OR SQLType LIKE 'numeric%'
				OR SQLType LIKE 'decimal%'
				OR SQLType LIKE 'varchar%'
				OR SQLType LIKE 'nvarchar%'
			),

		-- Aggregate function to use (SUM, MAX, ...).
		-- NULL if value should not be aggregated
		-- Example: 'SUM'
		AggregateFn varchar(10) NULL,

		-- Message to use if Class.Property is not known.
		-- Example: 'Information about physical memory is not available'
		NotKnownMessage nvarchar(256) NOT NULL,

		-- Message to use if Class.Property does not pass this condition
		-- Example: '{0:b} physical memory is not enough for this computer; must be upgraded to at least 256Mb'
		FailMessage nvarchar(256) NOT NULL,

		-- Message to use if Class.Property passes this condition.
		-- Example: 'Computer has enough physical memory ({0:b})'
		PassMessage nvarchar(256) NOT NULL,
	
		-- Friendly Name of the rule to display in the UI
		FriendlyName nvarchar(256) NULL,

		-- Whether the rule will be ignored
		Enabled bit DEFAULT (1) NOT NULL,

		-- Whether the rule is a default
		DefaultRule bit DEFAULT (0) NOT NULL
	);

	-- Populate table with some standard hardware assessment rules

	--For actual expression value less than 128MB is used (126MB)
	INSERT INTO HardwareAssessmentRule (
		AssessmentSeverityID, Class, Property,
		ComparisonExpr, SQLType, AggregateFn,
		NotKnownMessage, FailMessage, PassMessage,
		FriendlyName, Enabled, DefaultRule
	)
	VALUES (
		2, 'Win32_ComputerSystem', 'TotalPhysicalMemory',
		'< 536870912', 'numeric(13)', NULL, 
		'Information about physical memory is not available for this computer',
		'This computer only has {0:b} physical memory which is not sufficient for migration.',
		'This computer has {0:b} of physical memory installed which is sufficient for migration.',
		'Total Physical Memory', 1, 1
	);

	--For actual expression value less than 1000MHz is used (997MHz)
	INSERT INTO HardwareAssessmentRule (
		AssessmentSeverityID, Class, Property, 
		ComparisonExpr, SQLType, AggregateFn,
		NotKnownMessage, FailMessage, PassMessage,
		FriendlyName, Enabled, DefaultRule
	)
	VALUES (
		3, 'Win32_Processor', 'MaxClockSpeed',
		'< 997', 'int', 'MAX',
		'Information about CPU speed is not available for this computer',
		'This computer''s CPU only runs at {0} MHz which is not sufficient for migration.',
		'This computer''s CPU runs at {0} MHz which is sufficient for migration.',
		'Clock speed of the Processor', 1, 1
	);

	INSERT INTO HardwareAssessmentRule (
		AssessmentSeverityID, Class, Property, 
		ComparisonExpr, SQLType, AggregateFn,
		NotKnownMessage, FailMessage, PassMessage,
		FriendlyName, Enabled, DefaultRule
	)
	VALUES (
		3, 'Win32_BIOS', 'BiosCharacteristics',
		'NOT LIKE ''%,32[^0-9]%''', NULL, NULL,
		'Information about BIOS characteristics is not available for this computer',
		'This computer''s BIOS does not support ACPI. ACPI support is required for migration.',
		'This computer''s BIOS supports ACPI.',
		'ACPI support in BIOS', 1, 1
	);

	--For actual expression value less than 2GB is used (1.86GB) 
	INSERT INTO HardwareAssessmentRule (
		AssessmentSeverityID, Class, Property, 
		ComparisonExpr, SQLType, AggregateFn,
		NotKnownMessage, FailMessage, PassMessage,
		FriendlyName, Enabled, DefaultRule
	)
	VALUES (
		2, 'Win32_DiskDrive', 'Size',
		'< 20000000000', 'numeric(13)', 'MAX',
		'Information about disk drives is not available for this computer',
		'The capacity of this computer''s largest disk drive is only {0:b} which is not sufficient for migration.',
		'The capacity of this computer''s largest disk drive is {0:b} which is sufficient for migration.',
		'Size of Disk', 1, 1
	);

END /* HardwareAssessmentRule */
GO


/* ----------------------------------------------------------------- */

IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'HardwareAssessmentCache'
)
BEGIN
	PRINT 'HardwareAssessmentCache Table:'
	CREATE TABLE HardwareAssessmentCache(

		--Computer for which assessment is calculated, FK to Computer		
		ComputerID int NOT NULL,		
		--NOTE: Constraint is not created as it would not allow delete
		--in Computer table
			--CONSTRAINT FK_HardwareAssessmentCache_Computer
				--REFERENCES Computer(ComputerID),

		/*
		 * FK to OU of Computer. This is denormalised, since it's
		 * dependent on other data in this row, but useful for
		 * clustering!
		 */
		ComputerOUID int NOT NULL,
		--NOTE: Constraint is not created as it would not allow delete
		--in SDOU table
			--CONSTRAINT FK_HardwareAssessmentCache_ComputerOU
				--REFERENCES SDOU(OUID),

		--AssessmentRule applied for computer, FK to HardwareAssessmentRule
		AssessmentRuleID int NOT NULL
			CONSTRAINT FK_HardwareAssessmentCache_AssessmentRule
				REFERENCES HardwareAssessmentRule(AssessmentRuleID),

		--Resulting Assessment for computer, FK to HardwareAssessmentSeverity
		--based on AssessmentRule
		AssessmentSeverityID int NOT NULL
			CONSTRAINT FK_HardwareAssessmentCache_AssessmentSeverity
				REFERENCES HardwareAssessmentSeverity(AssessmentSeverityID),

		--Date when Hardware was last inventoried for given computer
		--Value can be NULL if computer never reported inventory
		LastHardwareReportDate datetime,
	
		--DateTime when the record for given computer was evaluated
		CacheEvaluatedDate datetime DEFAULT GETDATE() NOT NULL

		CONSTRAINT PK_HardwareAssessmentCache 
			PRIMARY KEY CLUSTERED (AssessmentRuleID, ComputerID)

	);

	PRINT '	HardwareAssessmentByComputerOU index'
	CREATE INDEX HardwareAssessmentCacheByComputerOU
		ON HardwareAssessmentCache(ComputerOUID, ComputerID);

	PRINT '	HardwareAssessmentByComputer index'
	CREATE INDEX HardwareAssessmentCacheByComputer 
		ON HardwareAssessmentCache(ComputerID);

END /* HardwareAssessmentCache */
ELSE
	PRINT 'Using existing HardwareAssessmentCache table'

GO
/*
 * $Header: /cvsroot/nd/branched/src/database/schema/src/RolloutGroups-schema.sql,v 1.14 2008/10/13 06:41:33 pca Exp $
 *
 * DESCRIPTION:
 *
 *	Schema for Group and related tables.
 *	-Creates required TABLES
 *	
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */


USE ManageSoft
GO

/* ----------------------------------------------------------------- */
PRINT '---------------------------------------------------------';
PRINT '--- Creating Windows Deployment: Rollout Group tables ---';
PRINT '---------------------------------------------------------';
/* ----------------------------------------------------------------- */

IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutGroup'
)
BEGIN
	PRINT 'RolloutGroup Table:'
	CREATE TABLE RolloutGroup(

		-- Auto-generated Identity of RolloutGroup
		GroupID int NOT NULL IDENTITY
			CONSTRAINT PK_RolloutGroup PRIMARY KEY,			

		-- Rollout Group Name
		-- NOTE: Can have duplicate names though not under same parent
		-- This will have to be managed at resolving time
		GroupName nvarchar(256) NOT NULL, 			

		-- 0 = represents the Topmost RolloutGroup. Can have multiple groups
		-- at this level.	
		-- Ex: When source is Ghost; then Topmost Parent represents
		-- the GhostServer; under which the Root Groups are kept 		
		ParentID int NOT NULL DEFAULT(0),			

		--REVISIT:
		-- Can be NULL when it represents Topmost RolloutGroup
		-- (can have NULL if not a reference)
		--ParentID int NULL REFERENCES RolloutGroup(GroupID),		
				
		-- Represents the Source of RolloutGroup
		-- Ex: Ghost 7.5		
		Source nvarchar(64) NULL	
	);
	
END /* RolloutGroup */
ELSE
	PRINT 'Using existing RolloutGroup table'

GO

/* ----------------------------------------------------------------- */

IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutGroupMember'
)
BEGIN
	PRINT 'RolloutGroupMember Table:'
	CREATE TABLE RolloutGroupMember(
			
		-- This RolloutGroup ....
		GroupID int NOT NULL
			CONSTRAINT FK_RolloutGroupMember_GroupID
				REFERENCES RolloutGroup(GroupID),
				
		-- ...contains Computer (ManageSoft ComputerID)
		-- And other types like User but *NOT* another 
		-- RolloutGroup (info in Group table)		
		MemberID int NOT NULL,
		
		-- type of the member - comes from TargetType table and
		-- must be a Computer (1) or in future User(2)
		MemberType int NOT NULL,
		
		--Unique as a Group can have a member listed only once
		CONSTRAINT UN_RolloutGroupMember UNIQUE(GroupID, MemberID, MemberType)
	);
	
END /* RolloutGroupMember */
ELSE
	PRINT 'Using existing RolloutGroupMember table'

GO
	
PRINT ' RolloutGroupMemberByType index'
IF NOT EXISTS (
    SELECT *
    FROM sysindexes
    WHERE name = 'RolloutGroupMemberByType'
)
	CREATE CLUSTERED INDEX RolloutGroupMemberByType
		ON RolloutGroupMember(MemberType, MemberID);

GO

/* ----------------------------------------------------------------- */

IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'GhostServer'
)
BEGIN
	PRINT 'GhostServer Table:'
	CREATE TABLE GhostServer(

		GhostServerID int NOT NULL IDENTITY
			CONSTRAINT PK_GhostServer PRIMARY KEY,
		
		-- GhostServer is the *Parent* of root Rollout groups for a given 
		-- Ghost Console Server
		GroupID int NOT NULL 
			CONSTRAINT FK_GhostServer_GroupID
				REFERENCES RolloutGroup(GroupID),
		
		-- Name of Ghost Console Server
		-- This can be duplicates but unique in a domain
		ServerName nvarchar(256) NOT NULL, 
		
		--Name of the Domain to which Ghost Server belongs to		
		DomainName nvarchar(256) DEFAULT(NULL), 
		
		-- Network adapter physical address of the node
		-- Can be MAC address or token ring address
		PhysicalAddress varchar(64) NULL DEFAULT(NULL),
		
		-- ManageSoft Computer ID
		ComputerID int NULL DEFAULT(NULL),
		
		-- DateTime when Group XML was generated
		LastUpdateDateTime DATETIME NULL
	);
	
END /* GhostServer */
ELSE
	PRINT 'Using existing GhostServer table'

GO

/* ----------------------------------------------------------------- */
IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutGroupToGhostComputer'
)
BEGIN
	PRINT 'RolloutGroupToGhostComputer Table:'
	CREATE TABLE RolloutGroupToGhostComputer(

		-- This Rollout Group ....
		GroupID int NOT NULL
			CONSTRAINT FK_RolloutGroupToGhostComputer_GroupID
				REFERENCES RolloutGroup(GroupID),				
		
		-- ...contains Computer
		--NOTE: Can be NULL if PhysicalAddress is not.				
		ComputerName nvarchar(64) NULL DEFAULT(NULL),
				
		-- Can be NULL if ComputerName is not.
		PhysicalAddress varchar(64) NULL DEFAULT(NULL),
		
		-- Is resolved into RolloutGroupMember table 
		-- with ManageSoft ComputerIDs 
		IsResolved int NOT NULL DEFAULT(0)
	);
	
END /* RolloutGroupToGhostComputer */
ELSE
	PRINT 'Using existing RolloutGroupToGhostComputer table'

GO

/* ----------------------------------------------------------------- */


IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'NameMap'
)
BEGIN
	PRINT 'NameMap Table:'
	CREATE TABLE NameMap(

		-- This name mapping ....
		ID int NOT NULL IDENTITY,
		
		-- old computer name
		Name nvarchar(64) NULL DEFAULT(NULL),

		-- old computer domain
		Domain int DEFAULT(NULL),

		-- new computer name
		NewName nvarchar(64) NULL DEFAULT(NULL),

		-- new computer domain
		NewDomain int DEFAULT(NULL),

		-- the name the comuter currently has
		CurrentID int NULL DEFAULT(NULL),
		
		-- the OU new computers are assigned into
		-- this is a foreign key onto the Organization table
		NewOU int NULL DEFAULT(NULL),
		
		-- the MACAddress of new computers. Existing ones will
		-- get their address from inventory.
		MACAddress nvarchar(32) NULL DEFAULT(NULL),
		
		-- the IP address of new computers
		IPAddress nvarchar(32) NULL DEFAULT(NULL)
	);
	
END /* NameMap */
ELSE
	PRINT 'Using existing NameMap table'
	IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE COLUMN_NAME='NewOU' AND TABLE_NAME='NameMap')
	BEGIN
		ALTER TABLE NameMap
			ADD NewOU int NULL DEFAULT(NULL)
	END
	IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE COLUMN_NAME='MACAddress' AND TABLE_NAME='NameMap')
	BEGIN
        ALTER TABLE NameMap
                ADD MACAddress nvarchar(32) NULL DEFAULT(NULL)
	END
	IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE COLUMN_NAME='IPAddress' AND TABLE_NAME='NameMap')
	BEGIN
        ALTER TABLE NameMap
                ADD IPAddress nvarchar(32) NULL DEFAULT(NULL)
	END
GO

PRINT ' NameMapNameDomain index'
IF NOT EXISTS (
    SELECT *
    FROM sysindexes
    WHERE name = 'NameMapNameDomain'
)
	CREATE INDEX NameMapNameDomain
		ON NameMap(name, domain);

GO

PRINT ' NameMapNewNameDomain index'
IF NOT EXISTS (
    SELECT *
    FROM sysindexes
    WHERE name = 'NameMapNewNameDomain'
)
	CREATE INDEX NameMapNewNameDomain
		ON NameMap(newName, newDomain);

GO

PRINT ' NameMapID index'
IF NOT EXISTS (
    SELECT *
    FROM sysindexes
    WHERE name = 'NameMapID'
)
	CREATE CLUSTERED INDEX NameMapID
		ON NameMap(CurrentId);

GO

-- Populate rollout status summary as we did-pre ManageSoft 7.2
IF NOT EXISTS (SELECT * FROM DatabaseConfiguration WHERE Property = 'WiDepSchemaVersion')
	INSERT	RolloutStatusSummary (
			ComputerID,
			LastStatus,
			CompleteDateTime
		)
		SELECT	C.ComputerID,
			'Success' AS LastStatus,
			getDate() AS CompleteDateTime
		FROM	Computer AS C
			LEFT OUTER JOIN RolloutStatusSummary as R ON R.ComputerID = C.ComputerID,
			OperatingSystem AS O
		WHERE	C.OperatingSystemID = O.OperatingSystemID
		AND	R.ComputerID IS NULL
		AND	(O.OperatingSystemName LIKE 'Microsoft Windows 2000%'
		OR	O.OperatingSystemName LIKE 'Microsoft Windows XP%'
		OR	O.OperatingSystemName LIKE 'Microsoft Windows Server 2003%')

IF NOT EXISTS (SELECT * FROM DatabaseConfiguration WHERE Property = 'WiDepSchemaVersion')
BEGIN
	INSERT INTO DatabaseConfiguration (Property, Value) VALUES ('WiDepSchemaVersion','8.0')
END
GO

/* ----------------------------------------------------------------- */
/* AUTOMATICALLY GENERATED on Sat Jan 17 18:57:33 AUSEDT 2009
 * for ManageSoft release 8.0 (build 8.4033)
 *
 * DO NOT EDIT THIS FILE BY HAND!
 * 
 * Protected by US patents 6,006,034 and 6,360,366.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

USE ManageSoft
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/RolloutGroupParentName.sql,v 1.4 2003/04/23 04:35:30 vj Exp $
 *
 * DESCRIPTION: Stored Procedure: RolloutGroupParentName
 * - returns full name of group
 *
 *	Input:
 *	- @inGroupID
 *	- @outFullName (output parameter)
 *
 *	Returns:
 *	- none
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'RolloutGroupParentName' AND xtype = 'P'
)
        DROP PROCEDURE RolloutGroupParentName
GO

CREATE PROCEDURE RolloutGroupParentName
	@inGroupID INT,  -- Pass in GroupID
	@outFullName NVARCHAR(800) OUTPUT
AS
BEGIN

DECLARE @fullName as NVARCHAR (800)
DECLARE @grpName as NVARCHAR(64)
DECLARE @tmpName as NVARCHAR(64)
DECLARE @parentID as INT

DECLARE grp_cur CURSOR FOR
	SELECT GroupName, ParentID 
	FROM RolloutGroup
	WHERE GroupID = @inGroupID

OPEN grp_cur
FETCH NEXT FROM grp_cur INTO @tmpName, @parentID

CLOSE grp_cur
DEALLOCATE grp_cur	

SET @fullName  = ''
SET @grpName  = @tmpName

--Check parent is not root or server and exists	
WHILE @parentID <> 0 AND EXISTS(SELECT * FROM RolloutGroup WHERE GroupID=@parentID)
BEGIN
	DECLARE grpParent_cur CURSOR FOR 
		SELECT GroupName , ParentID  
		FROM RolloutGroup
		WHERE GroupID = @parentID		

	OPEN grpParent_cur
	FETCH NEXT from grpParent_cur into @tmpName, @parentID		
				
	CLOSE grpParent_cur
	DEALLOCATE grpParent_cur
	
	--Check parent is not root or server and exists	
	IF @parentID <> 0 AND EXISTS(SELECT * FROM RolloutGroup WHERE GroupID=@parentID)
	BEGIN	
		IF @fullname <> ''
			SET @fullname = @tmpName + '->' + @fullname
		ELSE
			SET @fullname = @tmpName 			
	END
	ELSE
	BEGIN
		IF @fullname <> ''
			SET @fullname = @tmpName + '->' + @fullname
		ELSE
			SET @fullname = @tmpName 
		BREAK
	END
END 	

IF @fullname <> ''
	SET  @outFullName = @grpName + ' (' + @fullName + ')'
ELSE
	SET  @outFullName = @grpName

RETURN 0 

END

GO

CopyPermission 'RolloutGroupParentName', 'mgsOSBreakdown'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/ListRolloutGroups.sql,v 1.7 2008/10/29 00:58:01 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: ListRolloutGroups
 * - returns full name of group
 *
 *	Input:
 *	- @rolloutgrp
 *	- @oudn 
 *	- @ouMatch
 *
 *	Returns:	
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'ListRolloutGroups' AND xtype = 'P'
)
        DROP PROCEDURE ListRolloutGroups
GO

CREATE PROCEDURE ListRolloutGroups
	@rolloutgrp NVARCHAR(800)= N'%',	--Default: For all Groups
	@oudn VARCHAR(800) = '%',	--Default: For all OUs
	@ouMatch BIT=1			--Include sub-OUs? Default: 1
AS

	-- Retrieve Summary Data for ALL GROUPs
	CREATE TABLE #parentListTbl
	(	
		GroupID INT,
		GroupFullName NVARCHAR(800) COLLATE database_default
	)
	
	DECLARE grpList_cursor CURSOR FOR
		SELECT GroupID, ParentID FROM RolloutGroup
	
	DECLARE @grpID INT
	DECLARE @fullName NVARCHAR(800)
	DECLARE @parentID INT
	
	OPEN grpList_cursor
	FETCH NEXT FROM grpList_cursor INTO @grpID, @parentID
	WHILE @@FETCH_STATUS =0
	BEGIN
		--Retrieve parent name
		EXECUTE RolloutGroupParentName @grpID, @outFullName = @fullName OUTPUT 			
		INSERT INTO #parentListTbl VALUES ( @grpID, @fullName ) 	
		SET @fullName = NULL -- reset		
		FETCH NEXT FROM grpList_cursor INTO @grpID, @parentID
	END
	
	CLOSE grpList_cursor
	DEALLOCATE grpList_cursor
	
	CREATE TABLE #Organization(OUID int PRIMARY KEY)
	EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

	SELECT DISTINCT			
		grp.GroupID,
		parent.GroupFullName

	FROM	
		#parentListTbl AS parent, 
		RolloutGroup AS grp, 
		RolloutGroupMember AS mem,
		Computer AS c,
		#Organization AS s
	WHERE	
		parent.GroupID = grp.GroupID
		AND grp.GroupName LIKE @rolloutgrp 
		AND grp.GroupID = mem.GroupID
		AND mem.MemberID = c.ComputerID
		AND c.ComputerOUID = s.OUID

	DROP TABLE #parentListTbl
	
GO

CopyPermission 'ListRolloutGroups', 'mgsOSBreakdown'
GO
/* 
 * $Id: NetworkDevicesDiscoverySummary.sql,v 1.10 2007/07/24 06:30:56 pca Exp $
 * $Locker:  $
 *
 * DESCRIPTION: Stored Procedure: NetworkDevicesDiscoverySummary
 *	Input:
 * 	- @oudn: selected OU Name. - All OUs/ Specific OU  
 *	- @ouMatch: match on child OUs as well
 *	- @subnet: match on IP Subnet. Default = all subnets
 *
 *	Returns:
 *	- Total Count for Computer, Total
 *	- Count for Computers with Inventories, Inventory 
 *  - NOTE: Count for Computers without Inventories, NonInventory = Total - Inventory
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 *
 */
 
-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'NetworkDevicesDiscoverySummary' AND xtype = 'P'
)
        DROP PROCEDURE NetworkDevicesDiscoverySummary
GO


CREATE PROCEDURE NetworkDevicesDiscoverySummary
	@oudn VARCHAR(800)='%',
	@ouMatch BIT=1, --Values:[0=do exact OU Match | 1=include child-OUs in match]
	@subnet VARCHAR(128)='%'  --subnet (size=64*2 --to take into acct encoded chars)
AS
BEGIN

/*
 * COMMENT: This is required so that subnets with NULL values are picked up
 * when @subnet='%'; but not pick the NULL subnet when @subnet <> '%'
 */

--Boolean whether to do subnet match or not
DECLARE @allSubnetMatch BIT
IF @subnet = '%'
	SET @allSubnetMatch = 0 --FALSE
ELSE
	SET @allSubnetMatch = 1 --TRUE
	
CREATE TABLE #Organization(OUID int PRIMARY KEY)
EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

	SELECT  				
		COUNT(N.DeviceID) as Total,
		COUNT(DISTINCT I.ComputerID) as Inventory --Any Inventory HW or SW
		--NonInventory = Total - Inventory	(done by data manager)	
	
	FROM 
	NetworkDevice as N 
		LEFT OUTER JOIN 
		(
			Computer as C 
				LEFT OUTER JOIN InventoryReport as I
				ON (C.ComputerID = I.ComputerID AND I.UserID = 1)
		) 
		ON N.ComputerID  = C.ComputerID 
		INNER JOIN #Organization AS S
		ON S.OUID = C.ComputerOUID
		
	WHERE N.DuplicateID IS NULL 
		AND N.DeviceRole IN (0, 1, 2)
		AND 
		( 
			@allSubnetMatch = 0 --if expression is false, then do subnet match
			OR N.IPSubnet LIKE @subnet
		)
END
GO

CopyPermission 'NetworkDevicesDiscoverySummary', 'mgsOSBreakdown'
GO
/* 
 * $Id: NetworkDevicesDiscoverySummaryList.sql,v 1.11 2007/07/24 06:30:56 pca Exp $
 * $Locker:  $
 *
 * DESCRIPTION: Stored Procedure: NetworkDevicesDiscoverySummaryList
 *	Input:
 * 	- @oudn: selected OU Name. - All OUs/ Specific OU  
 *	- @ouMatch: match on child OUs as well
 *	- @subnet: match on IP Subnet. Default = all subnets
 *
 *	Returns:
 *	- OU, 
 *	- Discovery columns like DeviceName, IPAddress, PhysicalAddress, ...,
 *	- LastSoftwareReportDate, LastHardwareReportDate

 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 *
 */
 
-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'NetworkDevicesDiscoverySummaryList' AND xtype = 'P'
)
        DROP PROCEDURE NetworkDevicesDiscoverySummaryList
GO


CREATE PROCEDURE NetworkDevicesDiscoverySummaryList
	@oudn VARCHAR(800)='%',
	@ouMatch BIT=1, --Values:[0=do exact OU Match | 1=include child-OUs in match]
	@subnet VARCHAR(128)='%'  --subnet (size=64*2 --to take into acct encoded chars)
AS
BEGIN

/*
 * COMMENT: This is required so that subnets with NULL values are picked up
 * when @subnet='%'; but not pick the NULL subnet when @subnet <> '%'
 */

--Boolean whether to do subnet match or not
DECLARE @allSubnetMatch BIT
IF @subnet = '%'
	SET @allSubnetMatch = 0 --FALSE
ELSE
	SET @allSubnetMatch = 1 --TRUE
	
CREATE TABLE #Organization(OUID int PRIMARY KEY)
EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

	SELECT
		dbo.OUName(S.OUID) AS OU,
		C.ComputerID, C.ComputerCN,
		N.DeviceName, N.DNSFullName, N.NTDomainName,
		N.IPAddress, N.IPSubnet, N.IPSubnetMask,
		N.PhysicalAddress,
		DeviceRole =
			CASE N.DeviceRole				
				WHEN 0 THEN 'Computer'
				WHEN 1 THEN 'Workstation'
				WHEN 2 THEN 'Server'
				WHEN 3 THEN 'Printer'
				WHEN 4 THEN 'Switch'
				WHEN 5 THEN 'Router'
				WHEN 6 THEN 'Hub'
				ELSE '' -- Unknown; includes NULL
			END,  
		N.OperatingSystem,
		IsManaged = 
			CASE N.IsManaged
				WHEN 0 THEN 'no'
				WHEN 1 THEN 'yes'
				ELSE '' -- Unknown; includes NULL
			END,
		N.[Description], N.SystemDescription, N.SystemLocation,
		N.SystemContact, N.FirstDiscovered, N.LastUpdate,
		N.LastDataSourceType, I.SWDate, I.HWDate
	FROM NetworkDevice AS N
	LEFT OUTER JOIN 
	(
		Computer as C 
			LEFT OUTER JOIN InventoryReport as I
			ON (C.ComputerID = I.ComputerID AND I.UserID = 1)
	) 
	ON N.ComputerID = C.ComputerID 
	INNER JOIN #Organization as S
	ON S.OUID = C.ComputerOUID	
		
	WHERE N.DuplicateID IS NULL 
	  AND ( 
		--if expression is false, do subnet match
		@allSubnetMatch = 0 OR N.IPSubnet LIKE @subnet
	  )	
END
GO


CopyPermission 'NetworkDevicesDiscoverySummaryList', 'mgsOSBreakdown'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiHardwareAssessmentProcedures.sql,v 1.17 2008/10/29 00:58:01 pca Exp $
 *
 * DESCRIPTION:
 *
 *	This file contains all the related Hardware Assessment stored procedures.
 * 	- WiMiHardwareAssessmentCreateViewsOnRules
 * 	- WiMiDeleteOldRulesFromCacheNView
 * 	- WiMiUpdateHardwareAssessmentViews
 * 	- WiMiPopulateHardwareAssessmentCache
 * 	- WiMiGetComputerSetFromComputerNInventoryReportTbl
 * 	- WiMiGetComputerAssessmentForComputerSet
 * 	- WiMiValidateNUpdateAssessmentCache
 *	
 *	NOTE: Procedures should be added in the order specified 
 *	as they are dependent on other procedures existing in database. 
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

------------------------------------------------------------------
--STORED PROCEDURE: WiMiHardwareAssessmentCreateViewsOnRules
------------------------------------------------------------------

/*
 * DESCRIPTION:
 *
 *	Stored procedure WiMiHardwareAssessmentCreateViewsOnRules:  
 *	Creates views corresponding to Assessment Rules (all or
 *	only new / modified ones).
 *
 *	IMPORTANT NOTE: Only call this procedure with default	@oudn /
 *	@ouMatch filter as currently they are not stored anywhere.
 *	
 *	Input:
 *		@oudn, @ouMatch
 *		@getModifiedRulesOnly
 *
 *	Output:
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiHardwareAssessmentCreateViewsOnRules' AND xtype = 'P'
)
        DROP PROCEDURE WiMiHardwareAssessmentCreateViewsOnRules
GO

/*
 * COMMENT: in future, add parameter for AssessmentRuleSet
 * so as to do processing only for given AssessmentRuleSet
 */
CREATE PROCEDURE WiMiHardwareAssessmentCreateViewsOnRules
	@getModifiedRulesOnly BIT=0  --Modified Rules [0:false|1:true].
AS
BEGIN

--------------------------------------------------------------------------
--Variables set from CURSOR that reads from HardwareAssessmentRule table
--------------------------------------------------------------------------
DECLARE @viewname varchar(64)
DECLARE @ruleId int 
DECLARE @severityFailId int -- Fail
DECLARE @class varchar(256) 
DECLARE @property varchar(64) 
DECLARE @compexpr varchar(1024) 
DECLARE @sqltype varchar(20)
DECLARE @aggr varchar(10) 
DECLARE @msgFail varchar(256)
DECLARE @msgUnk varchar(256)
DECLARE @msgOk varchar(256)

-- LHS value of Comparision Expression
DECLARE @lhsexpr varchar(350) SET @lhsexpr = ''
-- Property Value used in Comparision Expression evaluation
DECLARE @value varchar(400) SET @value = ''
-- Group By Clause
DECLARE @grpbyClause varchar(200) set @grpbyClause = ''

DECLARE @severityUnkId int -- Unknown
SELECT @severityUnkId = AssessmentSeverityID 
	from HardwareAssessmentSeverity 
	where AssessmentSeverityMessage LIKE 'Not Inventoried'

DECLARE @severityOkId int -- Pass
SELECT @severityOkId = AssessmentSeverityID 
	from HardwareAssessmentSeverity 
	where AssessmentSeverityMessage LIKE 'Meets requirements'


BEGIN TRANSACTION

-----------------------------------------------------------------
--Cursor contains AssessmentRuleIDs only for modified/new rules 
--for which a new view has to be created.
--Note: Views for disabled rules are also created.
-----------------------------------------------------------------
DECLARE rules_cursor CURSOR FOR
	SELECT DISTINCT
		r.AssessmentRuleID, r.AssessmentSeverityID, r.Class, r.Property,
		r.ComparisonExpr, r.SQLType, r.AggregateFn, r.FailMessage, r.NotknownMessage, r.PassMessage		
	FROM HardwareAssessmentRule r 
		LEFT OUTER JOIN HardwareAssessmentCache w
		ON r.AssessmentRuleID = w.AssessmentRuleID 
	WHERE 
	(	
		@getModifiedRulesOnly = 0  -- if true, then get all Rules		
		OR w.AssessmentRuleID IS NULL -- new rules
		OR DATEDIFF(second, w.CacheEvaluatedDate, r.RuleModifiedDate) > 0 --modified rules
	) -- filter on new/ modified rules 

OPEN rules_cursor
FETCH NEXT FROM rules_cursor
	INTO	@ruleId, @severityFailId, @class, @property, 
		@compexpr, @sqltype, @aggr, @msgFail, @msgUnk, @msgOk
WHILE @@FETCH_STATUS = 0
BEGIN

	--------------------------------------------------------------------------
	--Creates a View for each rule for all Computers in a given OUDN/ OUMatch
	--------------------------------------------------------------------------
	SET @viewname = 'HardwareAssessmentView_onRule' + CAST(@ruleId AS VARCHAR(4))

	IF @sqltype IS NULL
	begin
		-- type of HardwareValue.Value
		SET @sqltype = 'VARCHAR(256)'
	end
	
	/* 
	 * NOTE: 
	 * We apply MAX when there is no aggregation applied to HardwareValue.Value.
	 * Therefore, we pick up only one record for a given Hardware Property irrespective
	 * whether its a string or number and then apply comparison expression on this value
	 * to determine Assessment Severity.
	 */
	 
	IF @aggr IS NULL
	begin
		-- type of Aggregate Function (default)
		SET @aggr = 'MAX' 
	end

	--LHS value of Comparision Expression	
	SET @lhsExpr = 'hv.Value' 
	-- Property Value used in Comparision Expression evaluation
	SET @value = 'hv.Value'

	IF @sqltype IS NOT NULL -- cast to SQLType
	begin
		SET @lhsExpr = 'CAST(' + @lhsExpr + ' AS ' + @sqltype + ')'	
	end
	IF @aggr IS NOT NULL -- apply Aggregate Function
	begin
		SET @lhsExpr = @aggr + '(' + @lhsExpr + ')'	
		SET @value = 'CAST(' + @lhsExpr + 'AS VARCHAR(256))'
	end

	-- Group By Clause 
	IF @aggr IS NOT NULL		
		SET @grpbyClause = 'GROUP BY c.ComputerID, c.ComputerOUID, ir.HWDate, hc.Class, hp.Property'

	-- escape ' as its used within dynamic SQL
	DECLARE @eMsgOk varchar(512)
	SET @eMsgOk = REPLACE(@msgOk, '''', '''''' )
	DECLARE @eMsgFail VARCHAR(512)
	SET @eMsgFail = REPLACE(@msgFail, '''', '''''' )
	DECLARE @eMsgUnk VARCHAR(512)
	SET @eMsgUnk = REPLACE(@msgUnk, '''', '''''' )
	
	-- Convert data type INT into VARCHAR when constructing SQL query
	-- NOTE: Required for SQL 7.0 (it complains at runtime)
	DECLARE @severityUnkId_var varchar(4)
	SET @severityUnkId_var = CAST(@severityUnkId as varchar(4))
	DECLARE @severityFailId_var varchar(4)
	SET @severityFailId_var = CAST(@severityFailId as varchar(4))
	DECLARE @severityOkId_var varchar(4)
	SET @severityOkId_var = CAST(@severityOkId as varchar(4))
	DECLARE @ruleId_var varchar(5)
	SET @ruleId_var = CAST(@ruleId as varchar(5))

	--------------------------------------------------------------------------
	--Create a View per Hardware Assessment Rule
	--------------------------------------------------------------------------
	
	IF EXISTS (
		SELECT * FROM dbo.sysobjects
		WHERE [name] = @viewname AND xtype = 'V'
	)
		EXEC('DROP VIEW [' + @viewname + ']')

	EXEC('
	CREATE VIEW ' + @viewname + ' 
	AS
	SELECT 
	c.ComputerID,
	c.ComputerOUID,
	ir.HWDate AS LastHardwareReportDate, 
	CASE
		WHEN ' + @lhsExpr + ' IS NULL THEN ' + @severityUnkId_var + ' 
		WHEN ' + @lhsExpr + ' ' + @compexpr + ' THEN ' + @severityFailId_var + ' 
		ELSE ' + @severityOkId_var + '
	END AS AssessmentSeverityId,
	CASE
		WHEN ' + @lhsExpr + ' IS NULL THEN ''' + @eMsgUnk + '''
		WHEN ' + @lhsExpr + ' ' + @compexpr + ' THEN ''' + @eMsgFail + ''' 	
		ELSE ''' + @eMsgOk + '''
	END AS AssessmentMessage,
	' + @ruleId_var + ' as AssessmentRuleId,
	hc.Class AS Class,
	hp.Property AS Property, 
	' + @value + ' as PropertyValue 
	FROM	Computer AS c
		LEFT OUTER JOIN InventoryReport AS ir
		ON (ir.ComputerID = c.ComputerID AND ir.UserID = 1)
		LEFT OUTER JOIN HardwareObject AS ho
			INNER JOIN HardwareClass AS hc
			ON (hc.HardwareClassID = ho.HardwareClassID AND hc.Class = ''' + @class + ''')
			INNER JOIN HardwareValue AS hv
			ON (hv.HardwareObjectID = ho.HardwareObjectID)
			INNER JOIN HardwareProperty AS hp
			ON (hp.HardwarePropertyID = hv.HardwarePropertyID AND hp.Property = ''' + @property + ''')
		ON ho.ComputerID = c.ComputerID
	' + @grpbyClause
	)
	
	EXEC CopyPermission @viewname, 'HardwareAssessmentRule'
	
	--Reset the Group By Clause 
	SET @grpbyClause = ''

	---------------------------------------------------------------------------
	FETCH NEXT FROM rules_cursor
		INTO	@ruleId, @severityFailId, @class, @property, 
			@compexpr, @sqltype, @aggr, @msgFail, @msgUnk, @msgOk
END

CLOSE rules_cursor
DEALLOCATE rules_cursor

COMMIT TRANSACTION

END
GO

EXEC CopyPermission 'WiMiHardwareAssessmentCreateViewsOnRules', 'mgsOSBreakdown'
GO

------------------------------------------------------------------
--STORED PROCEDURE: WiMiDeleteOldRulesFromCacheNView
------------------------------------------------------------------

/*
 * DESCRIPTION:
 *
 *	Stored procedure WiMiDeleteOldRulesFromCacheNView:  
 *	Deletes records from Cache table and Drops views corresponding 
 *	to deleted Assessment Rules
 *	
 *
 *	Input:
 *		@delCacheTbl, @delViews
 *
 *	Output:
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiDeleteOldRulesFromCacheNView' AND xtype = 'P'
)
        DROP PROCEDURE WiMiDeleteOldRulesFromCacheNView
GO
 
CREATE PROCEDURE WiMiDeleteOldRulesFromCacheNView
	@delCacheTbl BIT=1, -- Delete from Cache table {false:0| true:1}  
	@delViews BIT=1 -- Drop views {false:0| true:1}
AS
BEGIN

--------------------------------------------------------------
--Update Cache table by deleting rules that are disabled
--------------------------------------------------------------
DELETE HardwareAssessmentCache 
	FROM HardwareAssessmentCache w, HardwareAssessmentRule r
	WHERE w.AssessmentRuleID = r.AssessmentRuleID
		AND r.Enabled = 0

--------------------------------------------------------------
--Update Cache table by deleting rules that no longer exist
--------------------------------------------------------------
IF (@delCacheTbl = 1)
BEGIN
	DELETE HardwareAssessmentCache 
	FROM HardwareAssessmentCache w
	WHERE w.AssessmentRuleID NOT IN (
			SELECT AssessmentRuleID 
			FROM HardwareAssessmentRule 
	)
END  -- end of IF loop

--------------------------------------------------------------
--Update by Views corresponding to rules that no longer exist
--------------------------------------------------------------
IF (@delViews = 1)
BEGIN
	--Read from System table the name of views
	--SELECT * FROM dbo.sysobjects
	--WHERE [name] LIKE 'HardwareAssessmentView_onRule%' AND xtype = 'V'
	
	CREATE TABLE #ViewName ( ViewName VARCHAR(64) COLLATE database_default )

	DECLARE @viewName VARCHAR(64)
	DECLARE rulesID_cursor CURSOR FOR
		SELECT AssessmentRuleID FROM HardwareAssessmentRule

	OPEN rulesID_cursor
	DECLARE @rulesID  INT
	FETCH NEXT FROM rulesID_cursor INTO @rulesID
	WHILE @@FETCH_STATUS = 0
	BEGIN		
		SET @viewName = 'HardwareAssessmentView_onRule' + CAST(@rulesID as Varchar(4))		
		INSERT INTO #ViewName VALUES (@viewName)		
		FETCH NEXT FROM rulesID_cursor INTO @rulesID
	END
	CLOSE rulesID_cursor
	DEALLOCATE rulesID_cursor

	DECLARE oldrules_cursor CURSOR FOR
		SELECT  [name]
		FROM dbo.sysobjects 
			WHERE [name] LIKE 'HardwareAssessmentView_onRule%' 
			AND xtype = 'V' 	
			AND [name] NOT IN ( Select viewname from #ViewName )

	OPEN oldrules_cursor
	FETCH NEXT FROM oldrules_cursor INTO @viewname
	WHILE @@FETCH_STATUS = 0
	BEGIN
		IF EXISTS (
			SELECT * FROM dbo.sysobjects
			WHERE [name] = @viewname AND xtype = 'V'
		)
			EXEC ('DROP VIEW ' +  @viewname )	

		FETCH NEXT FROM oldrules_cursor INTO @viewname
	END
	CLOSE oldrules_cursor
	DEALLOCATE oldrules_cursor
	
	DROP TABLE #ViewName

END -- end of IF loop

END 
GO

EXEC CopyPermission 'WiMiDeleteOldRulesFromCacheNView', 'mgsOSBreakdown'
GO


------------------------------------------------------------------
--STORED PROCEDURE: WiMiUpdateHardwareAssessmentViews
------------------------------------------------------------------

/*
 * DESCRIPTION:
 *
 *	Stored procedure WiMiUpdateHardwareAssessmentViews:  
 *	Updates Hardware Assessment Views by deleting out dated ones
 *	and creating views for new and modified Assessment Rules.
 *	
 *	Input:
 *		@getModifiedRulesOnly
 *
 *	Output:
 *
 *	Usage:
 *	EXEC 	WiMiUpdateHardwareAssessmentViews 1
 *
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiUpdateHardwareAssessmentViews' AND xtype = 'P'
)
        DROP PROCEDURE WiMiUpdateHardwareAssessmentViews
GO

/*
 * COMMENT: in future, add parameter for AssessmentRuleSet
 * so as to do processing only for given AssessmentRuleSet
 */
 
CREATE PROCEDURE WiMiUpdateHardwareAssessmentViews
	@getModifiedRulesOnly BIT=0 --Create views for New/Modified rules only. Default: all rules.
AS
BEGIN

BEGIN TRANSACTION
	--Drop Views for deleted Assessment Rules
	EXEC WiMiDeleteOldRulesFromCacheNView @delCacheTbl=0 , @delViews=1 
COMMIT TRANSACTION

BEGIN TRANSACTION
	--Create views for new/modified Assessment Rules
	--Note: Currently, we create view on ALL OUs.	
	EXECUTE WiMiHardwareAssessmentCreateViewsOnRules 
		@getModifiedRulesOnly --Get new/modified Rules
COMMIT TRANSACTION

END
GO

EXEC CopyPermission 'WiMiUpdateHardwareAssessmentViews', 'mgsOSBreakdown'
GO


------------------------------------------------------------------
--STORED PROCEDURE: WiMiPopulateHardwareAssessmentCache
------------------------------------------------------------------

/*
 * DESCRIPTION:
 *
 *	Stored procedure WiMiPopulateHardwareAssessmentCache:  
 *	Populates Cache table. Optionally, creates views for all or only
 *	new and modified Assessment Rules.
 *	
 *	Input:
 *		@oudn, @ouMatch
 *		@recreateViews, @getModifiedRulesOnly
 *
 *	Output:
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiPopulateHardwareAssessmentCache' AND xtype = 'P'
)
        DROP PROCEDURE WiMiPopulateHardwareAssessmentCache
GO

/*
 * COMMENT: in future, add parameter for AssessmentRuleSet
 * so as to do processing only for given AssessmentRuleSet
 */

CREATE PROCEDURE WiMiPopulateHardwareAssessmentCache
	@recreateView BIT=0, -- Re-create views = {0:False | 1:True}. Default: false 
	@getModifiedRulesOnly BIT=0 --New/Modified rules. Default: all rules; no new/modified.
AS
BEGIN

-----------------------------------------------------------------
--Create a new view(s) based on lastest rules.
--Optionally, only modified/new rule's corresponding view is re-created
-----------------------------------------------------------------

/*
 * REVISIT: User does not have rights to Create and Drop Views.
 * Currently, this updated separately.
 *
 * IF (@recreateView = 1)
 * BEGIN
 *	--Currently, we create view on ALL OUs.	
 *	EXECUTE WiMiHardwareAssessmentCreateViewsOnRules 
 *		'%', --Do all OUs
 *		1, --Include child OUs
 *		@getModifiedRulesOnly --Get new/modified Rules
 * END
 *
 */

--else... use the existing Views

-----------------------------------------------------------------
--Populate with fresh set of recordsets
-----------------------------------------------------------------
BEGIN TRANSACTION

DECLARE @viewName Varchar(64)
--Insert/Update SQL string contains name of view
DECLARE @SQLstmt nvarchar(4000)

DECLARE rulesID_cursor CURSOR FOR
	SELECT DISTINCT r.AssessmentRuleID --, w.AssessmentRuleID
	FROM HardwareAssessmentRule r 
		LEFT OUTER JOIN HardwareAssessmentCache w
		ON r.AssessmentRuleID = w.AssessmentRuleID 
	WHERE  
	(	
		@getModifiedRulesOnly = 0  -- if true, then get all Rules		
		OR w.AssessmentRuleID IS NULL -- new rules
		OR DATEDIFF(second, w.CacheEvaluatedDate, r.RuleModifiedDate) > 0 --modified rules
	) -- filter on new/ modified rules
	AND r.Enabled = 1 -- filter on Enabled rules

OPEN rulesID_cursor

DECLARE @rulesID  int
FETCH NEXT FROM rulesID_cursor INTO	@rulesID
WHILE @@FETCH_STATUS = 0
BEGIN
	SET @viewName = 'HardwareAssessmentView_onRule' + CAST(@rulesID as Varchar(4))
	
	----------------------------------------------------------
	-- Set Assessment Rule exists in Cache Flag for given OUDN
	-- AND Determine OU condition
	----------------------------------------------------------	
	DECLARE @ruleExists BIT SET @ruleExists = 0 --initialise

		--Check if Assessment Rule exists in cache for given OUDN		
		SELECT @ruleExists = 
			CASE 
				WHEN w.AssessmentRuleID IS NOT NULL THEN 1
			END
			FROM 
				Computer c, 
				#Organization s,
				HardwareAssessmentCache w
			WHERE
				c.ComputerOUID = s.OUID --driven by Computer table for correctness
				AND c.ComputerID = w.ComputerID 
				AND AssessmentRuleID = @rulesID

	----------------------------------------------------------
	--Update existing records when Rule exists (though may be modified)
	----------------------------------------------------------
	IF (@ruleExists = 1)
	BEGIN
		SET @SQLstmt = 
			N'UPDATE HardwareAssessmentCache
			SET 		
				ComputerOUID = tmp.ComputerOUID,
				LastHardwareReportDate = tmp.LastHardwareReportDate,
				AssessmentSeverityID = tmp.AssessmentSeverityID,
				CacheEvaluatedDate = GetDate()
			FROM
			(
				SELECT 
					v.ComputerID, 
					v.ComputerOUID,
					v.LastHardwareReportDate,
					v.AssessmentRuleID,
					v.AssessmentSeverityID
				FROM ' + @viewName  + ' v, #Organization s
				WHERE v.ComputerOUID = s.OUID 
			) AS tmp, HardwareAssessmentCache AS w	
			WHERE w.ComputerID = tmp.ComputerID
			AND w.AssessmentRuleID = tmp.AssessmentRuleID'
	END

	----------------------------------------------------------
	--Insert records either when Rule does not exist OR when
	--Views and Rules are up to date but data in Cache table is
	--out of date for given @oudn/@ouMatch values.
	----------------------------------------------------------
	
	DECLARE @cacheOutOfDate BIT
	--When Views are up to date and require to update cache for all Rules
	IF (@recreateView = 0 AND @getModifiedRulesOnly = 0)
		SET @cacheOutOfDate = 1 -- true
	ELSE
		SET @cacheOutOfDate = 0 -- false		

	--IF (@ruleExists = 0) OR (@recreateView = 0 AND @getModifiedRulesOnly = 0)	
	IF (@ruleExists = 0) OR (@cacheOutOfDate = 1)
	BEGIN	
		SET @SQLstmt = 
			N'INSERT INTO HardwareAssessmentCache
			(
				ComputerID, 
				ComputerOUID, 
				LastHardwareReportDate, 
				AssessmentRuleID,
				AssessmentSeverityID, 
				CacheEvaluatedDate
			)
			SELECT 
				v.ComputerID, 
				v.ComputerOUID,
				v.LastHardwareReportDate,
				v.AssessmentRuleID,
				v.AssessmentSeverityID,
				GetDate() as CacheEvalutedDate 
			FROM ' + @viewName  + ' v, #Organization s
			WHERE v.ComputerOUID = s.OUID'
	END

	EXEC sp_executesql @SQLstmt

	FETCH NEXT FROM rulesID_cursor INTO	@rulesID
END
CLOSE rulesID_cursor
DEALLOCATE rulesID_cursor

COMMIT TRANSACTION

END
GO

EXEC CopyPermission 'WiMiPopulateHardwareAssessmentCache', 'mgsOSBreakdown'
GO


------------------------------------------------------------------
--STORED PROCEDURE: WiMiGetComputerSetFromComputerNInventoryReportTbl
------------------------------------------------------------------

/*
 * DESCRIPTION:
 *
 *	Stored procedure WiMiGetComputerSetFromComputerNInventoryReportTbl:  
 *	Gets set of computers that have been added to Computer table
 *	and those with new inventory since cache was last evaluated.
 *	
 *	Input:
 *		@oudn, @ouMatch		
 *
 *	Output:
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiGetComputerSetFromComputerNInventoryReportTbl' AND xtype = 'P'
)
        DROP PROCEDURE WiMiGetComputerSetFromComputerNInventoryReportTbl
GO

CREATE PROCEDURE WiMiGetComputerSetFromComputerNInventoryReportTbl
AS
BEGIN
	--Add recordsets for new ComputerIDs or new Inventories
	INSERT INTO #ComputerSet
	SELECT DISTINCT
	c.ComputerID,
	ExistingComputer =
		CASE
			WHEN w.ComputerID IS NULL THEN 0 -- Add new computer
			ELSE 1 --Update existing Computer
		END 
	FROM 
	#Organization s INNER JOIN Computer c 
	ON s.OUID = c.ComputerOUID 
		LEFT OUTER JOIN InventoryReport ir
		ON 
		(
			c.ComputerID = ir.ComputerID 
			AND ir.UserID = 1 --HW Inventory only
		)
			LEFT OUTER JOIN HardwareAssessmentCache w	
			ON
			(
				c.ComputerOUID = w.ComputerOUID 
				AND c.ComputerID = w.ComputerID 
			)
	WHERE 
		(
			w.ComputerID IS NULL 
			OR ir.ComputerID IS NULL
			OR DATEDIFF(second, w.CacheEvaluatedDate, ir.HWDate ) > 0
		) --record not in Cache or Inventory or new HW Inventory
END -- end of BEGIN loop
GO

EXEC CopyPermission 'WiMiGetComputerSetFromComputerNInventoryReportTbl', 'mgsOSBreakdown'
GO

------------------------------------------------------------------
--STORED PROCEDURE: WiMiGetComputerAssessmentForComputerSet
------------------------------------------------------------------

/*
 * DESCRIPTION:
 *
 *	Stored procedure WiMiGetComputerAssessmentForComputerSet:  
 *	Gets assessment against each Assessment Rule for a given set of computers. 
 *	
 *	Input:
 *		@tblname		
 *
 *	Output:
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiGetComputerAssessmentForComputerSet' AND xtype = 'P'
)
        DROP PROCEDURE WiMiGetComputerAssessmentForComputerSet
GO

CREATE PROCEDURE WiMiGetComputerAssessmentForComputerSet
	@tblname varchar(64)='#ComputerAssessmentResults'
AS
BEGIN

--------------------------------------------------------------------------
--Variables set from CURSOR that reads from HardwareAssessmentRule table
--------------------------------------------------------------------------
DECLARE @ruleId int 
DECLARE @severityFailId int 
DECLARE @class varchar(256) 
DECLARE @property varchar(64) 
DECLARE @compexpr varchar(1024) 
DECLARE @sqltype varchar(20)
DECLARE @aggr varchar(10) 

--LHS value of Comparision Expression
DECLARE @lhsexpr varchar(350) SET @lhsexpr = ''
-- Property Value used in Comparision Expression evaluation
DECLARE @value varchar(400) SET @value = ''
-- Group By Clause
DECLARE @grpbyClause varchar(200) set @grpbyClause = ''

DECLARE @severityUnkId int -- Unknown
SELECT @severityUnkId = AssessmentSeverityID 
	from HardwareAssessmentSeverity 
	where AssessmentSeverityMessage LIKE 'Not Inventoried'

DECLARE @severityOkId int -- Pass
SELECT @severityOkId = AssessmentSeverityID 
	from HardwareAssessmentSeverity 
	where AssessmentSeverityMessage LIKE 'Meets requirements'

DECLARE rules_cursor CURSOR FOR
	SELECT	AssessmentRuleID, AssessmentSeverityID, Class, Property,
		ComparisonExpr, SQLType, AggregateFn		
	FROM HardwareAssessmentRule

OPEN rules_cursor
FETCH NEXT FROM rules_cursor
	INTO	@ruleId, @severityFailId, @class, @property, 
		@compexpr, @sqltype, @aggr
WHILE @@FETCH_STATUS = 0
BEGIN

	--------------------------------------------------------------------------
	--Inserts into TABLE for each rule for all Computers in a given OUDN/ OUMatch
	--------------------------------------------------------------------------
	IF @sqltype IS NULL
		-- type of HardwareProperty.Value
		set @sqltype = 'VARCHAR(256)' 

	--LHS value of Comparision Expression	
	SET @lhsExpr = 'hv.Value' 
	-- Property Value used in Comparision Expression evaluation
	SET @value = 'hv.Value'

	IF @sqltype IS NOT NULL -- cast to SQLType
	begin
		SET @lhsExpr = 'CAST(' + @lhsExpr + ' AS ' + @sqltype + ')'	
	end
	IF @aggr IS NOT NULL -- apply Aggregate Function
	begin
		SET @lhsExpr = @aggr + '(' + @lhsExpr + ')'	
		SET @value = 'CAST(' + @lhsExpr + 'AS VARCHAR(256))'
	end

	-- Group By Clause
	IF @aggr IS NOT NULL
		SET @grpbyClause = 'GROUP BY c.ComputerID, c.ComputerOUID, ir.HWDate'
		
	-- Convert data type INT into VARCHAR when constructing SQL query
	-- NOTE: Required for SQL 7.0 (it complains at runtime)
	DECLARE @severityUnkId_var varchar(4)
	SET @severityUnkId_var = CAST(@severityUnkId as varchar(4))
	DECLARE @severityFailId_var varchar(4)
	SET @severityFailId_var = CAST(@severityFailId as varchar(4))
	DECLARE @severityOkId_var varchar(4)
	SET @severityOkId_var = CAST(@severityOkId as varchar(4))
	DECLARE @ruleId_var varchar(5)
	SET @ruleId_var = CAST(@ruleId as varchar(5))
	
	--------------------------------------------------------------------------
	--Inserts into TABLE per Hardware Assessment Rule
	--------------------------------------------------------------------------

	--print ('
	EXEC('
	INSERT INTO ' + @tblname + ' 
	(
		ComputerID,
		ComputerOUID,
		LastHardwareReportDate,
		AssessmentSeverityId,
		AssessmentRuleId
	)
	SELECT 
	c.ComputerID,
	c.ComputerOUID,
	ir.HWDate AS LastHardwareReportDate, 
	CASE
		when ' + @lhsExpr + ' IS NULL then ' + @severityUnkId_var + ' 
		when ' + @lhsExpr + ' ' + @compexpr + ' then ' + @severityFailId_var + ' 
		else ' + @severityOkId_var + '
	END as AssessmentSeverityId,
	' + @ruleId_var + ' as AssessmentRuleId
	FROM	Computer AS c
		LEFT OUTER JOIN InventoryReport AS ir
		ON (ir.ComputerID = c.ComputerID AND ir.UserID = 1)
		LEFT OUTER JOIN HardwareObject AS ho
			INNER JOIN HardwareClass AS hc
			ON (hc.HardwareClassID = ho.HardwareClassID AND hc.Class = ''' + @class + ''')
			INNER JOIN HardwareValue AS hv
			ON (hv.HardwareObjectID = ho.HardwareObjectID)
			INNER JOIN HardwareProperty AS hp
			ON (hp.HardwarePropertyID = hv.HardwarePropertyID AND hp.Property = ''' + @property + ''')
		ON ho.ComputerID = c.ComputerID
	WHERE (
		c.ComputerID IN (Select ComputerID from #ComputerSet)  
		)		
	' + @grpbyClause
	)	
	
	--Reset the Group By Clause
	set @grpbyClause = ''
	---------------------------------------------------------------------------

	FETCH NEXT FROM rules_cursor
		INTO	@ruleId, @severityFailId, @class, @property, 
			@compexpr, @sqltype, @aggr
END

CLOSE rules_cursor
DEALLOCATE rules_cursor

END
GO

EXEC CopyPermission 'WiMiGetComputerAssessmentForComputerSet', 'mgsOSBreakdown'
GO


------------------------------------------------------------------
--STORED PROCEDURE: WiMiValidateNUpdateAssessmentCache
------------------------------------------------------------------

/*
 * DESCRIPTION:
 *
 *	Stored procedure WiMiValidateNUpdateAssessmentCache:  
 *	Validates and updates HardwareAssessmentCache table, which 
 *	in turn may re-create views for Assessment Rules. This is called
 *	from 	2 high level Summary Hardware Assessment stored procedures
 *
 *	Input:
 *		@oudn, @ouMatch
 *
 *	Output:
 *
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiValidateNUpdateAssessmentCache' AND xtype = 'P'
)
        DROP PROCEDURE WiMiValidateNUpdateAssessmentCache
GO

CREATE PROCEDURE WiMiValidateNUpdateAssessmentCache
	@oudn Varchar(800)='%',
	@ouMatch BIT=1
AS
BEGIN

/* 
 ------------------------------------------------------------
 1. Check if HardwareAssessmentRule have been modified since
 Cache was last evalutated
 ------------------------------------------------------------
*/ 

-------------------------------------------------------------------
--i. Delete recordsets from Cache table and Drop Views corresponding
-- to deleted Assessment Rules
-------------------------------------------------------------------
-- REVISIT: User does not have rights to Create and Drop Views
-- EXEC WiMiDeleteOldRulesFromCacheNView @delCacheTbl=1 , @delViews=1 

--Delete from Cache but use existing views
EXEC WiMiDeleteOldRulesFromCacheNView @delCacheTbl=1 , @delViews=0 

-------------------------------------------------------------------
--ii. When Assessment rules have been modified / added
-------------------------------------------------------------------
DECLARE @return BIT --Flag to check if to return out of Proc

BEGIN TRANSACTION

DECLARE @oldestCacheDate DATETIME
SELECT @oldestCacheDate = MIN(CacheEvaluatedDate) FROM HardwareAssessmentCache

DECLARE @latestRuleDate DATETIME
SELECT @latestRuleDate = MAX(RuleModifiedDate) FROM HardwareAssessmentRule WHERE Enabled = 1

CREATE TABLE #Organization(OUID int PRIMARY KEY)
EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

--If Cache table is empty; @oldestCacheDate will be NULL
IF (DATEDIFF(second, @oldestCacheDate, @latestRuleDate) IS NULL )
BEGIN 

/*
 * NOTE: Called procedure currently does not recreate Views;
 * however, other logic for insert vs update into Cache table 
 * needs to know whether it was required or not.
 */
	EXECUTE WiMiPopulateHardwareAssessmentCache 
 		1, -- recreate views
 		0 -- recreate views & populate for all rules

	SET @return = 1 -- Set flag to TRUE
END

--ELSE ....Carry on processing

--If Assessment Rules have been modified, re-do Cache
ELSE IF (DATEDIFF(second, @oldestCacheDate, @latestRuleDate) > 0)
BEGIN
		
	/*
	* NOTE: Called procedure currently does not recreate Views;
	* however, other logic for insert vs update into Cache table 
	* needs to know whether it was required or not.
	*/
		
	/* 
	 * REVISIT 30-September-2002: 
	 * Not executing below procedure for "only recreate views & 
	 * populate for new/modified rules" i.e. @getModifiedRulesOnly=1
	 * as this does not work for scenario when
	 * - a specific is OU selected (not incl. child OUs)
	 * - OU was empty before and now computer(s) have moved into it
	 * - browsed to this OU after going to its child OU	 
 	 */
 	 
	--Deletes old View & creates new one AND populates Cache from View
	--for new/modified Assessment Rules
	EXECUTE WiMiPopulateHardwareAssessmentCache 
 		1, -- recreate views
 		0 -- recreate views & populate for ALL rules
 	
 		
 	SET @return = 0 -- Set flag to FALSE

END 

COMMIT TRANSACTION

IF (@return =1)
	RETURN --No more processing required.
	
--ELSE ....Carry on processing

/* 
 ------------------------------------------------------------
 2. Check if Cache table contains entries for OUDN
 ------------------------------------------------------------
*/ 

BEGIN TRANSACTION

DECLARE @rowCount INT	
SET @rowCount = -1 --Initialize

	-------------------------------------------------------------------
	--i. Update OUIDs for existing records in Cache for computers that
	-- have moved into OU
	-------------------------------------------------------------------
	UPDATE HardwareAssessmentCache
		SET 		
			ComputerOUID = c.ComputerOUID
		FROM 
		#Organization s, 
		Computer c, 
		HardwareAssessmentCache w
		WHERE s.OUID = c.ComputerOUID -- driven by Computer table for correctness 
			AND c.ComputerID = w.ComputerID
			AND c.ComputerOUID <> w.ComputerOUID
			
	-------------------------------------------------------------------
	--ii. Delete recordsets from Cache for Computers that do not exist 
	--in given OUDN anymore (either moved out of or deleted from)
	-------------------------------------------------------------------
	DELETE HardwareAssessmentCache 
		FROM HardwareAssessmentCache w, #Organization s
		WHERE w.ComputerOUID = s.OUID --driven by Cache table
			AND w.ComputerID NOT IN (
				SELECT c.ComputerID FROM Computer c
				WHERE c.ComputerOUID = w.ComputerOUID
			)
			
	-------------------------------------------------------------------
	--iii. Now determine the row count in Cache table for given OUDN
	-------------------------------------------------------------------
	SELECT @rowCount=COUNT(w.ComputerID) FROM HardwareAssessmentCache w, #Organization s
		WHERE w.ComputerOUID = s.OUID

--Cache does not contain any recordset for given OUDN
--then, populate Cache for given OUDN
IF (@rowCount < 1)
BEGIN
	--Populate Cache from View. Does not re-create view(s)
	EXECUTE WiMiPopulateHardwareAssessmentCache 
		0, -- (default) do not re-create views
		0 --(default) no new/modified rules 

END -- end of IF loop
--Cache contains recordsets for given OUDN
--then, populate/update Cache for a set of Computers
ELSE 
BEGIN
	-----------------------------------------------------------------
	--i. Create TABLE to store a set of ComputerIDs
	-----------------------------------------------------------------
	CREATE TABLE #ComputerSet
	(
		ComputerID INT, 
		ExistingComputer BIT -- Values {0:Add new | 1: Update Existing}
	)

	-----------------------------------------------------------------
	--ii. Check if Cache table contains all Computers for given OU
	--and if computer records been evaluated with lastest Inventory.
	--Then determine computer set & insert into #ComputerSet table
	-----------------------------------------------------------------	
	EXECUTE  WiMiGetComputerSetFromComputerNInventoryReportTbl

	--------------------------------------------------------------------
	--iii. Create #ComputerAssessmentResults table based on
	--a set of computers and Populate with assessment
	--------------------------------------------------------------------
	CREATE TABLE #ComputerAssessmentResults
	(
		ComputerID INT,
		ComputerOUID INT,
		LastHardwareReportDate DATETIME,
		AssessmentSeverityId INT,
		AssessmentRuleId INT
	)
	
	EXECUTE WiMiGetComputerAssessmentForComputerSet
		'#ComputerAssessmentResults' --table name
	
	-------------------------------------------------------------------
	--iv. Update *existing* records in Cache table 
	-------------------------------------------------------------------	

	UPDATE HardwareAssessmentCache
	SET 		
		ComputerOUID = tmp.ComputerOUID,
		LastHardwareReportDate = tmp.LastHardwareReportDate,
		AssessmentSeverityID = tmp.AssessmentSeverityID,
		CacheEvaluatedDate = GetDate()
	FROM
	(
		SELECT 
			ComputerID, 
			ComputerOUID,
			LastHardwareReportDate,
			AssessmentRuleID,
			AssessmentSeverityID
		FROM #ComputerAssessmentResults
	) AS tmp, HardwareAssessmentCache AS w
	WHERE w.ComputerID = tmp.ComputerID
	AND w.AssessmentRuleID = tmp.AssessmentRuleID

	-------------------------------------------------------------------
	--v. Insert *new* records into Cache table 
	-------------------------------------------------------------------	
	INSERT INTO HardwareAssessmentCache
	(
		ComputerID, 
		ComputerOUID, 
		LastHardwareReportDate, 
		AssessmentRuleID,
		AssessmentSeverityID, 
		CacheEvaluatedDate
	)
	SELECT 
		t.ComputerID, 
		t.ComputerOUID,
		t.LastHardwareReportDate,		
		t.AssessmentRuleID,
		t.AssessmentSeverityID,
		GetDate() AS CacheEvaluatedDate
	FROM #ComputerAssessmentResults t, #ComputerSet cs
	WHERE 
		t.ComputerID = cs.ComputerID 
		AND cs.ExistingComputer = 0 -- computer does not exist in Cache

	--------------------------------------------------------------------
	--vi. Drop temporary tables
	--------------------------------------------------------------------
	DROP TABLE #ComputerAssessmentResults
	DROP TABLE #ComputerSet
	
END -- end of ELSE loop

COMMIT TRANSACTION

END -- end of BEGIN loop

GO

EXEC CopyPermission 'WiMiValidateNUpdateAssessmentCache', 'mgsOSBreakdown'

/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiHardwareDisposition.sql,v 1.7 2007/07/24 06:30:56 pca Exp $
 *
 * DESCRIPTION:
 *
 *	Stored procedure WiMiHardwareDisposition: Obtain hardware
 *	disposition information for computers.
 *
 *	Input:
 *		@oudn, @ouMatch, @compID, @disposition
 *
 *	Output:
 *		ComputerID, ComputerName, OUDN,
 *		LastHardwareReportDate, Disposition
 *
 *		Disposition is one of 'Not inventoried', 'Requires
 *		upgrade', 'Requires replacement' or 'Meets
 *		requirements'
 *
 * TO THINK ABOUT: When inventory has been reported, but a property
 * value is NULL (i.e. the property was not reported in the
 * inventory), what should we return? Currently "Not inventoried" is
 * returned. This is typically displayed as "missing inventory
 * information" to users.
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiHardwareDisposition' AND xtype = 'P'
)
        DROP PROCEDURE WiMiHardwareDisposition
GO

CREATE PROCEDURE [WiMiHardwareDisposition]
	@oudn VARCHAR(800) = '%',	-- Default: For all OUs
	@ouMatch BIT = 1,			-- Include sub-OUs. Default: 1 i.e. True
	@disposition VARCHAR(20) = NULL,	-- Disposition to filter on	
	@groupID INT, 			--filter on rollout group
	@grpfilter INT=0 			-- do not filter
AS
BEGIN

-- Validates and Updates HardwareAssessmentCache table
EXEC WiMiValidateNUpdateAssessmentCache @oudn, @ouMatch

IF @disposition IS NULL
	SET @disposition = '%'
	
CREATE TABLE #Organization(OUID int PRIMARY KEY)
EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

	IF @grpfilter <> 0 -- do filter
		SELECT 
			tmp.ComputerID,
			tmp.ComputerName,			
			tmp.OUDN,
			tmp.OldName,
			tmp.NewName,
			tmp.LastHardwareReportDate,
			sev.AssessmentSeverityMessage as Disposition
		FROM
		(
			SELECT 
			cache.ComputerID,
			c.ComputerCN as ComputerName, -- current name
			dbo.OUName(s.OUID) as OUDN,
			CASE 
				WHEN d1.DN IS NULL THEN map.[name]
				WHEN RIGHT(dbo.OUName(s.OUID), LEN(d1.DN)) = d1.DN THEN map.[name]
				ELSE map.[name] + ' (' + d1.DN + ')'	
			END as OldName,
			CASE 
				WHEN d2.DN IS NULL THEN map.newName
				WHEN RIGHT(dbo.OUName(s.OUID), LEN(d2.DN)) = d2.DN THEN map.newName
				ELSE map.newName + ' (' + d2.DN + ')'	
			END as NewName,
			MAX(cache.LastHardwareReportDate) as LastHardwareReportDate,			
			MAX(sev.AssessmentSeverityNumber) as AssessmentSeverityNumber
			FROM 	
			(#Organization AS s INNER JOIN HardwareAssessmentCache AS cache
			 ON (cache.ComputerOUID = s.OUID)	 
				INNER JOIN HardwareAssessmentSeverity AS sev
				ON cache.AssessmentSeverityID = sev.AssessmentSeverityID
			)	
			INNER JOIN (
				Computer AS c 
					LEFT OUTER JOIN (
					Namemap as map 
						INNER JOIN Domain as d1
						ON map.Domain = d1.OrganizationID
						INNER JOIN Domain as d2
						ON map.newDomain = d2.OrganizationID
					)
					ON c.ComputerID = map.currentID
				) -- name-mapping for pre and post rollout computer names
				LEFT OUTER JOIN ( 
					RolloutGroup as grp INNER JOIN RolloutGroupMember as mem
					ON grp.GroupID = mem.GroupID
				)
				ON c.ComputerID = mem.MemberID
			ON cache.ComputerID = c.ComputerID
			WHERE grp.GroupID = @groupID							
			GROUP BY cache.ComputerID, c.ComputerCN, dbo.OUName(s.OUID), map.[name], d1.DN, map.newName, d2.DN, s.OUID
		) AS tmp, HardwareAssessmentSeverity AS sev
		WHERE tmp.AssessmentSeverityNumber = sev.AssessmentSeverityNumber
			AND sev.AssessmentSeverityMessage LIKE @disposition
	ELSE
		SELECT 
			tmp.ComputerID,
			tmp.ComputerName,			
			tmp.OUDN,
			tmp.OldName,
			tmp.NewName,
			tmp.LastHardwareReportDate,
			sev.AssessmentSeverityMessage as Disposition
		FROM
		(
			SELECT 
			cache.ComputerID,
			c.ComputerCN as ComputerName, -- current name			
			dbo.OUName(s.OUID) as OUDN,
			CASE 
				WHEN d1.DN IS NULL THEN map.[name]
				WHEN RIGHT(dbo.OUName(s.OUID), LEN(d1.DN)) = d1.DN THEN map.[name]
				ELSE map.[name] + ' (' + d1.DN + ')'	
			END as OldName,
			CASE 
				WHEN d2.DN IS NULL THEN map.newName
				WHEN RIGHT(dbo.OUName(s.OUID), LEN(d2.DN)) = d2.DN THEN map.newName
				ELSE map.newName + ' (' + d2.DN + ')'	
			END as NewName,
			MAX(cache.LastHardwareReportDate) as LastHardwareReportDate,
			MAX(sev.AssessmentSeverityNumber) as AssessmentSeverityNumber
			FROM 	
			(#Organization AS s INNER JOIN HardwareAssessmentCache AS cache
			ON (cache.ComputerOUID = s.OUID)	 
				INNER JOIN HardwareAssessmentSeverity AS sev
				ON cache.AssessmentSeverityID = sev.AssessmentSeverityID
			)	
			INNER JOIN 
			(Computer AS c 
				LEFT OUTER JOIN 
				(Namemap as map
					INNER JOIN Domain as d1
					ON map.Domain = d1.OrganizationID
					INNER JOIN Domain as d2
					ON map.newDomain = d2.OrganizationID
				)
				ON c.ComputerID = map.currentID
			) -- name-mapping for pre and post rollout computer names		
			ON cache.ComputerID = c.ComputerID
		GROUP BY cache.ComputerID, c.ComputerCN, dbo.OUName(s.OUID), map.[name], d1.DN, map.newName, d2.DN, s.OUID
		) AS tmp, HardwareAssessmentSeverity AS sev
		WHERE tmp.AssessmentSeverityNumber = sev.AssessmentSeverityNumber
			AND sev.AssessmentSeverityMessage LIKE @disposition
END
GO

EXEC CopyPermission 'WiMiHardwareDisposition', 'mgsOSBreakdown'
GO

/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiHardwareDispositionDetails.sql,v 1.7 2008/10/29 00:58:01 pca Exp $
 *
 * DESCRIPTION:
 *
 *	Stored procuedre WiMiHardwareDispositionDetails: Obtain detailed
 *	hardware disposition information for a specified computer.
 *
 *	Input:
 *		@compID
 *
 *	Output:
 *		ComputerID, ComputerCN, OperatingSystem,
 *		LastHardwareReportDate, Disposition, OUDN,
 *		Value*, Message*, Disposition*
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiHardwareDispositionDetails' AND xtype = 'P'
)
        DROP PROCEDURE WiMiHardwareDispositionDetails
GO

CREATE PROCEDURE [WiMiHardwareDispositionDetails]
	@compID INT	-- ID of computer to get disposition information for
AS
BEGIN
-----------------------------------
--Check if Views are upto date
-----------------------------------

/*
 * REVISIT: User does not have rights to Create and Drop Views.
 * Currently, this updated separately.
 *
 * EXECUTE WiMiHardwareAssessmentCreateViewsOnRules 
 *	'%', --Do all OUs
 *	1, --Include child OUs
 *	1 --Get new/modified Rules
 */
 
----------------------------------------

CREATE TABLE #ComputerAssessmentDetails
(
	ComputerID INT,
	OperatingSystem VARCHAR(128) COLLATE database_default,
	LastHardwareReportDate DATETIME,
	AssessmentSeverityNumber INT,
	AssessmentSeverityMessage VARCHAR(64) COLLATE database_default,
	AssessmentMessage VARCHAR(256) COLLATE database_default,
	AssessmentRuleID INT,
	Class VARCHAR(256) COLLATE database_default,
	Property VARCHAR(256) COLLATE database_default,
	PropertyValue VARCHAR(256) COLLATE database_default
)
-----------------------------------
--Build the following dynamically
-----------------------------------
DECLARE @viewName Varchar(64)

DECLARE rulesID_cursor CURSOR FOR
	SELECT DISTINCT r.AssessmentRuleID
	FROM HardwareAssessmentRule r
	WHERE r.Enabled = 1 
	
OPEN rulesID_cursor

DECLARE @rulesID  int
FETCH NEXT FROM rulesID_cursor INTO	@rulesID
WHILE @@FETCH_STATUS = 0
BEGIN
	SET @viewName = 'HardwareAssessmentView_onRule' + CAST(@rulesID as Varchar(4))
	
	--Check if view exists as it is created separately
	IF EXISTS (
		SELECT * FROM dbo.sysobjects
		WHERE [name] = @viewname AND xtype = 'V'
	)
	BEGIN
		DECLARE @insertSQL nvarchar(4000)

		--Get assessment results for a Rule from its corresponding View
		SET @insertSQL =
		N'INSERT INTO #ComputerAssessmentDetails
		(
			ComputerID,
			OperatingSystem,
			LastHardwareReportDate,
			AssessmentSeverityNumber,
			AssessmentSeverityMessage,
			AssessmentMessage,
			AssessmentRuleID,
			Class,
			Property,
			PropertyValue
		)
		SELECT	v.ComputerID,
			os.OperatingSystemName,
			v.LastHardwareReportDate,
			sev.AssessmentSeverityNumber,
			sev.AssessmentSeverityMessage,
			v.AssessmentMessage,
			v.AssessmentRuleID,
			v.Class,
			v.Property,
			v.PropertyValue
		FROM	'+ @viewname + ' as v,
			Computer as c
			LEFT OUTER JOIN OperatingSystem as os
				ON os.OperatingSystemID = c.OperatingSystemID,
			HardwareAssessmentSeverity as sev
		WHERE	v.ComputerID = c.ComputerID
			AND v.AssessmentSeverityID = sev.AssessmentSeverityID
			AND v.ComputerID = ' + cast(@compID as varchar(10))

		EXEC sp_executesql @insertSQL		
	 
	END

	FETCH NEXT FROM rulesID_cursor INTO	@rulesID
END
CLOSE rulesID_cursor
DEALLOCATE rulesID_cursor

SELECT * FROM #ComputerAssessmentDetails

DROP TABLE #ComputerAssessmentDetails

END
GO

EXEC CopyPermission 'WiMiHardwareDispositionDetails', 'mgsOSBreakdown'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiHardwareDispositionSummary.sql,v 1.6 2007/07/24 06:30:56 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: WiMiHardwareDispositionSummary
 *	Input:
 * 	- @oudn: OU DN to search in. Default: '%' (= all OUs)
 *	- @ouMatch: match sub OUs? Default: true
 *
 *	Returns:
 *	- Disposition, Count
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiHardwareDispositionSummary' AND xtype = 'P'
)
        DROP PROCEDURE WiMiHardwareDispositionSummary
GO

CREATE PROCEDURE [WiMiHardwareDispositionSummary]
	@oudn VARCHAR(800) = '%', --Default: For all OUs
	@ouMatch BIT=1, --Include sub-OUs. Default: 1 (i.e. True)
	@groupID INT, --filter on rollout group
	@grpfilter INT=0 -- do not filter
AS
BEGIN

-- Validates and Updates HardwareAssessmentCache table
EXEC WiMiValidateNUpdateAssessmentCache @oudn, @ouMatch

	CREATE TABLE #Organization(OUID int PRIMARY KEY)
	EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

	IF @grpfilter <> 0 -- do filter
		SELECT
			tmp.Disposition,
			COUNT (DISTINCT tmp.ComputerID)as Total			
		FROM
		(
			SELECT
			cache.ComputerID,
			(
				SELECT AssessmentSeverityMessage
				FROM HardwareAssessmentSeverity
				WHERE AssessmentSeverityNumber = MAX(sev.AssessmentSeverityNumber) 
			)as Disposition		
			FROM 	
			(	#Organization AS s INNER JOIN HardwareAssessmentCache AS cache 
				ON (s.OUID = cache.ComputerOUID) 
					INNER JOIN HardwareAssessmentSeverity AS sev
					ON cache.AssessmentSeverityID = sev.AssessmentSeverityID
			)
			LEFT OUTER JOIN 
			(
				RolloutGroup as grp INNER JOIN RolloutGroupMember as mem
				ON grp.GroupID = mem.GroupID
			)
			ON cache.ComputerID = mem.MemberID					
			WHERE		
				grp.GroupID =  @groupID					
			GROUP By cache.ComputerID
		) as tmp --contains assessment for individual computer
		GROUP By tmp.Disposition
	ELSE
		SELECT
			tmp.Disposition,
			COUNT (DISTINCT tmp.ComputerID)as Total		
		FROM
		(
			SELECT
			cache.ComputerID,
			(
				SELECT AssessmentSeverityMessage
				FROM HardwareAssessmentSeverity
				WHERE AssessmentSeverityNumber = MAX(sev.AssessmentSeverityNumber) 
			)as Disposition		
			FROM 	#Organization AS s, 
				HardwareAssessmentCache AS cache, 
				HardwareAssessmentSeverity AS sev
			WHERE
				cache.ComputerOUID = s.OUID
				AND cache.AssessmentSeverityID = sev.AssessmentSeverityID
			GROUP By cache.ComputerID
		) as tmp --contains assessment for individual computer
		GROUP By tmp.Disposition
END
GO

EXEC CopyPermission 'WiMiHardwareDispositionSummary', 'mgsOSBreakdown'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiInventoryStatus.sql,v 1.5 2007/07/24 06:30:56 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: WiMiInventoryStatus
 *	Input:
 * 	- @oudn: OU DN to search in. Default: '%' (= all OUs)
 *	- @ouMatch: match sub OUs? Default: true
 *
 *	Returns:
 *	- Status, Total
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiInventoryStatus' AND xtype = 'P'
)
        DROP PROCEDURE WiMiInventoryStatus
GO

CREATE PROCEDURE [WiMiInventoryStatus]
	@oudn VARCHAR(800)='%', 
	@ouMatch BIT=1, --Values:[0=do exact OU Match | 1=include child-OUs in match]
	@allTargets INT=0, --Values:[0=only targets that have reported inventory | 1=all targets  | 2: Not Inventoried ]
	@groupID INT,
	@grpfilter INT=0
AS

BEGIN

CREATE TABLE #Organization(OUID int PRIMARY KEY)
EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

IF @allTargets < 2 -- ALL items or Inventoried items
	BEGIN 
		IF @grpfilter <> 0 -- do filter
			SELECT 
				c.ComputerID,
				c.ComputerCN AS [ComputerName], -- current name
				dbo.OUName(s.OUID) AS [OUDN],
				CASE 
					WHEN d1.DN IS NULL THEN map.[name]
					WHEN RIGHT(dbo.OUName(s.OUID), LEN(d1.DN)) = d1.DN THEN map.[name]
					ELSE map.[name] + ' (' + d1.DN + ')'	
				END as OldName,
				CASE 
					WHEN d2.DN IS NULL THEN map.newName
					WHEN RIGHT(dbo.OUName(s.OUID), LEN(d2.DN)) = d2.DN THEN map.newName
					ELSE map.newName + ' (' + d2.DN + ')'	
				END as NewName,
				ir.SWDate AS [LastSoftwareReportDate],		
				ir.HWDate AS [LastHardwareReportDate]			
			FROM 
			(RolloutGroup as grp INNER JOIN RolloutGroupMember as mem ON grp.GroupID = mem.GroupID)
			RIGHT OUTER JOIN
				(#Organization as s INNER JOIN 
					(Computer AS c LEFT OUTER JOIN 
						(Namemap AS map
							INNER JOIN Domain AS d1
							ON map.Domain = d1.OrganizationID
							INNER JOIN Domain AS d2
							ON map.newDomain = d2.OrganizationID
						)
						ON c.ComputerID = map.currentID		
					) -- name-mapping for pre and post rollout computer names
					ON (s.OUID = c.ComputerOUID)
				)
				LEFT OUTER JOIN InventoryReport AS ir
				ON c.ComputerID = ir.ComputerID AND ir.UserID = 1
				--reported "System" User i.e. All Users on Computer
			ON mem.MemberID = c.ComputerID			
			WHERE 
				grp.GroupID = @groupID
				AND (	@allTargets = 1
					OR SWDate IS NOT NULL
					OR HWDate IS NOT NULL
				) -- atleast one of Dates is NOT NULL			
			ORDER BY dbo.OUName(s.OUID), c.ComputerCN
		ELSE
			SELECT 
				c.ComputerID,
				c.ComputerCN AS [ComputerName], -- current name
				dbo.OUName(s.OUID) AS [OUDN],
				CASE 
					WHEN d1.DN IS NULL THEN map.[name]
					WHEN RIGHT(dbo.OUName(s.OUID), LEN(d1.DN)) = d1.DN THEN map.[name]
					ELSE map.[name] + ' (' + d1.DN + ')'	
				END as OldName,
				CASE 
					WHEN d2.DN IS NULL THEN map.newName
					WHEN RIGHT(dbo.OUName(s.OUID), LEN(d2.DN)) = d2.DN THEN map.newName
					ELSE map.newName + ' (' + d2.DN + ')'	
				END as NewName,
				ir.SWDate AS [LastSoftwareReportDate],		
				ir.HWDate AS [LastHardwareReportDate]	
			FROM 
			(#Organization AS s INNER JOIN 
				(Computer AS c LEFT OUTER JOIN 
						(Namemap AS map
							INNER JOIN Domain AS d1
							ON map.Domain = d1.OrganizationID
							INNER JOIN Domain AS d2
							ON map.newDomain = d2.OrganizationID
						)
						ON c.ComputerID = map.currentID		
				) -- name-mapping for pre and post rollout computer names
				ON (c.ComputerOUID = s.OUID)
			)
			LEFT OUTER JOIN InventoryReport AS ir
			ON c.ComputerID = ir.ComputerID AND ir.UserID = 1
			--reported "System" User i.e. All Users on Computer	
			WHERE 
				(	@allTargets = 1
					OR SWDate IS NOT NULL
					OR HWDate IS NOT NULL
				) -- atleast one of Dates is NOT NULL	
			ORDER BY dbo.OUName(s.OUID), c.ComputerCN	
	END
ELSE ---Only Not Inventoried items
	BEGIN
		IF @grpfilter <> 0 -- do filter
			SELECT 
				c.ComputerID,
				c.ComputerCN AS [ComputerName], -- current name
				dbo.OUName(s.OUID) AS [OUDN],
				CASE 
					WHEN d1.DN IS NULL THEN map.[name]
					WHEN RIGHT(dbo.OUName(s.OUID), LEN(d1.DN)) = d1.DN THEN map.[name]
					ELSE map.[name] + ' (' + d1.DN + ')'	
				END as OldName,
				CASE 
					WHEN d2.DN IS NULL THEN map.newName
					WHEN RIGHT(dbo.OUName(s.OUID), LEN(d2.DN)) = d2.DN THEN map.newName
					ELSE map.newName + ' (' + d2.DN + ')'	
				END as NewName,
				ir.SWDate AS [LastSoftwareReportDate],		
				ir.HWDate AS [LastHardwareReportDate]	
			FROM 
			(RolloutGroup as grp INNER JOIN RolloutGroupMember as mem ON grp.GroupID = mem.GroupID)
			RIGHT OUTER JOIN	
				(#Organization AS s 
					INNER JOIN 
					(Computer AS c 
						LEFT OUTER JOIN 
						(Namemap AS map
							INNER JOIN Domain AS d1
							ON map.Domain = d1.OrganizationID
							INNER JOIN Domain AS d2
							ON map.newDomain = d2.OrganizationID
						)
						ON c.ComputerID = map.currentID		
					) -- name-mapping for pre and post rollout computer names
					ON (c.ComputerOUID = s.OUID)
				)
				LEFT OUTER JOIN InventoryReport AS ir
				ON c.ComputerID = ir.ComputerID AND ir.UserID = 1
				--reported "System" User i.e. All Users on Computer
			ON mem.MemberID = c.ComputerID	
			WHERE
				grp.GroupID = @groupID 			
				AND (	SWDate IS NULL
					AND HWDate IS NULL
				)  -- Both Dates are NULL
		ELSE
			SELECT 
				c.ComputerID,
				c.ComputerCN AS [ComputerName], -- current name
				dbo.OUName(s.OUID) AS [OUDN],
				CASE 
					WHEN d1.DN IS NULL THEN map.[name]
					WHEN RIGHT(dbo.OUName(s.OUID), LEN(d1.DN)) = d1.DN THEN map.[name]
					ELSE map.[name] + ' (' + d1.DN + ')'	
				END as OldName,
				CASE 
					WHEN d2.DN IS NULL THEN map.newName
					WHEN RIGHT(dbo.OUName(s.OUID), LEN(d2.DN)) = d2.DN THEN map.newName
					ELSE map.newName + ' (' + d2.DN + ')'	
				END as NewName,
				ir.SWDate AS [LastSoftwareReportDate],		
				ir.HWDate AS [LastHardwareReportDate]	
			FROM 
			(#Organization AS s 
				INNER JOIN 
				(Computer AS c 
					LEFT OUTER JOIN 
					(Namemap AS map
						INNER JOIN Domain AS d1
						ON map.Domain = d1.OrganizationID
						INNER JOIN Domain AS d2
						ON map.newDomain = d2.OrganizationID
					)
					ON c.ComputerID = map.currentID		
				) -- name-mapping for pre and post rollout computer names
				ON (c.ComputerOUID = s.OUID)
			)
			LEFT OUTER JOIN InventoryReport AS ir
			ON c.ComputerID = ir.ComputerID AND ir.UserID = 1
			--reported "System" User i.e. All Users on Computer	
			WHERE 			
				(	SWDate IS NULL
					AND HWDate IS NULL
				)  -- Both Dates are NULL	
	END
END

RETURN

GO

CopyPermission 'WiMiInventoryStatus', 'mgsOSBreakdown'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiInventoryStatusSummary.sql,v 1.6 2007/07/24 06:30:56 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: WiMiInventoryStatusSummary
 *	Input:
 * 	- @oudn: OU DN to search in. Default: '%' (= all OUs)
 *	- @ouMatch: match sub OUs? Default: true
 *
 *	Returns:
 *	- Status, Total
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiInventoryStatusSummary' AND xtype = 'P'
)
        DROP PROCEDURE WiMiInventoryStatusSummary
GO

CREATE PROCEDURE [WiMiInventoryStatusSummary]
	@oudn VARCHAR(800) = '%',	--Default: For all OUs
	@ouMatch BIT=1,			--Include sub-OUs? Default: 1
	@groupID INT, 			--filter on rollout group
	@grpfilter INT=0 			-- do not filter
AS
	BEGIN
		CREATE TABLE #Organization(OUID int PRIMARY KEY)
		EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

		IF @grpfilter <> 0 -- do filter

			SELECT
			'Inventoried' as Status,
			COUNT(c.ComputerID) as Total
			FROM
			( 
				RolloutGroup as grp INNER JOIN RolloutGroupMember as mem
				ON grp.GroupID = mem.GroupID
			)
			RIGHT OUTER JOIN
				( #Organization as s INNER JOIN Computer as c 
					ON (s.OUID = c.ComputerOUID)
				)
				LEFT OUTER JOIN InventoryReport as ir
				ON (c.ComputerID = ir.ComputerID AND ir.UserID=1)
			ON mem.MemberID = c.ComputerID
			WHERE ir.HWDate IS NOT NULL AND grp.GroupID = @groupID
			
			UNION
			
			SELECT
				'Not inventoried' as Status,
				COUNT(c.ComputerID) as Total
			FROM
			( 
				RolloutGroup as grp INNER JOIN RolloutGroupMember as mem
				ON grp.GroupID = mem.GroupID
			)
			RIGHT OUTER JOIN
				( #Organization as s INNER JOIN Computer as c 
					ON (s.OUID = c.ComputerOUID)
				)
				LEFT OUTER JOIN InventoryReport as ir
				ON (c.ComputerID = ir.ComputerID AND ir.UserID=1)
			ON mem.MemberID = c.ComputerID
			WHERE ir.HWDate IS NULL AND grp.GroupID = @groupID
		
		ELSE
		
			SELECT
				'Inventoried' as Status,
				COUNT(c.ComputerID) as Total
			FROM
			( #Organization as s INNER JOIN Computer as c 
				ON (s.OUID = c.ComputerOUID)
			)
			LEFT OUTER JOIN InventoryReport as ir
			ON (c.ComputerID = ir.ComputerID AND ir.UserID=1)		
			WHERE ir.HWDate IS NOT NULL
			
			UNION
			
			SELECT
				'Not inventoried' as Status,
				COUNT(c.ComputerID) as Total
			FROM
				( #Organization as s INNER JOIN Computer as c 
					ON (s.OUID = c.ComputerOUID)
				)
				LEFT OUTER JOIN InventoryReport as ir
				ON (c.ComputerID = ir.ComputerID AND ir.UserID=1)		
			WHERE ir.HWDate IS NULL
	END
GO

CopyPermission 'WiMiInventoryStatusSummary', 'mgsOSBreakdown'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiMigrationProgress.sql,v 1.12 2008/10/29 00:58:01 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: WiMiMigrationProgress
 *	Input:
 *		@oudn, @ouMatch, @state, @compCN
 *
 *	Returns:
 *		ComputerID, ComputerName, OUDN, State
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiMigrationProgress' AND xtype = 'P'
)
        DROP PROCEDURE WiMiMigrationProgress
GO

CREATE PROCEDURE [WiMiMigrationProgress]
	@oudn VARCHAR(800) = '%',	--Default: For all OUs
	@ouMatch BIT=1,		--Include sub-OUs? Default: 1 i.e. True
	@state VARCHAR(20) = NULL,	--State to filter on	
	@groupID INT,   --GroupID to filter on
	@grpfilter INT= 0 -- do not filter on group
AS
BEGIN

	IF @state IS NULL
		SET @state = '%'

	-- Retrieve Summary Data for ALL GROUPs
	CREATE TABLE #parentListTbl
	(	
		GroupID int,
		GroupFullName nvarchar(800) COLLATE database_default
	)

	DECLARE grpList_cursor CURSOR FOR
		Select GroupID, ParentID from RolloutGroup

	DECLARE @grpID int
	DECLARE @fullName nvarchar(800)
	DECLARE @parentID int

	OPEN grpList_cursor
	FETCH NEXT FROM grpList_cursor INTO @grpID, @parentID
	WHILE @@FETCH_STATUS =0
	BEGIN
		--Retrieve parent name
		EXECUTE RolloutGroupParentName @grpID, @outFullName = @fullName OUTPUT 	
		INSERT INTO #parentListTbl VALUES ( @grpID, @fullName ) 
		SET @fullName = NULL -- reset	
		FETCH NEXT FROM grpList_cursor INTO @grpID, @parentID
	END

	CLOSE grpList_cursor
	DEALLOCATE grpList_cursor
	
	CREATE TABLE #Organization(OUID int PRIMARY KEY)
	EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

		IF @grpfilter <> 0 -- do filter
			SELECT 
				State, 
				ComputerID, 
				ComputerCN AS [ComputerName],  -- current name
				OU AS [OUDN], 
				OldName,
				NewName, 
				MemberID, 
				GroupID, 
				GroupFullName,
				LastStep,
				LastStepDescription,
				StepCount,
				LastDateTime
			FROM
			(
				SELECT 
					ms.State, 
					c.ComputerID,
					c.ComputerCN,
					dbo.OUName(s.OUID) AS OU,
					CASE 
						WHEN d1.DN IS NULL THEN map.[name]
						WHEN RIGHT(dbo.OUName(s.OUID), LEN(d1.DN)) = d1.DN THEN map.[name]
						ELSE map.[name] + ' (' + d1.DN + ')'	
					END as OldName,
					CASE 
						WHEN d2.DN IS NULL THEN map.newName
						WHEN RIGHT(dbo.OUName(s.OUID), LEN(d2.DN)) = d2.DN THEN map.newName
						ELSE map.newName + ' (' + d2.DN + ')'	
					END as NewName,
					ms.LastStep,
					ms.LastStepDescription,
					ms.StepCount,
					ms.LastDateTime
				FROM		
					WiMiMigrationStateByComputer AS ms
					INNER JOIN 	
						(Computer AS c 
							LEFT OUTER JOIN 
							(Namemap as map
								INNER JOIN Domain as d1
								ON map.Domain = d1.OrganizationID
								INNER JOIN Domain as d2
								ON map.newDomain = d2.OrganizationID
							)
							ON c.ComputerID = map.currentID		
						) -- name-mapping for pre and post rollout computer names
						INNER JOIN #Organization AS s		
						ON (c.ComputerOUID = s.OUID)
					ON ms.ComputerID = c.ComputerID
			) as t1
			LEFT OUTER JOIN
			(
				SELECT 
					mem.MemberID,
					grp.GroupID,
					parent.GroupFullName
				FROM	
					#parentListTbl as parent, 
					RolloutGroup as grp, 
					RolloutGroupMember as mem
				WHERE	
					parent.GroupID = grp.GroupID		
					AND grp.GroupID = mem.GroupID 
			) as t2		
			ON t1.ComputerID = t2.MemberID	
				
			WHERE	t2.GroupID = @groupID					
					AND t1.State LIKE @state

		ELSE
			SELECT 
				ms.State, 
				c.ComputerID,
				c.ComputerCN AS [ComputerName], -- current name
				dbo.OUName(s.OUID) AS [OUDN],
				CASE 
					WHEN d1.DN IS NULL THEN map.[name]
					WHEN RIGHT(dbo.OUName(s.OUID), LEN(d1.DN)) = d1.DN THEN map.[name]
					ELSE map.[name] + ' (' + d1.DN + ')'	
				END as OldName,
				CASE 
					WHEN d2.DN IS NULL THEN map.newName
					WHEN RIGHT(dbo.OUName(s.OUID), LEN(d2.DN)) = d2.DN THEN map.newName
					ELSE map.newName + ' (' + d2.DN + ')'	
				END as NewName,
				NULL as MemberID,
				NULL as GroupID,
				NULL as GroupFullName,
				ms.LastStep,
				ms.LastStepDescription,
				ms.StepCount,
				ms.LastDateTime
			FROM		
				WiMiMigrationStateByComputer AS ms
				INNER JOIN 	
					(Computer AS c 
						LEFT OUTER JOIN 
						(Namemap AS map
							INNER JOIN Domain AS d1
							ON map.Domain = d1.OrganizationID
							INNER JOIN Domain AS d2
							ON map.newDomain = d2.OrganizationID
						)
						ON c.ComputerID = map.currentID		
					) -- name-mapping for pre and post rollout computer names
					INNER JOIN #Organization AS s		
					ON (c.ComputerOUID = s.OUID)
				ON ms.ComputerID = c.ComputerID
			WHERE
				ms.State LIKE @state

END
GO


CopyPermission 'WiMiMigrationProgress', 'mgsOSBreakdown'
GO


/*
 * $ Exp $
 *
 * DESCRIPTION: Stored Procedure: WiMiMigrationProgressFailReason
 *	Input:
 *		@compID
 *
 *	Returns:
 *		PackageName, Version, PolicyAction, FailReason, PackageFullName
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiMigrationProgressFailReason' AND xtype = 'P'
)
        DROP PROCEDURE WiMiMigrationProgressFailReason
GO


CREATE PROCEDURE WiMiMigrationProgressFailReason
	@compID as int
AS
BEGIN
	SELECT	r.Phase,
		r.StepNumber,
		r.Description,
		LOWER(r.Status) AS Status,
		r.StatusMessage,
		NULL AS PackageName,
		NULL AS PackageVersion,
		NULL AS PackageFullName
	FROM	RolloutStatus as r, RolloutStatusType as t
	WHERE	r.ComputerID = @compID
			AND r.RolloutStatusTypeID = t.RolloutStatusTypeID
			AND t.[Name] = 'ComputerBuild'
	UNION ALL
	SELECT	NULL,
		NULL,
		NULL,
		i.Result,
		r.ReasonText AS FailReason,
		pf.PackageName,
		pv.Version,
		pp.PackageFullName
	FROM	Installation as i
		LEFT OUTER JOIN Reason AS r
			ON i.FailReasonID = r.ReasonID,
		PackageVersion as pv,
		PackageFamily as pf,
		PackagePath as pp
	WHERE	@compID = i.ComputerID		-- Installed on this computer
	AND	i.UserID = 1			-- For SYSTEM/all
	AND	'success' <> i.Result
	AND	'notrequired' <> i.Result
	AND	i.PackageVersionID = pv.PackageVersionID
	AND	pv.PackageFamilyID = pf.PackageFamilyID
	AND	pv.PackagePathID = pp.PackagePathID
	ORDER BY PackageName, PackageVersion, StepNumber
END
GO

CopyPermission 'WiMiMigrationProgressFailReason', 'mgsOSBreakdown'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiMigrationProgressSummary.sql,v 1.6 2007/07/24 06:30:56 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: WiMiMigrationProgressSummary
 *	Input:
 * 	- @oudn: OU DN to search in. Default: '%' (= all OUs)
 *	- @ouMatch: match sub OUs? Default: true
 *
 *	Returns:
 *	- State, Total
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiMigrationProgressSummary' AND xtype = 'P'
)
        DROP PROCEDURE WiMiMigrationProgressSummary
GO

CREATE PROCEDURE [WiMiMigrationProgressSummary]
	@oudn VARCHAR(800) = '%',	--Default: For all OUs
	@ouMatch BIT=1,			--Include sub-OUs? Default: 1
	@groupID INT, --filter on rollout group
	@grpfilter INT=0 -- do not filter
AS
	BEGIN
		CREATE TABLE #Organization(OUID int PRIMARY KEY)
		EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

		IF @grpfilter <> 0 -- do filter
			SELECT 
				State, 
				COUNT(ComputerID) as Total
			FROM
			(
				SELECT 
					ms.State, 
					c.ComputerID,
					c.ComputerCN,
					dbo.OUName(s.OUID) AS OU
				FROM	
					WiMiMigrationStateByComputer as ms,
					Computer AS c, 
					#Organization AS s
					
				WHERE
					ms.ComputerID = c.ComputerID
					AND c.ComputerOUID = s.OUID 
			) as t1
			LEFT OUTER JOIN
			(
				SELECT 
					mem.MemberID,
					grp.GroupID
				FROM						 
					RolloutGroup as grp, 
					RolloutGroupMember as mem
				WHERE	
					grp.GroupID = mem.GroupID 
			) as t2		
			ON t1.ComputerID = t2.MemberID	
				
			WHERE	t2.GroupID = @groupID
			
			GROUP BY State
			ORDER BY State	

		ELSE
			SELECT 
				ms.State, 
				COUNT(c.ComputerID) as Total
			FROM	
				WiMiMigrationStateByComputer as ms,
				Computer AS c, 
				#Organization AS s
				
			WHERE
				ms.ComputerID = c.ComputerID
				AND c.ComputerOUID = s.OUID 

			GROUP BY State
			ORDER BY State

	END

GO

CopyPermission 'WiMiMigrationProgressSummary', 'mgsOSBreakdown'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiImageCaptureProgress.sql,v 1.2 2007/07/24 06:30:56 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: WiMiImageCaptureProgress
 *	Input:
 *		@oudn, @ouMatch, @state, @compCN
 *
 *	Returns:
 *		ComputerID, ComputerName, OUDN, State
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiImageCaptureProgress' AND xtype = 'P'
)
        DROP PROCEDURE WiMiImageCaptureProgress
GO

CREATE PROCEDURE [WiMiImageCaptureProgress]
	@oudn VARCHAR(800) = '%',	--Default: For all OUs
	@ouMatch BIT=1,		--Include sub-OUs? Default: 1 i.e. True
	@state VARCHAR(50) = NULL
AS
BEGIN

	IF @state IS NULL
		SET @state = '%'
	
	CREATE TABLE #Organization(OUID int PRIMARY KEY)
	EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

		SELECT 
			ms.State, 
			c.ComputerID,
			c.ComputerCN AS [ComputerName], -- current name
			dbo.OUName(s.OUID) AS [OUDN],
			ms.LastStep,
			ms.LastStepDescription,
			ms.StepCount,
			ms.LastDateTime,
			ms.DataImageCaptureLocation,
			ms.ImageCaptureDateTime
		FROM		
			WiMiImageCaptureStateByComputer AS ms
			INNER JOIN 	
				(Computer AS c 
					LEFT OUTER JOIN 
					(Namemap AS map
						INNER JOIN Domain AS d1
						ON map.Domain = d1.OrganizationID
						INNER JOIN Domain AS d2
						ON map.newDomain = d2.OrganizationID
					)
					ON c.ComputerID = map.currentID		
				) -- name-mapping for pre and post rollout computer names
				INNER JOIN #Organization AS s		
				ON (c.ComputerOUID = s.OUID)
			ON ms.ComputerID = c.ComputerID
		WHERE
			ms.State LIKE @state

END
GO


CopyPermission 'WiMiImageCaptureProgress', 'mgsOSBreakdown'
GO

/*
 * $ Exp $
 *
 * DESCRIPTION: Stored Procedure: WiMiImageCaptureProgressInformation
 *	Input:
 *		@compID
 *
 *	Returns:
 *		PackageName, Version, PolicyAction, FailReason, PackageFullName
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiImageCaptureProgressInformation' AND xtype = 'P'
)
        DROP PROCEDURE WiMiImageCaptureProgressInformation
GO


CREATE PROCEDURE WiMiImageCaptureProgressInformation
	@compID as int
AS
BEGIN
	SELECT	r.StepNumber,
			r.Description,
			LOWER(r.Status) AS Status,
			r.StatusMessage
	FROM	RolloutStatus as r, RolloutStatusType as t
	WHERE	r.ComputerID = @compID
				AND r.RolloutStatusTypeID = t.RolloutStatusTypeID
				AND t.[Name] = 'ComputerImageCapture'
				ORDER BY StepNumber
END
GO

CopyPermission 'WiMiImageCaptureProgressInformation', 'mgsOSBreakdown'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiImageCaptureProgressSummary.sql,v 1.2 2007/07/24 06:30:56 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: WiMiImageCaptureProgressSummary
 *	Input:
 * 	- @oudn: OU DN to search in. Default: '%' (= all OUs)
 *	- @ouMatch: match sub OUs? Default: true
 *
 *	Returns:
 *	- State, Total
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiImageCaptureProgressSummary' AND xtype = 'P'
)
        DROP PROCEDURE WiMiImageCaptureProgressSummary
GO

CREATE PROCEDURE [WiMiImageCaptureProgressSummary]
	@oudn VARCHAR(800) = '%',	--Default: For all OUs
	@ouMatch BIT = 1			--Include sub-OUs? Default: 1
AS
	BEGIN
		CREATE TABLE #Organization(OUID int PRIMARY KEY)
		EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

		SELECT 
			ms.State, 
			COUNT(c.ComputerID) as Total
		FROM	
			WiMiImageCaptureStateByComputer as ms,
			Computer AS c, 
			#Organization AS s
			
		WHERE
			ms.ComputerID = c.ComputerID
			AND c.ComputerOUID = s.OUID 

		GROUP BY State
		ORDER BY State

	END

GO

CopyPermission 'WiMiImageCaptureProgressSummary', 'mgsOSBreakdown'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiDataCaptureProgress.sql,v 1.2 2007/07/24 06:30:56 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: WiMiDataCaptureProgress
 *	Input:
 *		@oudn, @ouMatch, @state, @compCN
 *
 *	Returns:
 *		ComputerID, ComputerName, OUDN, State
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiDataCaptureProgress' AND xtype = 'P'
)
        DROP PROCEDURE WiMiDataCaptureProgress
GO

CREATE PROCEDURE [WiMiDataCaptureProgress]
	@oudn VARCHAR(800) = '%',	--Default: For all OUs
	@ouMatch BIT=1,		--Include sub-OUs? Default: 1 i.e. True
	@state VARCHAR(50) = NULL
AS
BEGIN

	IF @state IS NULL
		SET @state = '%'
	
	CREATE TABLE #Organization(OUID int PRIMARY KEY)
	EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

		SELECT 
			ms.State, 
			c.ComputerID,
			c.ComputerCN AS [ComputerName], -- current name
			dbo.OUName(s.OUID) AS [OUDN],
			ms.LastStep,
			ms.LastStepDescription,
			ms.StepCount,
			ms.LastDateTime,
			ms.DataStoreLocation,
			ms.CaptureDateTime
		FROM		
			WiMiDataCaptureStateByComputer AS ms
			INNER JOIN 	
				(Computer AS c 
					LEFT OUTER JOIN 
					(Namemap AS map
						INNER JOIN Domain AS d1
						ON map.Domain = d1.OrganizationID
						INNER JOIN Domain AS d2
						ON map.newDomain = d2.OrganizationID
					)
					ON c.ComputerID = map.currentID		
				) -- name-mapping for pre and post rollout computer names
				INNER JOIN #Organization AS s		
				ON (c.ComputerOUID = s.OUID)
			ON ms.ComputerID = c.ComputerID
		WHERE
			ms.State LIKE @state

END
GO


CopyPermission 'WiMiDataCaptureProgress', 'mgsOSBreakdown'
GO

/*
 * $ Exp $
 *
 * DESCRIPTION: Stored Procedure: WiMiDataCaptureProgressInformation
 *	Input:
 *		@compID
 *
 *	Returns:
 *		PackageName, Version, PolicyAction, FailReason, PackageFullName
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiDataCaptureProgressInformation' AND xtype = 'P'
)
        DROP PROCEDURE WiMiDataCaptureProgressInformation
GO


CREATE PROCEDURE WiMiDataCaptureProgressInformation
	@compID as int
AS
BEGIN
	SELECT	r.StepNumber,
			r.Description,
			LOWER(r.Status) AS Status,
			r.StatusMessage
	FROM	RolloutStatus as r, RolloutStatusType as t
	WHERE	r.ComputerID = @compID
				AND r.RolloutStatusTypeID = t.RolloutStatusTypeID
				AND t.[Name] = 'ComputerDataCapture'
				ORDER BY StepNumber
END
GO

CopyPermission 'WiMiDataCaptureProgressInformation', 'mgsOSBreakdown'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiDataCaptureProgressSummary.sql,v 1.2 2007/07/24 06:30:56 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: WiMiDataCaptureProgressSummary
 *	Input:
 * 	- @oudn: OU DN to search in. Default: '%' (= all OUs)
 *	- @ouMatch: match sub OUs? Default: true
 *
 *	Returns:
 *	- State, Total
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiDataCaptureProgressSummary' AND xtype = 'P'
)
        DROP PROCEDURE WiMiDataCaptureProgressSummary
GO

CREATE PROCEDURE [WiMiDataCaptureProgressSummary]
	@oudn VARCHAR(800) = '%',	--Default: For all OUs
	@ouMatch BIT = 1			--Include sub-OUs? Default: 1
AS
	BEGIN
		CREATE TABLE #Organization(OUID int PRIMARY KEY)
		EXEC OrganizationIdentifyDescendentsByOUDN @oudn, @ouMatch, 0

		SELECT 
			ms.State, 
			COUNT(c.ComputerID) as Total
		FROM	
			WiMiDataCaptureStateByComputer as ms,
			Computer AS c, 
			#Organization AS s
			
		WHERE
			ms.ComputerID = c.ComputerID
			AND c.ComputerOUID = s.OUID 

		GROUP BY State
		ORDER BY State

	END

GO

CopyPermission 'WiMiDataCaptureProgressSummary', 'mgsOSBreakdown'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiRolloutGroups.sql,v 1.8 2004/03/09 07:25:11 kbui Exp $
 *
 * DESCRIPTION: Stored Procedure: RolloutGroupEntry
 *	Input:
 * 	- @groupName: name of group Default: '%' (= all groups)
 *	- @parentID: parentID of group. Default: 0 i.e. represents Ghost Server
 *
 *	Returns:
 *	- GroupID, GroupName, ParentID,	Source
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */
 
-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'RolloutGroupEntry' AND xtype = 'P'
)
        DROP PROCEDURE RolloutGroupEntry
GO
 
CREATE PROCEDURE RolloutGroupEntry
	@groupName nvarchar(256) = N'%',
	@parentID int = 0 --represents base or Ghost Server
AS

	SELECT 
		GroupID,
		GroupName,
		ParentID,
		Source
	FROM RolloutGroup rs
	WHERE GroupName LIKE @groupName
		AND ParentID = @parentID --when 0; its base ie. Ghost Server

GO

CopyPermission 'RolloutGroupEntry', 'mgsOSBreakdown'
GO 

/*
 *
 * DESCRIPTION: Stored Procedure: GhostServerEntry
 *	Input:
 * 	- @serverMAC: MAC Address of Ghost Server
 *
 *	Returns:
 *	- GhostServerID, GroupID, ServerName, 
 *	- PhysicalAddress, DomainName, LastUpdateDateTime
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */
 
-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'GhostServerEntry' AND xtype = 'P'
)
        DROP PROCEDURE GhostServerEntry
GO

CREATE PROCEDURE GhostServerEntry
	@serverMAC varchar(64) -- Ghost Server should always have a MAC address
AS
SELECT 
	gs.GhostServerID,
	gs.GroupID,
	gs.ServerName,
	gs.PhysicalAddress,
	gs.DomainName,
	gs.LastUpdateDateTime

FROM GhostServer gs
WHERE	gs.PhysicalAddress LIKE @serverMAC	

GO  

CopyPermission 'GhostServerEntry', 'mgsOSBreakdown'
GO 

/*
 *
 * DESCRIPTION: Stored Procedure: RolloutGroupGhostComputerEntry
 *	Input:
 *	- @groupID: GroupID of a group
 * 	- @serverMAC: MAC Address of computer in the group
 *
 *	Returns:
 *	- GroupID, ComputerName, PhysicalAddress, IsResolved
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'RolloutGroupGhostComputerEntry' AND xtype = 'P'
)
        DROP PROCEDURE RolloutGroupGhostComputerEntry
GO

CREATE PROCEDURE RolloutGroupGhostComputerEntry
	@groupID int, -- 
	@compMAC varchar(64) = '%'
AS
	SELECT 
		rc.GroupID,
		rc.ComputerName,
		rc.PhysicalAddress,
		rc.IsResolved

	FROM RolloutGroupToGhostComputer rc
	WHERE	rc.GroupID = @groupID	
		AND rc.PhysicalAddress LIKE @compMAC
		
GO 

CopyPermission 'RolloutGroupGhostComputerEntry', 'mgsOSBreakdown'
GO 

/*
 *
 * DESCRIPTION: Stored Procedure: ResolveGhostComputer
 * - resolves computers from ghost based on ManageSoft Computer
 * entries
 *
 *	Input:
 *	- none
 *
 *	Returns:
 *	- none
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'ResolveGhostComputer' AND xtype = 'P'
)
        DROP PROCEDURE ResolveGhostComputer
GO

CREATE PROCEDURE ResolveGhostComputer
AS
	DECLARE @memberType int
	SELECT @memberType = TargetTypeID FROM TargetType WHERE TargetTypeName = 'Computer'

	--|||||||||||||||||||||||||||||||||||||||||||||
	-- Create New entries in RolloutGroupMember
	--|||||||||||||||||||||||||||||||||||||||||||||
	
	--Only using MAC address as Ghost is a reliable source
	INSERT INTO RolloutGroupMember
	SELECT DISTINCT
		rc.GroupID,
		ho.ComputerID as MemberID,
		@memberType as MemberType
	FROM 	RolloutGroupToGhostComputer as rc,
		HardwareObject as ho,
		HardwareProperty as hp,
		HardwareValue as hv
	WHERE	ho.HardwareObjectID = hv.HardwareObjectID
		AND hv.HardwarePropertyID = hp.HardwarePropertyID
		AND hp.Property = 'MACAddress'		
		AND hv.Value = rc.PhysicalAddress
		AND rc.IsResolved = 0 --Not resolved yet
		AND NOT EXISTS (
			SELECT * 
			FROM RolloutGroupMember 
			WHERE
				GroupID = rc.GroupID 
				AND MemberID = ho.ComputerID
				AND MemberType = @memberType
			) -- to maintain unique rows
		

	UPDATE RolloutGroupToGhostComputer
		SET IsResolved = 1
	FROM	RolloutGroupToGhostComputer as rc,
		HardwareObject as ho,
		HardwareProperty as hp,
		HardwareValue as hv,
		RolloutGroupMember as rm
	WHERE	ho.HardwareObjectID = hv.HardwareObjectID
		AND hv.HardwarePropertyID = hp.HardwarePropertyID
		AND hp.Property = 'MACAddress'		
		AND hv.Value = rc.PhysicalAddress
		AND ho.ComputerID = rm.MemberID
		AND rm.GroupID = rc.GroupID
		AND rc.IsResolved = 0 -- Not resolved yet

	--|||||||||||||||||||||||||||||||||||||||||||||
	-- Delete /Update Old entries based on changes
	-- in ManageSoft tables i.e. Computer and 
	-- HardwareProperty.
	--|||||||||||||||||||||||||||||||||||||||||||||
			
	--Delete from RolloutGroupMember table if entry not in Computer table 
	DELETE RolloutGroupMember FROM RolloutGroupMember
		WHERE NOT EXISTS
				(SELECT * FROM Computer WHERE ComputerID = MemberID)
	
	--Update if entry exists in MAC list but not in HardwareProperty			
	UPDATE RolloutGroupToGhostComputer SET IsResolved = 0
		FROM RolloutGroupToGhostComputer as rc
		WHERE IsResolved = 1
			AND NOT EXISTS (
				SELECT *
				FROM	HardwareProperty as hp,
					HardwareValue as hv
				WHERE	hv.HardwarePropertyID = hp.HardwarePropertyID
					AND hp.Property = 'MACAddress'		
					AND hv.Value = rc.PhysicalAddress
			)

GO

CopyPermission 'ResolveGhostComputer', 'mgsOSBreakdown'
GO

/*
 *
 * DESCRIPTION: Stored Procedure: RemoveGhostComputer
 * - removes computers from ghost based on ManageSoft Computer
 * entries
 *
 *	Input:
 *	- none
 *
 *	Returns:
 *	- none
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'RemoveGhostComputer' AND xtype = 'P'
)
        DROP PROCEDURE RemoveGhostComputer
GO

CREATE PROCEDURE RemoveGhostComputer
	@groupID int,
	@compMAC varchar(64)
AS
	
	-- Delete from RolloutGroupToGhostComputer table
	DELETE	RolloutGroupToGhostComputer
	WHERE	GroupID = @groupID
		AND	PhysicalAddress = @compMAC
	
	DECLARE @memberType int
	SELECT @memberType = TargetTypeID FROM TargetType WHERE TargetTypeName = 'Computer'
	
	-- Delete from RolloutGroupMember table
	DELETE	RolloutGroupMember
	FROM	RolloutGroupMember, HardwareObject, HardwareValue, HardwareProperty
	WHERE	RolloutGroupMember.MemberID = HardwareObject.ComputerID
		AND	HardwareObject.HardwareObjectID = HardwareValue.HardwareObjectID
		AND	HardwareValue.HardwarePropertyID = HardwareProperty.HardwarePropertyID
		AND	HardwareProperty.Property = 'MACAddress'
		AND	HardwareValue.Value = @compMAC

GO

CopyPermission 'RemoveGhostComputer', 'mgsOSBreakdown'
GO

/*
 *
 * DESCRIPTION: Stored Procedure: RemoveGhostGroup
 * - removes ghost groups from ghost based on ManageSoft Computer
 * entries
 *
 *	Input:
 *	- none
 *
 *	Returns:
 *	- none
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'RemoveGhostGroup' AND xtype = 'P'
)
        DROP PROCEDURE RemoveGhostGroup
GO

CREATE PROCEDURE RemoveGhostGroup
	@groupID int
AS
	-- Delete from RemoveGhostGroup table
	DELETE	RolloutGroup
	WHERE	GroupID = @groupID
GO

CopyPermission 'RemoveGhostGroup', 'mgsOSBreakdown'
GO
/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiNameMap.sql,v 1.13 2007/07/24 06:30:56 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: NameMapDelete
 *	Input:
 * 	- @computerName: name of the computer to remove from the mapping
 *	- @computerDomain: domain the computerName
 *
 *	Returns:
 *	- nothing - removes the name mapping from the table
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */
 
-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'NameMapDelete' AND xtype = 'P'
)
        DROP PROCEDURE NameMapDelete
GO

CREATE PROCEDURE [NameMapDelete]
	@id int
AS

	IF EXISTS(SELECT * 
		FROM NameMap
		WHERE CurrentId = @id)
	BEGIN
		DELETE FROM NameMap
		WHERE CurrentId = @id
	END

GO

CopyPermission 'NameMapDelete', 'mgsOSBreakdown'
GO 

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'NameMapRemove' AND xtype = 'P'
)
        DROP PROCEDURE NameMapRemove
GO

CREATE PROCEDURE [NameMapRemove]
	@newName nvarchar(64),
	@newDomain nvarchar(100),
	@newOU nvarchar(512)
AS
	DECLARE @DomainID AS int
	DECLARE @newOUID AS int

	SELECT @DomainID = OrganizationID FROM Domain
	WHERE DN = @newDomain

	SELECT @newOUID = OrganizationID FROM Organization
	WHERE RDN = @newOU 
	  AND DomainID = @DomainID
	
	IF EXISTS(SELECT * 
		FROM NameMap
		WHERE newname = @newName
		  AND newDomain = @DomainID
		  AND newOU = @newOUID
		)
	BEGIN
		DELETE FROM NameMap
		WHERE newname = @newName
		  AND newDomain = @DomainID
		  AND newOU = @newOUID
	END

GO


CopyPermission 'NameMapRemove', 'mgsOSBreakdown'
GO 

/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiNameMap.sql,v 1.13 2007/07/24 06:30:56 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: NameMapUpdate
 *	Input:
 * 	- @computerName: name of the computer to update
 * 	- @computerDomain: domain of the computer to update
 * 	- @newName: new name for the computer
 * 	- @newDomain: new domain for the computer
 *
 *	Returns:
 	- nothing. updates/adds rows to the NameMap table
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */
 
-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'NameMapUpdate' AND xtype = 'P'
)
        DROP PROCEDURE NameMapUpdate
GO

CREATE PROCEDURE [NameMapUpdate]
	@computerName nvarchar(64),
	@computerDomain nvarchar(100),
	@newName nvarchar(64),
	@newDomain nvarchar(100),
	@id int,
	@newOU nvarchar(512),
	@updatedNewName nvarchar(64) -- used to rename computers that have no @id
AS
	DECLARE @DomainID AS int
	DECLARE @newDomainID AS int
	DECLARE @newOUID AS int

	SELECT @DomainID = OrganizationID FROM Domain
	WHERE DN = @computerDomain

	SELECT @newDomainID = OrganizationID FROM Domain
	WHERE DN = @newDomain
	
	SELECT @newOUID = OrganizationID FROM Organization
	WHERE RDN = @newOU 
	  AND DomainID = @newDomainID

	-- this condition updates a computer that exists in the 
	-- computer table (and therefore in Active Directory)
	IF EXISTS(SELECT * 
		FROM NameMap as nm, Domain as d
		WHERE nm.CurrentID = @id
		  AND @id IS NOT NULL)
		BEGIN
			UPDATE NameMap
			SET newName = @newName,
			    newDomain = @newDomainID,
			    newOU = NULL
			WHERE CurrentId = @id
		END
	-- this condition updates name mappings for new computers that
	-- do not exist in the Computer table or in Active Directory
	ELSE IF EXISTS (SELECT * 
		FROM NameMap as nm
		WHERE nm.[name] IS NULL
		  AND nm.domain IS NULL
		  AND nm.CurrentID IS NULL
		  AND nm.newOU = @newOUID
		  AND nm.newName = @newName
		  AND nm.newDomain = @newDomainID)
		BEGIN
			UPDATE NameMap
			SET newName = @updatedNewName
			WHERE [name] IS NULL
			AND domain IS NULL
			AND CurrentID IS NULL
			AND newOU = @newOUID
			AND newName = @newName
			AND newDomain = @newDomainID
		END
	-- This final case inserts new name mappings into the table
	--
	ELSE
		BEGIN
			INSERT INTO NameMap
			(CurrentId, [name], domain, newName, newDomain, newOU)
			VALUES
			(@id, @computerName, @DomainID, @newName, @newDomainID, @newOUID)
		END

GO

CopyPermission 'NameMapUpdate', 'mgsOSBreakdown'
GO 

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'NameMapUpdateWithProperties' AND xtype = 'P'
)
        DROP PROCEDURE NameMapUpdateWithProperties
GO

CREATE PROCEDURE [NameMapUpdateWithProperties]
	@computerName nvarchar(64),
	@computerDomain nvarchar(100),
	@newName nvarchar(64),
	@newDomain nvarchar(100),
	@id int,
	@ipAddress nvarchar(32),
	@macAddress nvarchar(32),
	@newOU nvarchar(512)
AS
	DECLARE @DomainID AS int
	DECLARE @newDomainID AS int
	DECLARE @newOUID AS int

	SELECT @DomainID = OrganizationID FROM Domain
	WHERE DN = @computerDomain

	SELECT @newDomainID = OrganizationID FROM Domain
	WHERE DN = @newDomain
	
	SELECT @newOUID = OrganizationID FROM Organization
	WHERE RDN = @newOU 
	  AND DomainID = @newDomainID

	-- this condition updates a computer that exists in the 
	-- computer table (and therefore in Active Directory)
	IF EXISTS(SELECT * 
		FROM NameMap as nm, Domain as d
		WHERE nm.CurrentID = @id
		  AND @id IS NOT NULL)
		BEGIN
			UPDATE NameMap
			SET ipAddress = @ipAddress,
			    macAddress = @macAddress,
			    newOU = NULL
			WHERE CurrentId = @id
		END
	-- this condition updates name mappings for new computers that
	-- do not exist in the Computer table or in Active Directory
	ELSE IF EXISTS (SELECT * 
		FROM NameMap as nm
		WHERE nm.[name] IS NULL
		  AND nm.domain IS NULL
		  AND nm.CurrentID IS NULL
		  AND nm.newOU = @newOUID
		  AND nm.newName = @newName
		  AND nm.newDomain = @newDomainID)
		BEGIN
			UPDATE NameMap
			SET ipAddress = @ipAddress,
			    macAddress = @macAddress
			WHERE [name] IS NULL
			AND domain IS NULL
			AND CurrentID IS NULL
			AND newOU = @newOUID
			AND newName = @newName
			AND newDomain = @newDomainID
		END
	-- This final case inserts new name mappings into the table
	--
	ELSE
		BEGIN
			INSERT INTO NameMap
			(CurrentId, [name], domain, newName, newDomain, ipAddress, macAddress, newOU)
			VALUES
			(@id, @computerName, @DomainID, @newName, @newDomainID, @ipAddress, @macAddress, @newOUID)
		END

GO

CopyPermission 'NameMapUpdateWithProperties', 'mgsOSBreakdown'
GO 



IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'NameMapLookupUpdateWithProperties' AND xtype = 'P'
)
        DROP PROCEDURE NameMapLookupUpdateWithProperties
GO

CREATE PROCEDURE [NameMapLookupUpdateWithProperties]
	@computerName nvarchar(64),
	@computerDomain nvarchar(100),
	@computerOU nvarchar(512),
	@newName nvarchar(64),
	@newDomain nvarchar(100),
	@ipAddress nvarchar(32),
	@macAddress nvarchar(32),
	@newOU nvarchar(512),
	@errorLevel int OUTPUT
AS
	DECLARE @DomainID AS int
	DECLARE @newDomainID AS int
	DECLARE @newOUID AS int
	DECLARE @id AS int
	SELECT @errorLevel = NULL

	SELECT @DomainID = OrganizationID FROM Domain
	WHERE DN = @computerDomain

	SELECT @newDomainID = OrganizationID FROM Domain
	WHERE DN = @newDomain
	
	SELECT @newOUID = OrganizationID FROM Organization
	WHERE RDN = @newOU 
	  AND DomainID = @newDomainID

	IF @computerOU IS NOT NULL
	BEGIN	
		SELECT	@id = ComputerID
		FROM	ComputerDirectory
		WHERE	DomainID = @DomainID
		AND	ComputerCN = @computerName
		AND	OURDN = @computerOU
		AND	DomainID IS NOT NULL
		AND	ComputerCN IS NOT NULL
		AND	OURDN IS NOT NULL
	END
	ELSE
	BEGIN
		SELECT	@id = ComputerID
		FROM	ComputerDirectory
		WHERE	DomainID = @DomainID
		AND	ComputerCN = @computerName
		AND	DomainID IS NOT NULL
		AND	ComputerCN IS NOT NULL
		AND	OURDN IS NOT NULL
	END

	-- the computer does not meet the rules 
	-- for being an old or new computer
	--	
	IF @id IS NULL AND @newOUID IS NULL
	BEGIN
		SELECT @errorLevel = 1 -- generic invalid row
		RETURN
	END

	-- this row should exist in the computer table
	-- because it has an old name specified	
	IF @computerName IS NOT NULL AND @id IS NULL
	BEGIN
		-- computer should exist, but doesn't
		SELECT @errorLevel = 2
		RETURN
	END
	
	-- This is not an error case, but we don't want to update the
	-- database if all we are doing is adding an existing 
	-- computer with no properties set.
	-- 
	IF @id IS NOT NULL 
		AND @macAddress IS NULL 
		AND @ipAddress IS NULL 
		AND @newName IS NULL
	BEGIN
		RETURN
	END

	-- this condition updates a computer that exists in the 
	-- computer table (and therefore in Active Directory)
	IF EXISTS(SELECT * 
		FROM NameMap as nm, Domain as d
		WHERE nm.CurrentID = @id
		  AND @id IS NOT NULL)
		BEGIN
			UPDATE NameMap
			SET ipAddress = @ipAddress,
			    macAddress = @macAddress,
			    newOU = NULL
			WHERE CurrentId = @id
		END
	-- this condition updates name mappings for new computers that
	-- do not exist in the Computer table or in Active Directory
	ELSE IF EXISTS (SELECT * 
		FROM NameMap as nm
		WHERE nm.[name] IS NULL
		  AND nm.domain IS NULL
		  AND nm.CurrentID IS NULL
		  AND nm.newOU = @newOUID
		  AND nm.newName = @newName
		  AND nm.newDomain = @newDomainID)
		BEGIN
			UPDATE NameMap
			SET ipAddress = @ipAddress,
			    macAddress = @macAddress
			WHERE [name] IS NULL
			AND domain IS NULL
			AND CurrentID IS NULL
			AND newOU = @newOUID
			AND newName = @newName
			AND newDomain = @newDomainID
		END
	-- This final case inserts new name mappings into the table
	--
	ELSE
		BEGIN
			-- set the new OUID to null if the computer exists in AD
			IF @id IS NOT NULL
			BEGIN
				SELECT @newOUID = NULL
			END
			INSERT INTO NameMap
			(CurrentId, [name], domain, newName, newDomain, ipAddress, macAddress, newOU)
			VALUES
			(@id, @computerName, @DomainID, @newName, @newDomainID, @ipAddress, @macAddress, @newOUID)
		END

GO

CopyPermission 'NameMapLookupUpdateWithProperties', 'mgsOSBreakdown'
GO 



/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiNameMap.sql,v 1.13 2007/07/24 06:30:56 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: RolloutGroupGhostComputerEntry
 *	Input:
 *	- @groupID: GroupID of a group
 * 	- @serverMAC: MAC Address of computer in the group
 *
 *	Returns:
 *	- GroupID, ComputerName, PhysicalAddress, IsResolved
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'GetHardwareProperty' AND xtype = 'P'
)
        DROP PROCEDURE GetHardwareProperty
GO

CREATE PROCEDURE [GetHardwareProperty]
	@computerName nvarchar(64),
	@computerDomain nvarchar(100),
	@hardwareClass nvarchar(64),
	@hardwareProperty nvarchar(50)
AS
	SELECT hv.Value
	FROM	HardwareClass AS hc,			-- HW class names
		HardwareObject AS ho,
		HardwareProperty as hp,			-- Property types
		HardwareValue as hv,
		Computer c,
		Organization AS o,
		Domain AS d
	WHERE	hv.HardwarePropertyID = hp.HardwarePropertyID
	  AND	hp.Property = @hardwareProperty		-- Right property
	  AND	hv.HardwareObjectID = ho.HardwareObjectID
	  AND	ho.HardwareClassID = hc.HardwareClassID
	  AND	hc.Class = @hardwareClass		-- Right class
	  AND	ho.ComputerID = c.ComputerID
	  AND	c.ComputerCN = @computerName		-- Right computer
	  AND	c.ComputerOUID = o.OrganizationID
	  AND	o.DomainID = d.OrganizationID
	  AND	d.DN = @computerDomain			-- Right domain
GO

CopyPermission 'GetHardwareProperty', 'mgsOSBreakdown'
GO 


-- USE MANAGESOFT
GO


IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'NameMapDuplicate' AND xtype = 'P'
)
        DROP PROCEDURE NameMapDuplicate
GO

CREATE PROCEDURE [NameMapDuplicate]
	@computerName nvarchar(64),
	@computerDomain nvarchar(100)
AS

	DECLARE @newDomainID AS int

	SELECT @newDomainID = OrganizationID FROM Domain
	WHERE DN = @computerDomain

	(
	SELECT c.ComputerCN
	FROM Computer as c, Domain as d, Organization as o
	WHERE c.ComputerOUID = o.OrganizationID
	  AND o.DomainID = d.OrganizationID
	  AND c.ComputerCN = @computerName
	  AND o.DomainID = @newDomainID
	)
	UNION
	(
	SELECT nm.newName
	FROM NameMap as nm, Domain as d
	WHERE nm.newName = @computerName
	  AND d.DN = @computerDomain
	  AND nm.newDomain = d.OrganizationID
	)
	
GO

CopyPermission 'NameMapDuplicate', 'mgsOSBreakdown'
GO 


-- USE MANAGESOFT
GO


IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'NameMapCleanup' AND xtype = 'P'
)
        DROP PROCEDURE NameMapCleanup
GO

CREATE PROCEDURE [NameMapCleanup]
AS

	DELETE FROM NameMap
	WHERE CurrentId NOT IN
	(
		SELECT ComputerId FROM Computer
	)
	AND newOU IS NULL
	
GO

CopyPermission 'NameMapCleanup', 'mgsOSBreakdown'
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'NameMapLink' AND xtype = 'P'
)
        DROP PROCEDURE NameMapLink
GO
CREATE PROCEDURE [NameMapLink]
	@nameMapId int,
	@computerId int
AS
	UPDATE	NameMap
	SET	Name = NewName,
		Domain = NewDomain,
		CurrentID = @computerId,
		NewOU = NULL
	WHERE	ID = @nameMapId
GO

CopyPermission 'NameMapLink', 'mgsOSBreakdown'
GO

/*
 * $Header: /cvsroot/nd/branched/src/database/queries/src/WiMiPersonalityMigration.sql,v 1.2 2007/07/24 06:30:56 pca Exp $
 *
 * DESCRIPTION: Stored Procedure: WiMiMWMPInstallerPackages
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */
 
-- USE MANAGESOFT
GO

IF EXISTS (
        SELECT * FROM dbo.sysobjects
        WHERE [name] = 'WiMiInstallerPackages' AND xtype = 'P'
)
        DROP PROCEDURE WiMiInstallerPackages
GO
 
CREATE PROCEDURE WiMiInstallerPackages
	@pPackageFullName nvarchar(256)
AS
	SELECT PackageFullName FROM PackagePath
	WHERE PackageFullName LIKE @pPackageFullName
GO
/*
 * $Id: HardwareAssessment-views.sql,v 1.3 2007/07/24 06:31:01 pca Exp $
 *
 * DESCRIPTION:
 *
 *	Schema for Windows Migration - Hardware Assessment.
  *	-Creates required VIEWS	
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */

USE ManageSoft
GO

--Creates views corresponding to Assessment Rules
EXECUTE WiMiUpdateHardwareAssessmentViews 
	0 -- create views for all assessment rules
	
GO
/*
 * $Id: site.sql,v 1.2 2002/09/09 02:26:07 ryang Exp $
 *
 * Description:
 *
 * SQL Script to build the ManageSoft site database on Microsoft SQL
 * Server.
 *
 * The steps performed by this script are:
 * 1. Create new database
 * 2. Create database table and initialiase data (skip if the table exists)
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 *
 * Copyright 2011 Flexera Software LLC
 */
USE ManageSoft
GO

-- Create Site table
IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'Site'
)
BEGIN
	-- Create table
	PRINT 'Creating Site table'
	CREATE TABLE Site
	(
		-- Auto-generated identity #
		SiteID INTEGER NOT NULL IDENTITY
			CONSTRAINT PK_Site PRIMARY KEY NONCLUSTERED,
			
		-- Name of site
		SiteName NVARCHAR(64) NULL DEFAULT(NULL),
			
		-- OU with full 'OU=foo, DC=....' form
		OU NVARCHAR(800) NULL DEFAULT(NULL),

		-- IP subnet
		IPSubnet VARCHAR(64) NULL DEFAULT(NULL),

		-- Description
		Description NVARCHAR(256) NULL DEFAULT(NULL),
	);
END
ELSE
	PRINT 'Using existing Site table'
GO
/*
 * $Id: site.sql,v 1.20 2007/07/24 06:30:56 pca Exp $
 *
 * Description:
 *
 * SQL Script to add ManageSoft AD population procedures.
 *
 * The steps performed by this script are:
 * 1. Add AddComputer procedure
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 *
 * Copyright 2011 Flexera Software LLC
 */
 
-- Use ManageSoft database
USE ManageSoft
GO

-- Create 'AddComputer' stored procedure
PRINT 'Creating procedure to add a computer'
IF EXISTS(SELECT * FROM sysobjects WHERE xtype = 'P'
	AND name = 'AddComputer'
)
	DROP PROCEDURE AddComputer
GO
CREATE PROCEDURE AddComputer
	@ComputerID INTEGER OUT,
	@ComputerCN NVARCHAR(64),
	@ComputerOU NVARCHAR(800),
	@OperatingSystem NVARCHAR(128) = NULL,
	@ADComputerGUID BINARY(16) = NULL,
	@ADComputerOUGUID BINARY(16) = NULL
AS
BEGIN
	
	-- Calculate DC of OU
	DECLARE @OUDC NVARCHAR(100)
	SET @OUDC = SUBSTRING(
		@ComputerOU,
		CHARINDEX('DC=', @ComputerOU, 0),
		LEN(@ComputerOU) - CHARINDEX('DC=', @ComputerOU, 0) + 1
	)
	
	-- Calculate RDN of OU
	DECLARE @OURDN NVARCHAR(400)
	SET @OURDN = SUBSTRING(
		@ComputerOU,
		1,
		LEN(@ComputerOU) - (LEN(@OUDC) + 1)
	)
	
	-- Create OU in ManageSoft SDOU table if necessary
	DECLARE @OUID INTEGER
	SELECT @OUID = SD.OUID
	FROM (
		SELECT o.OrganizationID as OUID,
			CASE
			    WHEN o.RDN IS NULL THEN	-- Unknown OU in this domain
				'<NULL>,'+d.DN		-- or the universal domain
			    WHEN o.RDN = '' THEN
				d.DN			-- DomainDN for the domain
			    ELSE
				o.RDN+','+d.DN		-- Ordinary OU or container
			END AS OU
		FROM Organization AS o, Domain AS d
		WHERE o.DomainID = d.OrganizationID
	) as SD
	WHERE SD.OU = @ComputerOU
	
	IF (@OUID IS NULL)
	BEGIN
		EXEC OrganizationDNCreate @OURDN, @OUDC, @ADComputerOUGUID
		SELECT @OUID = SD.OUID
		FROM (
			SELECT o.OrganizationID as OUID,
				CASE
				    WHEN o.RDN IS NULL THEN	-- Unknown OU in this domain
					'<NULL>,'+d.DN		-- or the universal domain
				    WHEN o.RDN = '' THEN
					d.DN			-- DomainDN for the domain
				    ELSE
					o.RDN+','+d.DN		-- Ordinary OU or container
				END AS OU
			FROM Organization AS o, Domain AS d
			WHERE o.DomainID = d.OrganizationID
		) as SD
		WHERE SD.OU = @ComputerOU
	END
	
	-- Get domain ID of OU
	DECLARE @DNID INTEGER
	SELECT @DNID = DomainID FROM Organization WHERE OrganizationID = @OUID

	-- Find existing computer in destination OU
	IF @ComputerID IS NULL
	BEGIN
		SELECT TOP 1 @ComputerID = ComputerID
		FROM Computer,
		(
			SELECT	o.OrganizationID as OUID,
				CASE
				    WHEN o.RDN IS NULL THEN	-- Unknown OU in this domain
					'<NULL>,'+d.DN		-- or the universal domain
				    WHEN o.RDN = '' THEN
					d.DN			-- DomainDN for the domain
				    ELSE
					o.RDN+','+d.DN		-- Ordinary OU or container
				END AS OU
			FROM Organization AS o, Domain AS d
			WHERE o.DomainID = d.OrganizationID
		) as SD
		WHERE ComputerCN = @ComputerCN
		AND ComputerOUID = SD.OUID
		AND SD.OU = @ComputerOU
	END
	
	-- Not found, so try computer with same DC
	IF @ComputerID IS NULL
	BEGIN
		-- Look for computer with same DC
		SELECT TOP 1 @ComputerID = ComputerID
		FROM	Computer,
			(
				SELECT	o.OrganizationID as OUID,
					CASE
					    WHEN o.RDN IS NULL THEN	-- Unknown OU in this domain
						'<NULL>,'+d.DN		-- or the universal domain
					    WHEN o.RDN = '' THEN
						d.DN			-- DomainDN for the domain
					    ELSE
						o.RDN+','+d.DN		-- Ordinary OU or container
					END AS OU
				FROM Organization AS o, Domain AS d
				WHERE o.DomainID = d.OrganizationID
			) as SD
		WHERE ComputerCN = @ComputerCN
		AND ComputerOUID = SD.OUID
		AND (@OUDC = SUBSTRING(
			SD.OU,
			CHARINDEX('DC=', SD.OU, 0),
			LEN(SD.OU) - CHARINDEX('DC=', SD.OU, 0) + 1
		))
	END

	-- Determine operating system ID
	DECLARE @OperatingSystemID int
	IF @OperatingSystem IS NOT NULL
	BEGIN
		IF NOT EXISTS (
			SELECT [@OperatingSystemID] = OperatingSystemID
			FROM OperatingSystem
			WHERE OperatingSystemName = @OperatingSystem
		)
		BEGIN
			INSERT INTO OperatingSystem
			(OperatingSystemName) VALUES (@OperatingSystem)

			SELECT @OperatingSystemID = OperatingSystemID
			FROM OperatingSystem
			WHERE OperatingSystemName = @OperatingSystem
		END
	END ELSE SET @OperatingSystemID = NULL

	-- Check to see if computer already exists
	IF @ComputerID IS NOT NULL
	BEGIN
		-- Update existing computer in the ManageSoft computer table
		UPDATE Computer
		SET	ComputerCN = @ComputerCN,
			ComputerOUID = @OUID,
			OperatingSystemID = ISNULL(@OperatingSystemID, OperatingSystemID),
			GUID = ISNULL(@ADComputerGUID, GUID)
		WHERE ComputerID = @ComputerID
		
		-- Update the OU in ManageSoft tables that have links to the computuer
		UPDATE Installation
			SET OrganizationID = @OUID
			WHERE ComputerID = @ComputerID
			AND UserID = 1	-- SYSTEM user
		DELETE FROM PackageApplies
			WHERE TargetTypeID = 1
			AND TargetID = @ComputerID
		
		-- Update name map table
		UPDATE NameMap
		SET NewDomain = @DNID
		WHERE CurrentID = @ComputerID
	END
	ELSE
	BEGIN
		-- Add computer to ManageSoft computer table
		INSERT INTO Computer
		(ComputerCN, ComputerOUID, OperatingSystemID, GUID)
		VALUES (@ComputerCN, @OUID, @OperatingSystemID, @ADComputerGUID)
		SET @ComputerID = @@IDENTITY
	END
END
GO

-- Stored Procedures to Select/Update/Insert/Delete site table
PRINT 'Creating procedures to Select, Update, Insert, Delete site table'
IF EXISTS (
	SELECT * FROM sysobjects 
	WHERE [name] = 'WiMiSiteSelect' AND xtype = 'P'
)
	DROP PROCEDURE WiMiSiteSelect
GO

CREATE PROCEDURE [WiMiSiteSelect]
AS
	SET NOCOUNT ON
	SELECT SiteID, SiteName, OU, IPSubnet, Description FROM Site
GO



IF EXISTS (
	SELECT * FROM sysobjects 
	WHERE [name] = 'WiMiSiteInsert' AND xtype = 'P'
)
	DROP PROCEDURE WiMiSiteInsert
GO

CREATE PROCEDURE [WiMiSiteInsert]
(
	@SiteName nvarchar(64),
	@OU nvarchar(800),
	@IPSubnet varchar(64),
	@Description nvarchar(256)
)
AS
	SET NOCOUNT OFF
	INSERT INTO Site(SiteName, OU, IPSubnet, Description) VALUES (@SiteName, @OU, @IPSubnet, @Description)
	SELECT SiteID, SiteName, OU, IPSubnet, Description FROM Site WHERE (SiteID = @@IDENTITY)
GO



IF EXISTS (
	SELECT * FROM sysobjects 
	WHERE [name] = 'WiMiSiteUpdate' AND xtype = 'P'
)
	DROP PROCEDURE WiMiSiteUpdate
GO

CREATE PROCEDURE [WiMiSiteUpdate]
(
	@SiteName nvarchar(64),
	@OU nvarchar(800),
	@IPSubnet varchar(64),
	@Description nvarchar(256),
	@Original_SiteID int,
	@Original_Description nvarchar(256),
	@Original_IPSubnet varchar(64),
	@Original_OU nvarchar(800),
	@Original_SiteName nvarchar(64),
	@SiteID int
)
AS
	SET NOCOUNT OFF
	UPDATE Site SET SiteName = @SiteName, OU = @OU, IPSubnet = @IPSubnet, Description = @Description WHERE (SiteID = @Original_SiteID) AND (Description = @Original_Description OR @Original_Description IS NULL AND Description IS NULL) AND (IPSubnet = @Original_IPSubnet OR @Original_IPSubnet IS NULL AND IPSubnet IS NULL) AND (OU = @Original_OU OR @Original_OU IS NULL AND OU IS NULL) AND (SiteName = @Original_SiteName OR @Original_SiteName IS NULL AND SiteName IS NULL)
	SELECT SiteID, SiteName, OU, IPSubnet, Description FROM Site WHERE (SiteID = @SiteID)
GO



IF EXISTS (
	SELECT * FROM sysobjects 
	WHERE [name] = 'WiMiSiteDelete' AND xtype = 'P'
)
	DROP PROCEDURE WiMiSiteDelete
GO

CREATE PROCEDURE [WiMiSiteDelete]
(
	@Original_SiteID int,
	@Original_Description nvarchar(256),
	@Original_IPSubnet varchar(64),
	@Original_OU nvarchar(800),
	@Original_SiteName nvarchar(64)
)
AS
	SET NOCOUNT OFF
	DELETE FROM Site WHERE (SiteID = @Original_SiteID) AND (Description = @Original_Description OR @Original_Description IS NULL AND Description IS NULL) AND (IPSubnet = @Original_IPSubnet OR @Original_IPSubnet IS NULL AND IPSubnet IS NULL) AND (OU = @Original_OU OR @Original_OU IS NULL AND OU IS NULL) AND (SiteName = @Original_SiteName OR @Original_SiteName IS NULL AND SiteName IS NULL)
GO
/*
 * $Id: imagelibrary.sql,v 1.7 2007/07/24 06:31:02 pca Exp $
 *
 * Description:
 *
 * SQL Script to build the ManageSoft image tables on Microsoft SQL
 * Server.
 *
 * The steps performed by this script are:
 * 1. Create new database
 * 2. Create database table and initialiase data (skip if the table exists)
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 *
 * Copyright 2011 Flexera Software LLC
 */
USE ManageSoft
GO

-- Create OSImageFolder table
IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'OSImageFolder'
)
BEGIN
	-- Create table
	PRINT 'Creating OSImageFolder table'
	CREATE TABLE OSImageFolder
	(
		-- Auto-generated identity #
		OSImageFolderID INTEGER NOT NULL IDENTITY
			CONSTRAINT PK_OSImageFolder PRIMARY KEY NONCLUSTERED,

		-- GUID of image folder
		ImageFolderGUID BINARY(16) NULL DEFAULT(NULL),
			
		-- Path of ImageFolder
		ImageFolderPath INTEGER REFERENCES OSImageFolder(OSImageFolderID),
			
		-- Name of ImageFolder
		ImageFolderName NVARCHAR(256) NULL DEFAULT(NULL)
	);
	
	-- Create indices
	CREATE UNIQUE INDEX OSImageFolderByGUID
	ON OSImageFolder (ImageFolderGUID)
	CREATE UNIQUE INDEX OSImageFolderByPathName
	ON OSImageFolder (ImageFolderPath, ImageFolderName)
	CREATE INDEX OSImageFolderByPath
	ON OSImageFolder (ImageFolderPath)
END
ELSE
	PRINT 'Using existing OSImageFolder table'
GO

-- Create OSImage table
IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'OSImage'
)
BEGIN
	-- Create table
	PRINT 'Creating OSImage table'
	CREATE TABLE OSImage
	(
		-- Auto-generated identity #
		OSImageID INTEGER NOT NULL IDENTITY
			CONSTRAINT PK_OSImage PRIMARY KEY NONCLUSTERED,

		-- GUID of image
		ImageGUID BINARY(16) NULL DEFAULT(NULL),
		
		-- GUID of image version
		ImageVersionGUID BINARY(16) NULL DEFAULT(NULL),
			
		-- Path of image
		ImagePath INTEGER NULL REFERENCES OSImageFolder(OSImageFolderID),
			
		-- Name of image
		ImageName NVARCHAR(256) NOT NULL,
		
		-- Version of image
		ImageVersion NVARCHAR(64) NOT NULL,
		
		-- Modify date and time of image
		ImportDateTime DATETIME NOT NULL,
		
		-- Operating system type
		OperatingSystem NVARCHAR(64) NULL DEFAULT(NULL),
		
		-- Type of image (unattend, sysprep)
		ImagingType NVARCHAR(64) NULL DEFAULT(NULL),
		
		-- File system of image
		FileSystem NVARCHAR(64) NULL DEFAULT(NULL),
		
		-- Size of image (in MB)
		ImageSize INTEGER NULL DEFAULT(NULL),
		
		-- Description of image
		Description NVARCHAR(256) NULL DEFAULT(NULL),
		
		-- Path of image in repository
		RepositoryPath NVARCHAR(256) NULL DEFAULT(NULL),

		-- Path of OS image file repository
		OSImageFile NVARCHAR(256) NULL DEFAULT(NULL),

		-- Whether or not the OSImage is active from the User Select Menu
		Active BIT Not NULL DEFAULT(1),

		-- The Image Command string
		ImageCommand NVARCHAR(256) NULL DEFAULT(NULL),

		-- The drive letter to use for the image
		ImageDrive NVARCHAR(16) NULL DEFAULT (NULL),

		-- The image index for WIM files
		ImageIndex INTEGER NULL DEFAULT(NULL)
		
	);
	
	-- Create indices
	CREATE UNIQUE INDEX OSImageByGUID
	ON OSImage (ImageGUID)
	CREATE UNIQUE INDEX OSImageByPathName
	ON OSImage (ImagePath, ImageName)
	CREATE INDEX OSImageByPath
	ON OSImage (ImagePath)
END
ELSE
	PRINT 'Using existing OSImage table'
GO
/*
 * $Id: imagelibrary.sql,v 1.6 2007/07/24 06:30:56 pca Exp $
 *
 * Description:
 *
 * SQL Script to build the ManageSoft image library stored procedures
 * on Microsoft SQL Server.
 *
 * The steps performed by this script are:
 * 1. Create new database
 * 2. Create database table and initialiase data (skip if the table exists)
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 *
 * Copyright 2011 Flexera Software LLC
 */
USE ManageSoft
GO

-- Procedure for selecting an OS image folder
PRINT 'Creating procedure to select OS image folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'OSImageFolderSelect' AND xtype = 'P')
	DROP PROCEDURE [OSImageFolderSelect];
GO
CREATE PROCEDURE [OSImageFolderSelect]
(
	@ImageFolderGUID binary(16)
)
AS
	SET NOCOUNT ON;
	SELECT A.OSImageFolderID, A.ImageFolderGUID, A.ImageFolderPath, A.ImageFolderName,
		COUNT(B.ImageFolderPath) AS ChildImageFolderCount, COUNT(C.ImagePath) AS ChildImageCount
		FROM OSImageFolder AS A
		LEFT OUTER JOIN OSImageFolder AS B
		ON A.OSImageFolderID = B.ImageFolderPath
		LEFT OUTER JOIN OSImage AS C
		ON A.OSImageFolderID = C.ImagePath
		WHERE (A.ImageFolderGUID = @ImageFolderGUID)
		GROUP BY A.OSImageFolderID, A.ImageFolderGUID, A.ImageFolderPath, A.ImageFolderName;
GO

-- Procedure for selecting an OS image folder
PRINT 'Creating procedure to select an OS image folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'OSImageFolderSelectID' AND xtype = 'P')
	DROP PROCEDURE [OSImageFolderSelectID];
GO
CREATE PROCEDURE [OSImageFolderSelectID]
(
	@OSImageFolderID int
)
AS
	SET NOCOUNT ON;
	SELECT A.OSImageFolderID, A.ImageFolderGUID, A.ImageFolderPath, A.ImageFolderName,
		COUNT(B.ImageFolderPath) AS ChildImageFolderCount, COUNT(C.ImagePath) AS ChildImageCount
		FROM OSImageFolder AS A
		LEFT OUTER JOIN OSImageFolder AS B
		ON A.OSImageFolderID = B.ImageFolderPath
		LEFT OUTER JOIN OSImage AS C
		ON A.OSImageFolderID = C.ImagePath
		WHERE (A.OSImageFolderID = @OSImageFolderID)
		GROUP BY A.OSImageFolderID, A.ImageFolderGUID, A.ImageFolderPath, A.ImageFolderName;
GO

-- Procedure for selecting child OS image folders
PRINT 'Creating procedure to select child OS image folders'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'OSImageFolderChildSelect' AND xtype = 'P')
	DROP PROCEDURE [OSImageFolderChildSelect];
GO
CREATE PROCEDURE [OSImageFolderChildSelect]
(
	@ImageFolderPath int = NULL
)
AS
	SET NOCOUNT ON;
	IF (@ImageFolderPath IS NULL)
		SELECT A.OSImageFolderID, A.ImageFolderGUID, A.ImageFolderPath, A.ImageFolderName,
			COUNT(B.ImageFolderPath) AS ChildImageFolderCount, COUNT(C.ImagePath) AS ChildImageCount
			FROM OSImageFolder AS A
			LEFT OUTER JOIN OSImageFolder AS B
			ON A.OSImageFolderID = B.ImageFolderPath
			LEFT OUTER JOIN OSImage AS C
			ON A.OSImageFolderID = C.ImagePath
			WHERE (A.ImageFolderPath IS NULL)
			GROUP BY A.OSImageFolderID, A.ImageFolderGUID, A.ImageFolderPath, A.ImageFolderName
			ORDER BY A.ImageFolderName;
	ELSE
		SELECT A.OSImageFolderID, A.ImageFolderGUID, A.ImageFolderPath, A.ImageFolderName,
			COUNT(B.ImageFolderPath) AS ChildImageFolderCount, COUNT(C.ImagePath) AS ChildImageCount
			FROM OSImageFolder AS A
			LEFT OUTER JOIN OSImageFolder AS B
			ON A.OSImageFolderID = B.ImageFolderPath
			LEFT OUTER JOIN OSImage AS C
			ON A.OSImageFolderID = C.ImagePath
			WHERE (A.ImageFolderPath = @ImageFolderPath)
			GROUP BY A.OSImageFolderID, A.ImageFolderGUID, A.ImageFolderPath, A.ImageFolderName
			ORDER BY A.ImageFolderName;
GO

-- Procedure for inserting an OS image folder
PRINT 'Creating procedure to insert an OS image folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'OSImageFolderInsert' AND xtype = 'P')
	DROP PROCEDURE [OSImageFolderInsert];
GO
CREATE PROCEDURE [OSImageFolderInsert]
(
	@ImageFolderGUID binary(16),
	@ImageFolderPath int,
	@ImageFolderName nvarchar(256)
)
AS
	SET NOCOUNT OFF;
	INSERT INTO OSImageFolder (ImageFolderGUID, ImageFolderPath, ImageFolderName)
		VALUES (@ImageFolderGUID, @ImageFolderPath, @ImageFolderName);
	SELECT A.OSImageFolderID, A.ImageFolderGUID, A.ImageFolderPath, A.ImageFolderName,
		COUNT(B.ImageFolderPath) AS ChildImageFolderCount, COUNT(C.ImagePath) AS ChildImageCount
		FROM OSImageFolder AS A
		LEFT OUTER JOIN OSImageFolder AS B
		ON A.OSImageFolderID = B.ImageFolderPath
		LEFT OUTER JOIN OSImage AS C
		ON A.OSImageFolderID = C.ImagePath
		WHERE (A.OSImageFolderID = @@IDENTITY)
		GROUP BY A.OSImageFolderID, A.ImageFolderGUID, A.ImageFolderPath, A.ImageFolderName;
GO

-- Procedure for updating an OS image folder
PRINT 'Creating procedure to update an OS image folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'OSImageFolderUpdate' AND xtype = 'P')
	DROP PROCEDURE [OSImageFolderUpdate];
GO
CREATE PROCEDURE [OSImageFolderUpdate]
(
	@ImageFolderGUID binary(16),
	@ImageFolderPath int,
	@ImageFolderName nvarchar(256),
	@Original_OSImageFolderID int,
	@Original_ImageFolderGUID binary(16),
	@Original_ImageFolderName nvarchar(256),
	@Original_ImageFolderPath int,
	@OSImageFolderID int
)
AS
	SET NOCOUNT OFF;
	UPDATE OSImageFolder
		SET ImageFolderGUID = @ImageFolderGUID, ImageFolderPath = @ImageFolderPath, ImageFolderName = @ImageFolderName
		WHERE (OSImageFolderID = @Original_OSImageFolderID)
		AND (ImageFolderGUID = @Original_ImageFolderGUID OR @Original_ImageFolderGUID IS NULL AND ImageFolderGUID IS NULL)
		AND (ImageFolderName = @Original_ImageFolderName OR @Original_ImageFolderName IS NULL AND ImageFolderName IS NULL)
		AND (ImageFolderPath = @Original_ImageFolderPath OR @Original_ImageFolderPath IS NULL AND ImageFolderPath IS NULL);
	SELECT A.OSImageFolderID, A.ImageFolderGUID, A.ImageFolderPath, A.ImageFolderName,
		COUNT(B.ImageFolderPath) AS ChildImageFolderCount, COUNT(C.ImagePath) AS ChildImageCount
		FROM OSImageFolder AS A
		LEFT OUTER JOIN OSImageFolder AS B
		ON A.OSImageFolderID = B.ImageFolderPath
		LEFT OUTER JOIN OSImage AS C
		ON A.OSImageFolderID = C.ImagePath
		WHERE (A.OSImageFolderID = @OSImageFolderID)
		GROUP BY A.OSImageFolderID, A.ImageFolderGUID, A.ImageFolderPath, A.ImageFolderName;
GO

-- Procedure for deleting an OS image folder
PRINT 'Creating procedure to delete an OS image folder'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'OSImageFolderDelete' AND xtype = 'P')
	DROP PROCEDURE [OSImageFolderDelete];
GO

CREATE PROCEDURE [OSImageFolderDelete]
(
	@Original_OSImageFolderID int,
	@Original_ImageFolderGUID binary(16),
	@Original_ImageFolderName nvarchar(256),
	@Original_ImageFolderPath int
)
AS
	SET NOCOUNT OFF;
	DELETE FROM OSImageFolder
		WHERE (OSImageFolderID = @Original_OSImageFolderID)
		AND (ImageFolderGUID = @Original_ImageFolderGUID OR @Original_ImageFolderGUID IS NULL AND ImageFolderGUID IS NULL)
		AND (ImageFolderName = @Original_ImageFolderName OR @Original_ImageFolderName IS NULL AND ImageFolderName IS NULL)
		AND (ImageFolderPath = @Original_ImageFolderPath OR @Original_ImageFolderPath IS NULL AND ImageFolderPath IS NULL);
GO

-- Procedure for selecting an OS image
PRINT 'Creating procedure to select an OS image'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'OSImageSelect' AND xtype = 'P')
	DROP PROCEDURE [OSImageSelect];
GO

CREATE PROCEDURE [OSImageSelect]
(
	@ImageGUID binary(16)
)
AS
	SET NOCOUNT ON;
	SELECT OSImageID, ImageGUID, ImageVersionGUID, ImagePath, ImageName, ImageVersion, ImportDateTime, OperatingSystem,
		ImagingType, FileSystem, ImageSize, Description, RepositoryPath, OSImageFile, Active, ImageCommand, ImageDrive,
		ImageIndex
		FROM OSImage WHERE (ImageGUID = @ImageGUID);
GO

-- Procedure for selecting an OS image
PRINT 'Creating procedure to select an OS image'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'OSImageSelectID' AND xtype = 'P')
	DROP PROCEDURE [OSImageSelectID];
GO
CREATE PROCEDURE [OSImageSelectID]
(
	@OSImageID int
)
AS
	SET NOCOUNT ON;
	SELECT OSImageID, ImageGUID, ImageVersionGUID, ImagePath, ImageName, ImageVersion, ImportDateTime, OperatingSystem,
		ImagingType, FileSystem, ImageSize, Description, RepositoryPath, OSImageFile, Active, ImageCommand, ImageDrive,
		ImageIndex
		FROM OSImage WHERE (OSImageID = @OSImageID);
GO

-- Procedure for selecting child OS images
PRINT 'Creating procedure to select child OS images'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'OSImageChildSelect' AND xtype = 'P')
	DROP PROCEDURE [OSImageChildSelect];
GO
CREATE PROCEDURE [OSImageChildSelect]
(
	@ImagePath int = NULL
)
AS
	SET NOCOUNT ON;
	IF (@ImagePath IS NULL)
		SELECT OSImageID, ImageGUID, ImageVersionGUID, ImagePath, ImageName, ImageVersion, ImportDateTime, OperatingSystem,
			ImagingType, FileSystem, ImageSize, Description, RepositoryPath, OSImageFile, Active, ImageCommand, ImageDrive,
			ImageIndex
			FROM OSImage
			WHERE (ImagePath IS NULL)
			ORDER BY ImageName;
	ELSE
		SELECT OSImageID, ImageGUID, ImageVersionGUID, ImagePath, ImageName, ImageVersion, ImportDateTime, OperatingSystem,
			ImagingType, FileSystem, ImageSize, Description, RepositoryPath, OSImageFile, Active, ImageCommand, ImageDrive,
			ImageIndex
			FROM OSImage
			WHERE (ImagePath = @ImagePath)
			ORDER BY ImageName;
GO

-- Procedure for inserting an OS image
PRINT 'Creating procedure to insert an OS image'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'OSImageInsert' AND xtype = 'P')
	DROP PROCEDURE [OSImageInsert];
GO
CREATE PROCEDURE [OSImageInsert]
(
	@ImageGUID binary(16),
	@ImageVersionGUID binary(16),
	@ImagePath int,
	@ImageName nvarchar(256),
	@ImageVersion nvarchar(64),
	@ImportDateTime datetime,
	@OperatingSystem nvarchar(64),
	@ImagingType nvarchar(64),
	@FileSystem nvarchar(64),
	@ImageSize int,
	@Description nvarchar(256),
	@RepositoryPath nvarchar(256),
	@OSImageFile nvarchar(256),
	@Active bit,
	@ImageCommand nvarchar(256),
	@ImageDrive nvarchar(16),
	@ImageIndex int
)
AS
	SET NOCOUNT OFF;
	INSERT INTO OSImage (ImageGUID, ImageVersionGUID, ImagePath, ImageName, ImageVersion, ImportDateTime, OperatingSystem,
		ImagingType, FileSystem, ImageSize, Description, RepositoryPath, OSImageFile, Active, ImageCommand, ImageDrive, ImageIndex)
		VALUES (@ImageGUID, @ImageVersionGUID, @ImagePath, @ImageName, @ImageVersion, @ImportDateTime, @OperatingSystem,
		@ImagingType, @FileSystem, @ImageSize, @Description, @RepositoryPath, @OSImageFile, @Active, @ImageCommand, @ImageDrive,
		@ImageIndex);
	SELECT OSImageID, ImageGUID, ImageVersionGUID, ImagePath, ImageName, ImageVersion, ImportDateTime, OperatingSystem,
		ImagingType, FileSystem, ImageSize, Description, RepositoryPath, OSImageFile, Active, ImageCommand, ImageDrive, ImageIndex
		FROM OSImage
		WHERE (OSImageID = @@IDENTITY);
GO

-- Procedure for updating an OS image
PRINT 'Creating procedure to update an OS image'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'OSImageUpdate' AND xtype = 'P')
	DROP PROCEDURE [OSImageUpdate];
GO
CREATE PROCEDURE [OSImageUpdate]
(
	@ImageGUID binary(16),
	@ImageVersionGUID binary(16),
	@ImagePath int,
	@ImageName nvarchar(256),
	@ImageVersion nvarchar(64),
	@ImportDateTime datetime,
	@OperatingSystem nvarchar(64),
	@ImagingType nvarchar(64),
	@FileSystem nvarchar(64),
	@ImageSize int,
	@Description nvarchar(256),
	@RepositoryPath nvarchar(256),
	@OSImageFile nvarchar(256),
	@Active bit,
	@ImageCommand nvarchar(256),
	@ImageDrive nvarchar(16),
	@ImageIndex int,
	@Original_OSImageID int,
	@Original_Description nvarchar(256),
	@Original_RepositoryPath nvarchar(256),
	@Original_FileSystem nvarchar(64),
	@Original_ImageGUID binary(16),
	@Original_ImageVersionGUID binary(16),
	@Original_ImageName nvarchar(256),
	@Original_ImagePath int,
	@Original_ImageSize int,
	@Original_ImageVersion nvarchar(64),
	@Original_ImagingType nvarchar(64),
	@Original_ImportDateTime datetime,
	@Original_OperatingSystem nvarchar(64),
	@Original_OSImageFile nvarchar(256),
	@Original_Active bit,
	@Original_ImageCommand nvarchar(256),
	@Original_ImageDrive nvarchar(16),
	@Original_ImageIndex int,
	@OSImageID int
)
AS
	SET NOCOUNT OFF;
	UPDATE OSImage
		SET ImageGUID = @ImageGUID, ImageVersionGUID = @ImageVersionGUID, ImagePath = @ImagePath, ImageName = @ImageName,
		ImageVersion = @ImageVersion, ImportDateTime = @ImportDateTime, OperatingSystem = @OperatingSystem,
		ImagingType = @ImagingType, FileSystem = @FileSystem, ImageSize = @ImageSize, Description = @Description, RepositoryPath = @RepositoryPath, OSImageFile = @OSImageFile, Active = @Active, ImageCommand = @ImageCommand, ImageDrive = @ImageDrive, ImageIndex = @ImageIndex
		WHERE (OSImageID = @Original_OSImageID)
		AND (Description = @Original_Description OR @Original_Description IS NULL AND Description IS NULL)
		AND (RepositoryPath = @Original_RepositoryPath OR @Original_RepositoryPath IS NULL AND RepositoryPath IS NULL)
		AND (OSImageFile = @Original_OSImageFile OR @Original_OSImageFile IS NULL AND OSImageFile IS NULL)
		AND (Active = @Original_Active OR @Original_Active IS NULL AND Active IS NULL)
		AND (ImageCommand = @Original_ImageCommand OR @Original_ImageCommand IS NULL AND ImageCommand IS NULL)
		AND (ImageDrive = @Original_ImageDrive OR @Original_ImageDrive IS NULL AND ImageDrive IS NULL)
		AND (ImageIndex = @Original_ImageIndex OR @Original_ImageIndex IS NULL AND ImageIndex IS NULL)
		AND (FileSystem = @Original_FileSystem OR @Original_FileSystem IS NULL AND FileSystem IS NULL)
		AND (ImageGUID = @Original_ImageGUID OR @Original_ImageGUID IS NULL AND ImageGUID IS NULL)
		AND (ImageVersionGUID = @Original_ImageVersionGUID OR @Original_ImageVersionGUID IS NULL AND ImageVersionGUID IS NULL)
		AND (ImageName = @Original_ImageName)
		AND (ImagePath = @Original_ImagePath OR @Original_ImagePath IS NULL AND ImagePath IS NULL)
		AND (ImageSize = @Original_ImageSize OR @Original_ImageSize IS NULL AND ImageSize IS NULL)
		AND (ImageVersion = @Original_ImageVersion)
		AND (ImagingType = @Original_ImagingType OR @Original_ImagingType IS NULL AND ImagingType IS NULL)
		AND (ImportDateTime = @Original_ImportDateTime)
		AND (OperatingSystem = @Original_OperatingSystem OR @Original_OperatingSystem IS NULL AND OperatingSystem IS NULL);
	SELECT OSImageID, ImageGUID, ImageVersionGUID, ImagePath, ImageName, ImageVersion, ImportDateTime, OperatingSystem,
		ImagingType, FileSystem, ImageSize, Description, RepositoryPath, OSImageFile, Active, ImageCommand, ImageDrive, ImageIndex
		FROM OSImage
		WHERE (OSImageID = @OSImageID);
GO

-- Procedure for deleting an OS image
PRINT 'Creating procedure to delete an OS image'
IF EXISTS (SELECT * FROM sysobjects WHERE name = 'OSImageDelete' AND xtype = 'P')
	DROP PROCEDURE [OSImageDelete];
GO
CREATE PROCEDURE [OSImageDelete]
(
	@Original_OSImageID int,
	@Original_Description nvarchar(256),
	@Original_RepositoryPath nvarchar(256),
	@Original_FileSystem nvarchar(64),
	@Original_ImageGUID binary(16),
	@Original_ImageVersionGUID binary(16),
	@Original_ImageName nvarchar(256),
	@Original_ImagePath int,
	@Original_ImageSize int,
	@Original_ImageVersion nvarchar(64),
	@Original_ImagingType nvarchar(64),
	@Original_ImportDateTime datetime,
	@Original_OperatingSystem nvarchar(64),
	@Original_OSImageFile nvarchar(256),
	@Original_Active bit,
	@Original_ImageCommand nvarchar(256),
	@Original_ImageDrive nvarchar(16),
	@Original_ImageIndex int
)
AS
	SET NOCOUNT OFF;
	DELETE FROM OSImage
		WHERE (OSImageID = @Original_OSImageID)
		AND (Description = @Original_Description OR @Original_Description IS NULL AND Description IS NULL)
		AND (RepositoryPath = @Original_RepositoryPath OR @Original_RepositoryPath IS NULL AND RepositoryPath IS NULL)
		AND (OSImageFile = @Original_OSImageFile OR @Original_OSImageFile IS NULL AND OSImageFile IS NULL)
		AND (Active = @Original_Active OR @Original_Active IS NULL AND Active IS NULL)
		AND (ImageCommand = @Original_ImageCommand OR @Original_ImageCommand IS NULL AND ImageCommand IS NULL)
		AND (ImageDrive = @Original_ImageDrive OR @Original_ImageDrive IS NULL AND ImageDrive IS NULL)
		AND (ImageIndex = @Original_ImageIndex OR @Original_ImageIndex IS NULL AND ImageIndex IS NULL)
		AND (FileSystem = @Original_FileSystem OR @Original_FileSystem IS NULL AND FileSystem IS NULL)
		AND (ImageGUID = @Original_ImageGUID OR @Original_ImageGUID IS NULL AND ImageGUID IS NULL)
		AND (ImageVersionGUID = @Original_ImageVersionGUID OR @Original_ImageVersionGUID IS NULL AND ImageVersionGUID IS NULL)
		AND (ImageName = @Original_ImageName)
		AND (ImagePath = @Original_ImagePath OR @Original_ImagePath IS NULL AND ImagePath IS NULL)
		AND (ImageSize = @Original_ImageSize OR @Original_ImageSize IS NULL AND ImageSize IS NULL)
		AND (ImageVersion = @Original_ImageVersion)
		AND (ImagingType = @Original_ImagingType OR @Original_ImagingType IS NULL AND ImagingType IS NULL)
		AND (ImportDateTime = @Original_ImportDateTime)
		AND (OperatingSystem = @Original_OperatingSystem OR @Original_OperatingSystem IS NULL AND OperatingSystem IS NULL);
GO

