/*
 * $Id: mwd-migpost79.sql 101389 2011-11-25 01:51:25Z rlowery $
 *
 * Finish migrating a OS Deployment Manager 7.2 database into OS Deployment Manager 7.9 format.
 *
 * Called after building the new tables, this script:
 * - migrates data from old tables
 *
 * Copyright 2011 Flexera Software LLC
 */

USE ManageSoft
GO

/*
 * Check that we are upgrading from a 7.2 database...
 */
DECLARE @schema_version char(10)
SET @schema_version = (SELECT Value FROM DatabaseConfiguration
		WHERE Property = 'WiDepSchemaVersion')

IF (@schema_version = '7.2')
BEGIN
	PRINT 'Completing migration to OS Deployment Manager 7.9 database schema...'
END
ELSE IF (@schema_version = '7.9')
BEGIN
	RAISERROR ('Your OS Deployment Manager database has already been migrated to the 7.9 schema.', 16, 1)
	RAISERROR ('For more information on migrating your database, refer to the OS Deployment Guide.', 16, 1)
END
ELSE
BEGIN
	RAISERROR ('This script can only migrate from the OS Deployment Manager database schema version 7.2.', 16, 1)
	RAISERROR ('Please migrate your database schema to version 7.2 before executing this script again.', 16, 1)
	RAISERROR ('For more information on migrating your database, refer to the MOS Deployment Guide.', 16, 1)
END
GO

-- Set all the Active flags of the OSImage field to 0 (false)
-- as all images currently imported prior to 7.9 should not be exposed to the ImageMenu
IF EXISTS(SELECT * FROM DatabaseConfiguration WHERE Property = 'MigrationStep' AND VALUE = 1000)
BEGIN
	Update OSImage Set Active = 0

	UPDATE DatabaseConfiguration SET Value = 1001 WHERE Property = 'MigrationStep'
END
GO

-- Update the RolloutStatusTypeID column to be 1 in the RolloutStatus table
IF EXISTS(SELECT * FROM DatabaseConfiguration WHERE Property = 'MigrationStep' AND VALUE = 1001)
BEGIN
	-- Set it all to 1 as this is the default
	Update RolloutStatus Set RolloutStatusTypeID = 1

	UPDATE DatabaseConfiguration SET Value = 1002 WHERE Property = 'MigrationStep'
END
GO

-- Update the RolloutStatusTypeID column to be 1 in the RolloutStatusSummary table
IF EXISTS(SELECT * FROM DatabaseConfiguration WHERE Property = 'MigrationStep' AND VALUE = 1002)
BEGIN
	-- Set it all to 1 as this is the default
	Update RolloutStatusSummary Set RolloutStatusTypeID = 1

	UPDATE DatabaseConfiguration SET Value = 1003 WHERE Property = 'MigrationStep'
END
GO

-- Set the foreign key values so that it now does not accept null values
IF EXISTS(SELECT * FROM DatabaseConfiguration WHERE Property = 'MigrationStep' AND VALUE = 1003)
BEGIN
	IF EXISTS (
		SELECT * FROM INFORMATION_SCHEMA.TABLES
		WHERE TABLE_NAME = 'RolloutStatus'
	) AND EXISTS (
		SELECT *
		FROM INFORMATION_SCHEMA.COLUMNS
		WHERE TABLE_NAME = 'RolloutStatus' AND COLUMN_NAME = 'RolloutStatusTypeID'
	)
	BEGIN
		ALTER TABLE RolloutStatus ALTER COLUMN RolloutStatusTypeID integer NOT NULL
	END

	UPDATE DatabaseConfiguration SET Value = 1004 WHERE Property = 'MigrationStep'
END
GO

-- Set the foreign key values so that it now does not accept null values
IF EXISTS(SELECT * FROM DatabaseConfiguration WHERE Property = 'MigrationStep' AND VALUE = 1004)
BEGIN
	IF EXISTS (
		SELECT * FROM INFORMATION_SCHEMA.TABLES
		WHERE TABLE_NAME = 'RolloutStatusSummary'
	) AND EXISTS (
		SELECT *
		FROM INFORMATION_SCHEMA.COLUMNS
		WHERE TABLE_NAME = 'RolloutStatusSummary' AND COLUMN_NAME = 'RolloutStatusTypeID'
	)
	BEGIN
		ALTER TABLE RolloutStatusSummary ALTER COLUMN RolloutStatusTypeID integer NOT NULL
	END

	UPDATE DatabaseConfiguration SET Value = 1005 WHERE Property = 'MigrationStep'
END
GO

-- Add the foreign key constraints to the RolloutStatus table
IF EXISTS(SELECT * FROM DatabaseConfiguration WHERE Property = 'MigrationStep' AND VALUE = 1005)
BEGIN

	IF NOT EXISTS (
	        SELECT *
	        FROM INFORMATION_SCHEMA.CONSTRAINT_COLUMN_USAGE
	        WHERE CONSTRAINT_NAME = 'FK_RolloutStatus_RolloutStatusType'
	)
	BEGIN
	        ALTER TABLE RolloutStatus
	                ADD CONSTRAINT FK_RolloutStatus_RolloutStatusType
	                FOREIGN KEY(RolloutStatusTypeID)
	                REFERENCES RolloutStatusType(RolloutStatusTypeID) ON DELETE NO ACTION
	END

	UPDATE DatabaseConfiguration SET Value = 1006 WHERE Property = 'MigrationStep'
END
GO

-- Add the foreign key constraints to the RolloutStatusSummary table
IF EXISTS(SELECT * FROM DatabaseConfiguration WHERE Property = 'MigrationStep' AND VALUE = 1006)
BEGIN

	IF NOT EXISTS (
	        SELECT *
	        FROM INFORMATION_SCHEMA.CONSTRAINT_COLUMN_USAGE
	        WHERE CONSTRAINT_NAME = 'FK_RolloutStatusSummary_RolloutStatusType'
	)
	BEGIN
	        ALTER TABLE RolloutStatusSummary
	                ADD CONSTRAINT FK_RolloutStatusSummary_RolloutStatusType
	                FOREIGN KEY(RolloutStatusTypeID)
	                REFERENCES RolloutStatusType(RolloutStatusTypeID) ON DELETE NO ACTION
	END

	UPDATE DatabaseConfiguration SET Value = 1007 WHERE Property = 'MigrationStep'
END
GO

-- Modify primary key on RolloutStatusSummary
IF EXISTS(SELECT * FROM DatabaseConfiguration WHERE Property = 'MigrationStep' AND VALUE = 1007)
BEGIN

	IF EXISTS (
		SELECT * FROM sysindexes
		WHERE name = 'PK_RolloutStatusSummary'
	)
	BEGIN
		ALTER TABLE RolloutStatusSummary
		DROP CONSTRAINT PK_RolloutStatusSummary
	END
	
	ALTER TABLE RolloutStatusSummary
	ADD CONSTRAINT PK_RolloutStatusSummary
		PRIMARY KEY NONCLUSTERED(ComputerID, RolloutStatusTypeID)

	UPDATE DatabaseConfiguration SET Value = 1008 WHERE Property = 'MigrationStep'

END
GO

-- Migrate the RolloutConfigSettings
IF EXISTS(SELECT * FROM DatabaseConfiguration WHERE Property = 'MigrationStep' AND VALUE = 1008)
BEGIN

	IF EXISTS (
		SELECT * FROM INFORMATION_SCHEMA.TABLES
		WHERE TABLE_NAME = 'RolloutConfigSetting'
	)
	BEGIN
	
		-- Create a temp table to perform all the modifications
		CREATE TABLE #TempRolloutConfigSetting (
			RolloutConfigSettingID INT IDENTITY (1, 1) NOT NULL,
			RolloutConfig INT,
			SettingType NVARCHAR(64),
			SettingName NVARCHAR(64),
			SettingValue NVARCHAR(256)
		)

		-- Copy all the data from RolloutConfig table into the temp table
		INSERT INTO #TempRolloutConfigSetting (
			RolloutConfig,
			SettingType,
			SettingName,
			SettingValue
		) SELECT RolloutConfig,
			SettingType,
			SettingName,
			SettingValue
		FROM RolloutConfigSetting

		-- Update the Miscellaneous sections to reflect the new schema
		Update #TempRolloutConfigSetting
			SET SettingType='Miscellaneous'
			WHERE SettingType='Logging' AND SettingName='LoggingEnabled'

		Update #TempRolloutConfigSetting
			SET SettingType='Miscellaneous'
			WHERE SettingType='Logging' AND SettingName='LogFilePath'

		-- Update the Upload share details to reflect the new shema
		Update #TempRolloutConfigSetting
			SET SettingType='Upload', SettingName='Protocol'
			WHERE SettingType='Logging' AND SettingName='UploadLogProtocol'
			
		Update #TempRolloutConfigSetting
			SET SettingType='Upload', SettingName='Share'
			WHERE SettingType='Logging' AND SettingName='UploadLogShare'
		
		Update #TempRolloutConfigSetting
			SET SettingType='Upload', SettingName='Username'
			WHERE SettingType='Logging' AND SettingName='UploadLogUsername'

		Update #TempRolloutConfigSetting
			SET SettingType='Upload', SettingName='Password'
			WHERE SettingType='Logging' AND SettingName='UploadLogPassword'

		-- Create dummy entries for the InheritUserCredentials entry
		INSERT INTO #TempRolloutConfigSetting (
			RolloutConfig,
			SettingType,
			SettingName,
			SettingValue
		) SELECT RolloutConfig,
			SettingType,
			SettingName,
			SettingValue
		FROM RolloutConfigSetting
			WHERE SettingType='Logging'
				AND SettingName='UploadLogProtocol'

		Update #TempRolloutConfigSetting
			SET SettingType='Upload', SettingName='InheritUserCredentials', SettingValue='False'
			WHERE SettingType='Logging' AND SettingName='UploadLogProtocol'

		-- Update the temp table to reflect the new schema
		Update #TempRolloutConfigSetting
			SET SettingName='Protocol'
			WHERE SettingType='Download' AND SettingName='DownloadProtocol'
		
		Update #TempRolloutConfigSetting
			SET SettingName='Share'
			WHERE SettingType='Download' AND SettingName='OSConfigShare'
		
		Update #TempRolloutConfigSetting
			SET SettingName='Username'
			WHERE SettingType='Download' AND SettingName='DownloadUsername'
			
		Update #TempRolloutConfigSetting
			SET SettingName='Password'
			WHERE SettingType='Download' AND SettingName='DownloadPassword'
		
		-- Create dummy entries for the InheritUserCredentials entry
		INSERT INTO #TempRolloutConfigSetting (
			RolloutConfig,
			SettingType,
			SettingName,
			SettingValue
		) SELECT RolloutConfig,
			SettingType,
			SettingName,
			SettingValue
		FROM RolloutConfigSetting
			WHERE SettingType='Download'
				AND SettingName='DownloadProtocol'

		Update #TempRolloutConfigSetting
			SET SettingType='Download', SettingName='InheritUserCredentials', SettingValue='False'
			WHERE SettingType='Download' AND SettingName='DownloadProtocol'

		-- Update the personality
		Update #TempRolloutConfigSetting
			SET SettingType='ComputerDataCapture', SettingName='Share'
			WHERE SettingType='Download' AND SettingName='PersonalityShare'			

		-- Make 3 copies of the Download credentials. These will be used for:
		-- Bootstrap, Capture, Personality
		DECLARE @settingType NVARCHAR(64)
		DECLARE @count INT
		SET @count = 0
		
		WHILE @count < 3
		BEGIN
			
			IF @count = 0
			BEGIN
				SET @settingType = 'Bootstrap'
			END
			ELSE IF @count = 1
			BEGIN
				SET @settingType = 'ComputerImageCapture'
			END
			ELSE IF @count = 2
			BEGIN
				SET @settingType = 'ComputerDataCapture'
			END
		
			-- Protocol
			INSERT INTO #TempRolloutConfigSetting (
				RolloutConfig,
				SettingType,
				SettingName,
				SettingValue
			) SELECT RolloutConfig,
				SettingType,
				SettingName,
				SettingValue
			FROM RolloutConfigSetting
				WHERE SettingType='Download'
					AND SettingName='DownloadProtocol'
		
			Update #TempRolloutConfigSetting
				SET SettingType=@settingType, SettingName='Protocol'
				WHERE SettingType='Download' AND SettingName='DownloadProtocol'
		
			IF @settingType <> 'ComputerDataCapture'
			BEGIN
				-- Share
				INSERT INTO #TempRolloutConfigSetting (
					RolloutConfig,
					SettingType,
					SettingName,
					SettingValue
				) SELECT RolloutConfig,
					SettingType,
					SettingName,
					SettingValue
				FROM RolloutConfigSetting
					WHERE SettingType='Download'
						AND SettingName='OSConfigShare'

				Update #TempRolloutConfigSetting
					SET SettingType=@settingType, SettingName='Share'
					WHERE SettingType='Download' AND SettingName='OSConfigShare'
			END

			-- Username
			INSERT INTO #TempRolloutConfigSetting (
				RolloutConfig,
				SettingType,
				SettingName,
				SettingValue
			) SELECT RolloutConfig,
				SettingType,
				SettingName,
				SettingValue
			FROM RolloutConfigSetting
				WHERE SettingType='Download'
					AND SettingName='DownloadUsername'

			Update #TempRolloutConfigSetting
				SET SettingType=@settingType, SettingName='Username'
				WHERE SettingType='Download' AND SettingName='DownloadUsername'

			-- Password
			INSERT INTO #TempRolloutConfigSetting (
				RolloutConfig,
				SettingType,
				SettingName,
				SettingValue
			) SELECT RolloutConfig,
				SettingType,
				SettingName,
				SettingValue
			FROM RolloutConfigSetting
				WHERE SettingType='Download'
					AND SettingName='DownloadPassword'

			Update #TempRolloutConfigSetting
				SET SettingType=@settingType, SettingName='Password'
				WHERE SettingType='Download' AND SettingName='DownloadPassword'

			-- Create dummy entries for the InheritUserCredentials entry
			INSERT INTO #TempRolloutConfigSetting (
				RolloutConfig,
				SettingType,
				SettingName,
				SettingValue
			) SELECT RolloutConfig,
				SettingType,
				SettingName,
				SettingValue
			FROM RolloutConfigSetting
				WHERE SettingType='Download'
					AND SettingName='DownloadProtocol'

			Update #TempRolloutConfigSetting
				SET SettingType=@settingType, SettingName='InheritUserCredentials', SettingValue='False'
				WHERE SettingType='Download' AND SettingName='DownloadProtocol'

			SET @count = @count + 1
		END

		UPDATE #TempRolloutConfigSetting
			SET SettingValue='ManageSoftCL$'
			WHERE SettingType='ComputerImageCapture' AND SettingName='Share'

		UPDATE #TempRolloutConfigSetting
			SET SettingValue='file://'
			WHERE SettingType='ComputerImageCapture' AND SettingName='Protocol'

		-- Delete all entries from the RolloutConfigSetting table
		DELETE RolloutConfigSetting

		-- Copy data back to the RolloutConfigSetting table
		-- Disable the autoinc key temporarily
		SET IDENTITY_INSERT RolloutConfigSetting ON
		INSERT INTO RolloutConfigSetting(
			RolloutConfigSettingID,
			RolloutConfig,
			SettingType,
			SettingName,
			SettingValue
		) SELECT RolloutConfigSettingID,
			RolloutConfig,
			SettingType,
			SettingName,
			SettingValue
		FROM #TempRolloutConfigSetting
		SET IDENTITY_INSERT RolloutConfigSetting OFF

		DROP TABLE #TempRolloutConfigSetting
	END

	UPDATE DatabaseConfiguration SET Value = 1009 WHERE Property = 'MigrationStep'
END
GO

-- Update Phase column entries to have Postwork
IF EXISTS(SELECT * FROM DatabaseConfiguration WHERE Property = 'MigrationStep' AND VALUE = 1009)
BEGIN
	IF EXISTS (
		SELECT * FROM INFORMATION_SCHEMA.TABLES
		WHERE TABLE_NAME = 'RolloutStatus'
	) AND EXISTS (
		SELECT *
		FROM INFORMATION_SCHEMA.COLUMNS
		WHERE TABLE_NAME = 'RolloutStatus' AND COLUMN_NAME = 'Phase'
	)
	BEGIN
		UPDATE RolloutStatus SET Phase='Postwork'
	END

	UPDATE DatabaseConfiguration SET Value = 1010 WHERE Property = 'MigrationStep'
END
GO

-- Update the hardware assessment rules to comply with Vista
IF EXISTS(SELECT * FROM DatabaseConfiguration WHERE Property = 'MigrationStep' AND VALUE = 1010)
BEGIN
	IF EXISTS (
		SELECT *
		FROM INFORMATION_SCHEMA.TABLES
		WHERE TABLE_NAME = 'HardwareAssessmentRule'
	)
	BEGIN

		IF EXISTS (
			SELECT *
			FROM HardwareAssessmentRule
			WHERE Class = 'Win32_ComputerSystem'
				AND Property = 'TotalPhysicalMemory'
				AND ComparisonExpr = '< 132120576'
		)
		BEGIN
			UPDATE HardwareAssessmentRule
			SET ComparisonExpr = '< 536870912',
				RuleModifiedDate = GetDate()
			WHERE Class = 'Win32_ComputerSystem'
				AND Property = 'TotalPhysicalMemory'
				AND ComparisonExpr = '< 132120576'
		END

		IF EXISTS (
			SELECT *
			FROM HardwareAssessmentRule
			WHERE Class = 'Win32_Processor'
				AND Property = 'MaxClockSpeed'
				AND ComparisonExpr = '< 297'
		)
		BEGIN
			UPDATE HardwareAssessmentRule
			SET ComparisonExpr = '< 997',
				RuleModifiedDate = GetDate()
			WHERE Class = 'Win32_Processor'
				AND Property = 'MaxClockSpeed'
				AND ComparisonExpr = '< 297'
		END

		IF EXISTS (
			SELECT *
			FROM HardwareAssessmentRule
			WHERE Class = 'Win32_DiskDrive'
				AND Property = 'Size'
				AND ComparisonExpr = '< 2000000000'
		)
		BEGIN		
			UPDATE HardwareAssessmentRule
			SET ComparisonExpr = '< 20000000000',
				RuleModifiedDate = GetDate()
			WHERE Class = 'Win32_DiskDrive'
				AND Property = 'Size'
				AND ComparisonExpr = '< 2000000000'
		END
	END

	UPDATE DatabaseConfiguration SET Value = 1011 WHERE Property = 'MigrationStep'
END
GO

-- Set all the hardware assessment SQLtypes to be a big int
IF EXISTS(SELECT * FROM DatabaseConfiguration WHERE Property = 'MigrationStep' AND VALUE = 1011)
BEGIN

	IF EXISTS (
		SELECT *
		FROM INFORMATION_SCHEMA.TABLES
		WHERE TABLE_NAME = 'HardwareAssessmentRule'
	)
	BEGIN
		UPDATE HardwareAssessmentRule
		SET SQLType = 'bigint'
		WHERE SQLType = 'int'
	END

	UPDATE DatabaseConfiguration SET Value = 1012 WHERE Property = 'MigrationStep'
END
GO

--===========================================================================
--		Update the DatabaseConfiguration table
--
--		NOTE: do this last
--===========================================================================
IF EXISTS(SELECT * FROM DatabaseConfiguration WHERE Property = 'MigrationStep' AND VALUE = 1012)
BEGIN
	PRINT 'Recording successful migration'
	UPDATE DatabaseConfiguration SET Value = '7.9'
		WHERE Property = 'WiDepSchemaVersion'
	DELETE DatabaseConfiguration WHERE Property = 'MigrationStep'
END
GO
