/*
 * $Header$
 *
 * DESCRIPTION:
 *
 *	Schema for Group and related tables.
 *	-Creates required TABLES
 *	
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */


USE ManageSoft
GO

/* ----------------------------------------------------------------- */
PRINT '---------------------------------------------------------';
PRINT '--- Creating Windows Deployment: Rollout Group tables ---';
PRINT '---------------------------------------------------------';
/* ----------------------------------------------------------------- */

IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutGroup'
)
BEGIN
	PRINT 'RolloutGroup Table:'
	CREATE TABLE RolloutGroup(

		-- Auto-generated Identity of RolloutGroup
		GroupID int NOT NULL IDENTITY
			CONSTRAINT PK_RolloutGroup PRIMARY KEY,			

		-- Rollout Group Name
		-- NOTE: Can have duplicate names though not under same parent
		-- This will have to be managed at resolving time
		GroupName nvarchar(256) NOT NULL, 			

		-- 0 = represents the Topmost RolloutGroup. Can have multiple groups
		-- at this level.	
		-- Ex: When source is Ghost; then Topmost Parent represents
		-- the GhostServer; under which the Root Groups are kept 		
		ParentID int NOT NULL DEFAULT(0),			

		--REVISIT:
		-- Can be NULL when it represents Topmost RolloutGroup
		-- (can have NULL if not a reference)
		--ParentID int NULL REFERENCES RolloutGroup(GroupID),		
				
		-- Represents the Source of RolloutGroup
		-- Ex: Ghost 7.5		
		Source nvarchar(64) NULL	
	);
	
END /* RolloutGroup */
ELSE
	PRINT 'Using existing RolloutGroup table'

GO

/* ----------------------------------------------------------------- */

IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutGroupMember'
)
BEGIN
	PRINT 'RolloutGroupMember Table:'
	CREATE TABLE RolloutGroupMember(
			
		-- This RolloutGroup ....
		GroupID int NOT NULL
			CONSTRAINT FK_RolloutGroupMember_GroupID
				REFERENCES RolloutGroup(GroupID),
				
		-- ...contains Computer (ManageSoft ComputerID)
		-- And other types like User but *NOT* another 
		-- RolloutGroup (info in Group table)		
		MemberID int NOT NULL,
		
		-- type of the member - comes from TargetType table and
		-- must be a Computer (1) or in future User(2)
		MemberType int NOT NULL,
		
		--Unique as a Group can have a member listed only once
		CONSTRAINT UN_RolloutGroupMember UNIQUE(GroupID, MemberID, MemberType)
	);
	
END /* RolloutGroupMember */
ELSE
	PRINT 'Using existing RolloutGroupMember table'

GO
	
PRINT ' RolloutGroupMemberByType index'
IF NOT EXISTS (
    SELECT *
    FROM sysindexes
    WHERE name = 'RolloutGroupMemberByType'
)
	CREATE CLUSTERED INDEX RolloutGroupMemberByType
		ON RolloutGroupMember(MemberType, MemberID);

GO

/* ----------------------------------------------------------------- */

IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'GhostServer'
)
BEGIN
	PRINT 'GhostServer Table:'
	CREATE TABLE GhostServer(

		GhostServerID int NOT NULL IDENTITY
			CONSTRAINT PK_GhostServer PRIMARY KEY,
		
		-- GhostServer is the *Parent* of root Rollout groups for a given 
		-- Ghost Console Server
		GroupID int NOT NULL 
			CONSTRAINT FK_GhostServer_GroupID
				REFERENCES RolloutGroup(GroupID),
		
		-- Name of Ghost Console Server
		-- This can be duplicates but unique in a domain
		ServerName nvarchar(256) NOT NULL, 
		
		--Name of the Domain to which Ghost Server belongs to		
		DomainName nvarchar(256) DEFAULT(NULL), 
		
		-- Network adapter physical address of the node
		-- Can be MAC address or token ring address
		PhysicalAddress varchar(64) NULL DEFAULT(NULL),
		
		-- ManageSoft Computer ID
		ComputerID int NULL DEFAULT(NULL),
		
		-- DateTime when Group XML was generated
		LastUpdateDateTime DATETIME NULL
	);
	
END /* GhostServer */
ELSE
	PRINT 'Using existing GhostServer table'

GO

/* ----------------------------------------------------------------- */
IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'RolloutGroupToGhostComputer'
)
BEGIN
	PRINT 'RolloutGroupToGhostComputer Table:'
	CREATE TABLE RolloutGroupToGhostComputer(

		-- This Rollout Group ....
		GroupID int NOT NULL
			CONSTRAINT FK_RolloutGroupToGhostComputer_GroupID
				REFERENCES RolloutGroup(GroupID),				
		
		-- ...contains Computer
		--NOTE: Can be NULL if PhysicalAddress is not.				
		ComputerName nvarchar(64) NULL DEFAULT(NULL),
				
		-- Can be NULL if ComputerName is not.
		PhysicalAddress varchar(64) NULL DEFAULT(NULL),
		
		-- Is resolved into RolloutGroupMember table 
		-- with ManageSoft ComputerIDs 
		IsResolved int NOT NULL DEFAULT(0)
	);
	
END /* RolloutGroupToGhostComputer */
ELSE
	PRINT 'Using existing RolloutGroupToGhostComputer table'

GO

/* ----------------------------------------------------------------- */


IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'NameMap'
)
BEGIN
	PRINT 'NameMap Table:'
	CREATE TABLE NameMap(

		-- This name mapping ....
		ID int NOT NULL IDENTITY,
		
		-- old computer name
		Name nvarchar(64) NULL DEFAULT(NULL),

		-- old computer domain
		Domain int DEFAULT(NULL),

		-- new computer name
		NewName nvarchar(64) NULL DEFAULT(NULL),

		-- new computer domain
		NewDomain int DEFAULT(NULL),

		-- the name the comuter currently has
		CurrentID int NULL DEFAULT(NULL),
		
		-- the OU new computers are assigned into
		-- this is a foreign key onto the Organization table
		NewOU int NULL DEFAULT(NULL),
		
		-- the MACAddress of new computers. Existing ones will
		-- get their address from inventory.
		MACAddress nvarchar(32) NULL DEFAULT(NULL),
		
		-- the IP address of new computers
		IPAddress nvarchar(32) NULL DEFAULT(NULL)
	);
	
END /* NameMap */
ELSE
	PRINT 'Using existing NameMap table'
	IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE COLUMN_NAME='NewOU' AND TABLE_NAME='NameMap')
	BEGIN
		ALTER TABLE NameMap
			ADD NewOU int NULL DEFAULT(NULL)
	END
	IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE COLUMN_NAME='MACAddress' AND TABLE_NAME='NameMap')
	BEGIN
        ALTER TABLE NameMap
                ADD MACAddress nvarchar(32) NULL DEFAULT(NULL)
	END
	IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE COLUMN_NAME='IPAddress' AND TABLE_NAME='NameMap')
	BEGIN
        ALTER TABLE NameMap
                ADD IPAddress nvarchar(32) NULL DEFAULT(NULL)
	END
GO

PRINT ' NameMapNameDomain index'
IF NOT EXISTS (
    SELECT *
    FROM sysindexes
    WHERE name = 'NameMapNameDomain'
)
	CREATE INDEX NameMapNameDomain
		ON NameMap(name, domain);

GO

PRINT ' NameMapNewNameDomain index'
IF NOT EXISTS (
    SELECT *
    FROM sysindexes
    WHERE name = 'NameMapNewNameDomain'
)
	CREATE INDEX NameMapNewNameDomain
		ON NameMap(newName, newDomain);

GO

PRINT ' NameMapID index'
IF NOT EXISTS (
    SELECT *
    FROM sysindexes
    WHERE name = 'NameMapID'
)
	CREATE CLUSTERED INDEX NameMapID
		ON NameMap(CurrentId);

GO

IF NOT EXISTS (SELECT * FROM DatabaseConfiguration WHERE Property = 'WiDepSchemaVersion')
BEGIN
	INSERT INTO DatabaseConfiguration (Property, Value) VALUES ('WiDepSchemaVersion','8.6')
END
GO

/* ----------------------------------------------------------------- */
