/*
 * $Header$
 *
 * DESCRIPTION:
 *
 *	Schema for Windows Migration - Hardware Assessment.
 *	-Creates required TABLES
 *	
 *
 * Protected by US patent 6,006,034; patents pending.
 * Copyright 2011 Flexera Software LLC
 * Reproduction, adaptation, or translation without prior permission
 * is prohibited.
 */


USE ManageSoft
GO

/* ----------------------------------------------------------------- */
PRINT '-----------------------------------------';
PRINT '--- Creating Windows Migration tables ---'
PRINT '-----------------------------------------';
/* ----------------------------------------------------------------- */

IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'HardwareAssessmentSeverity'
)
BEGIN
	PRINT 'HardwareAssessmentSeverity Table:'
	CREATE TABLE HardwareAssessmentSeverity(

		-- Unique Identifier of HardwareAssessmentSeverity
		AssessmentSeverityID int NOT NULL PRIMARY KEY,			

		--Unique number representing the order of the severity for assessment
		--For calculating final assessment, AssessmentSeverity with higher number overrides 
		--Example: if a computer has assessment 'Requires replacement' for one rule and 
		--	assessment 'Requires upgrade' for another. Final assessment for the said
		--	computer will be 'Requires replacement' as it has higher severity.
		AssessmentSeverityNumber int NOT NULL 
			CONSTRAINT UN_AssessmentSeverityNunmber UNIQUE,

		--Text representing the assessment severity
		--Example: 'Requires replacement', 'Requires upgrade'
		AssessmentSeverityMessage varchar(64) NOT NULL	
			CONSTRAINT UN_AssessmentSeverityMessage UNIQUE
	);
	
	-- Populate table with initial/default hardware assessment severity
	INSERT INTO HardwareAssessmentSeverity VALUES (1, 1, 'Meets requirements');
	INSERT INTO HardwareAssessmentSeverity VALUES (2, 2, 'Requires upgrade');
	INSERT INTO HardwareAssessmentSeverity VALUES (3, 3, 'Requires replacement');
	INSERT INTO HardwareAssessmentSeverity VALUES (4, 4, 'Not inventoried');

END /* HardwareAssessmentSeverity */
ELSE
	PRINT 'Using existing HardwareAssessmentSeverity table'

GO

/* ----------------------------------------------------------------- */

IF EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'HardwareAssessmentRule'
)
BEGIN
	PRINT 'Using existing HardwareAssessmentRule table'
	IF NOT EXISTS (
		SELECT *
		FROM INFORMATION_SCHEMA.COLUMNS
		WHERE TABLE_NAME = 'HardwareAssessmentRule' AND COLUMN_NAME = 'FriendlyName'
	)
	BEGIN
		PRINT 'Adding FriendlyName column to HardwareAssessmentRule table'
		ALTER TABLE HardwareAssessmentRule ADD FriendlyName nvarchar(256) NULL
		PRINT 'Adding DefaultRule values to HardwareAssessmentRule entries'
	END
	IF NOT EXISTS (
		SELECT *
		FROM INFORMATION_SCHEMA.COLUMNS
		WHERE TABLE_NAME = 'HardwareAssessmentRule' AND COLUMN_NAME = 'Enabled'
	)
	BEGIN
		PRINT 'Adding Enabled column to HardwareAssessmentRule table'
		ALTER TABLE HardwareAssessmentRule ADD Enabled bit DEFAULT (1) NOT NULL 
	END
	IF NOT EXISTS (
		SELECT *
		FROM INFORMATION_SCHEMA.COLUMNS
		WHERE TABLE_NAME = 'HardwareAssessmentRule' AND COLUMN_NAME = 'DefaultRule'
	)
	BEGIN
		PRINT 'Adding DefaultRule column to HardwareAssessmentRule table'
		ALTER TABLE HardwareAssessmentRule ADD DefaultRule bit DEFAULT (0) NOT NULL
	END
END
GO

IF EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'HardwareAssessmentRule'
)
BEGIN
	-- If we added new columns, set their values for the default rules.
	UPDATE HardwareAssessmentRule
		SET FriendlyName = 'Total Physical Memory', DefaultRule = 1
		WHERE FriendlyName IS NULL
		  AND Class = 'Win32_ComputerSystem'
		  AND Property = 'TotalPhysicalMemory'
	UPDATE HardwareAssessmentRule
		SET FriendlyName = 'Clock speed of the Processor', DefaultRule = 1
		WHERE FriendlyName IS NULL
		  AND Class = 'Win32_Processor'
		  AND Property = 'CurrentClockSpeed'
	UPDATE HardwareAssessmentRule
		SET FriendlyName = 'ACPI support in BIOS', DefaultRule = 1
		WHERE FriendlyName IS NULL
		  AND Class = 'Win32_BIOS'
		  AND Property = 'BiosCharacteristics'
	UPDATE HardwareAssessmentRule
		SET FriendlyName = 'Size of Disk', DefaultRule = 1
		WHERE FriendlyName IS NULL
		  AND Class = 'Win32_DiskDrive'
		  AND Property = 'Size'
	UPDATE HardwareAssessmentRule
		SET FriendlyName = 'Custom Rule', DefaultRule = 0
		WHERE FriendlyName IS NULL
		
	--Update Property for HW Class 'Win32_Processor'
	UPDATE HardwareAssessmentRule
		SET Property = 'MaxClockSpeed', RuleModifiedDate = GetDate()
		WHERE Class = 'Win32_Processor'
			AND Property = 'CurrentClockSpeed'
END
GO

IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'HardwareAssessmentRule'
)
BEGIN
	PRINT 'HardwareAssessmentRule Table:'
	CREATE TABLE HardwareAssessmentRule(
		-- Auto-generated identity
		AssessmentRuleID int NOT NULL IDENTITY
			CONSTRAINT PK_HardwareAssessmentRule PRIMARY KEY NONCLUSTERED,

		--DateTime the rule was either created or modified. 
		RuleModifiedDate datetime 
			DEFAULT GETDATE() NOT NULL,

		--AssessmentRule belongs to which set of rules
		--By default: all rules belong to one default set and value is NULL 
		AssessmentRuleSet varchar(64),  --DEFAULT 'default', 

		--Assessment of computer if this assessment rule FAILS. 
		--Must be one of 'Requires upgrade' or 'Requires replacement'. 
		AssessmentSeverityID int NOT NULL
			CONSTRAINT FK_HardwareAssessmentServerity_SeverityMessage 
				REFERENCES HardwareAssessmentSeverity(AssessmentSeverityID),			

		-- Class of property to check.
		-- Example: 'Win32_OperatingSystem', 'Win32_BIOS'
		Class varchar(256) NOT NULL,

		-- Name of the property to check.
		-- Example: 'TotalPhysicalMemory', 'Version'
		Property varchar(64) NOT NULL,

		-- Comparsion expression to use.
		-- Examples: '< 261664', 'in ('A', 'B', 'C')'
		ComparisonExpr varchar(1024) NOT NULL,

		-- SQL type to cast the value to for comparision.
		-- NULL if no cast is required (i.e. use string comparison).
		-- Example: 'int'
		SQLType varchar(20) NULL
			CHECK (	SQLType IS NULL
				OR SQLType IN ('bigint', 'int', 'smallint', 'tinyint', 'bit', 'float', 'real')
				OR SQLType LIKE 'numeric%'
				OR SQLType LIKE 'decimal%'
				OR SQLType LIKE 'varchar%'
				OR SQLType LIKE 'nvarchar%'
			),

		-- Aggregate function to use (SUM, MAX, ...).
		-- NULL if value should not be aggregated
		-- Example: 'SUM'
		AggregateFn varchar(10) NULL,

		-- Message to use if Class.Property is not known.
		-- Example: 'Information about physical memory is not available'
		NotKnownMessage nvarchar(256) NOT NULL,

		-- Message to use if Class.Property does not pass this condition
		-- Example: '{0:b} physical memory is not enough for this computer; must be upgraded to at least 256Mb'
		FailMessage nvarchar(256) NOT NULL,

		-- Message to use if Class.Property passes this condition.
		-- Example: 'Computer has enough physical memory ({0:b})'
		PassMessage nvarchar(256) NOT NULL,
	
		-- Friendly Name of the rule to display in the UI
		FriendlyName nvarchar(256) NULL,

		-- Whether the rule will be ignored
		Enabled bit DEFAULT (1) NOT NULL,

		-- Whether the rule is a default
		DefaultRule bit DEFAULT (0) NOT NULL
	);

	-- Populate table with some standard hardware assessment rules

	--For actual expression value less than 128MB is used (126MB)
	INSERT INTO HardwareAssessmentRule (
		AssessmentSeverityID, Class, Property,
		ComparisonExpr, SQLType, AggregateFn,
		NotKnownMessage, FailMessage, PassMessage,
		FriendlyName, Enabled, DefaultRule
	)
	VALUES (
		2, 'Win32_ComputerSystem', 'TotalPhysicalMemory',
		'< 1073741824', 'numeric(13)', NULL, 
		'Information about physical memory is not available for this computer',
		'This computer only has {0:b} physical memory which is not sufficient for migration.',
		'This computer has {0:b} of physical memory installed which is sufficient for migration.',
		'Total Physical Memory', 1, 1
	);

	--For actual expression value less than 1000MHz is used (997MHz)
	INSERT INTO HardwareAssessmentRule (
		AssessmentSeverityID, Class, Property, 
		ComparisonExpr, SQLType, AggregateFn,
		NotKnownMessage, FailMessage, PassMessage,
		FriendlyName, Enabled, DefaultRule
	)
	VALUES (
		3, 'Win32_Processor', 'MaxClockSpeed',
		'< 997', 'int', 'MAX',
		'Information about CPU speed is not available for this computer',
		'This computer''s CPU only runs at {0} MHz which is not sufficient for migration.',
		'This computer''s CPU runs at {0} MHz which is sufficient for migration.',
		'Clock speed of the Processor', 1, 1
	);

	INSERT INTO HardwareAssessmentRule (
		AssessmentSeverityID, Class, Property, 
		ComparisonExpr, SQLType, AggregateFn,
		NotKnownMessage, FailMessage, PassMessage,
		FriendlyName, Enabled, DefaultRule
	)
	VALUES (
		3, 'Win32_BIOS', 'BiosCharacteristics',
		'NOT LIKE ''%,32[^0-9]%''', NULL, NULL,
		'Information about BIOS characteristics is not available for this computer',
		'This computer''s BIOS does not support ACPI. ACPI support is required for migration.',
		'This computer''s BIOS supports ACPI.',
		'ACPI support in BIOS', 1, 1
	);

	--For actual expression value less than 2GB is used (1.86GB) 
	INSERT INTO HardwareAssessmentRule (
		AssessmentSeverityID, Class, Property, 
		ComparisonExpr, SQLType, AggregateFn,
		NotKnownMessage, FailMessage, PassMessage,
		FriendlyName, Enabled, DefaultRule
	)
	VALUES (
		2, 'Win32_DiskDrive', 'Size',
		'< 20000000000', 'numeric(13)', 'MAX',
		'Information about disk drives is not available for this computer',
		'The capacity of this computer''s largest disk drive is only {0:b} which is not sufficient for migration.',
		'The capacity of this computer''s largest disk drive is {0:b} which is sufficient for migration.',
		'Size of Disk', 1, 1
	);

END /* HardwareAssessmentRule */
GO


/* ----------------------------------------------------------------- */

IF NOT EXISTS (
	SELECT *
	FROM INFORMATION_SCHEMA.TABLES
	WHERE TABLE_NAME = 'HardwareAssessmentCache'
)
BEGIN
	PRINT 'HardwareAssessmentCache Table:'
	CREATE TABLE HardwareAssessmentCache(

		--Computer for which assessment is calculated, FK to Computer		
		ComputerID int NOT NULL,		
		--NOTE: Constraint is not created as it would not allow delete
		--in Computer table
			--CONSTRAINT FK_HardwareAssessmentCache_Computer
				--REFERENCES Computer(ComputerID),

		/*
		 * FK to OU of Computer. This is denormalised, since it's
		 * dependent on other data in this row, but useful for
		 * clustering!
		 */
		ComputerOUID int NOT NULL,
		--NOTE: Constraint is not created as it would not allow delete
		--in SDOU table
			--CONSTRAINT FK_HardwareAssessmentCache_ComputerOU
				--REFERENCES SDOU(OUID),

		--AssessmentRule applied for computer, FK to HardwareAssessmentRule
		AssessmentRuleID int NOT NULL
			CONSTRAINT FK_HardwareAssessmentCache_AssessmentRule
				REFERENCES HardwareAssessmentRule(AssessmentRuleID),

		--Resulting Assessment for computer, FK to HardwareAssessmentSeverity
		--based on AssessmentRule
		AssessmentSeverityID int NOT NULL
			CONSTRAINT FK_HardwareAssessmentCache_AssessmentSeverity
				REFERENCES HardwareAssessmentSeverity(AssessmentSeverityID),

		--Date when Hardware was last inventoried for given computer
		--Value can be NULL if computer never reported inventory
		LastHardwareReportDate datetime,
	
		--DateTime when the record for given computer was evaluated
		CacheEvaluatedDate datetime DEFAULT GETDATE() NOT NULL

		CONSTRAINT PK_HardwareAssessmentCache 
			PRIMARY KEY CLUSTERED (AssessmentRuleID, ComputerID)

	);

	PRINT '	HardwareAssessmentByComputerOU index'
	CREATE INDEX HardwareAssessmentCacheByComputerOU
		ON HardwareAssessmentCache(ComputerOUID, ComputerID);

	PRINT '	HardwareAssessmentByComputer index'
	CREATE INDEX HardwareAssessmentCacheByComputer 
		ON HardwareAssessmentCache(ComputerID);

END /* HardwareAssessmentCache */
ELSE
	PRINT 'Using existing HardwareAssessmentCache table'

GO
